from irslinger.pyslinger import IR
from logger import log

from hardware.remotes import string_inverse, convert_int_to_binary_string

class ACRemote(IR):
	def __init__(self, output_pin, host='localhost'):
		self.protocol_config = {
			'duty_cycle': 0.5,
			'frequency': 38000,
			'trailing_pulse': True,
			'leading_pulse_duration': 3302,
			'leading_gap_duration': 1600,
			'one_pulse_duration': 467,
			'one_gap_duration': 1179,
			'zero_pulse_duration': 491,
			'zero_gap_duration': 338
		}
		self.host = host
		self.output_pin = output_pin
		log.debug("Output Pin = {}".format(output_pin))

	def update(self, prev_status, status, fan_speed, mode, temperature):
		#self.ir = IR(26, "NEC", self.protocol_config)
		self.ir = IR(self.output_pin, "NEC", self.protocol_config, host=self.host)

		log.debug("processing ir command with values status={}, fan_speed={}, mode={}, temperature={}".format(status, fan_speed, mode, temperature))

		if not status:
			command = "00101000 11000110 00000000 00001000 00001000 01000000 10111111"
		else:
			byte1to5 = "00101000 11000110 00000000 00001000 00001000"

			byte6 = self.process_power(status)

			byte7to8 = "10010000 00001100"

			byte9 = self.process_temperature(temperature, prev_status)

			byte10 = self.process_mode(mode) + "0000"

			byte11 = self.process_fan_speed(fan_speed, False) + self.process_swing(True)

			byte12to14 = "00000000 00000000 00000000"

			byte15 = "100001" + self.process_silent(False)

			byte16 = self.process_checksum(byte9, byte10, byte11, byte15) # [208 - (Soma dos bytes 9-15)] % 256

			command = byte1to5 + " " + byte6 + " " + byte7to8 + " " + byte9 + " " + byte10 + " " + byte11 + " " + byte12to14 + " " + byte15 + " " + byte16

		log.debug("Sending infra red code: " + command)

		self.ir.send_code(command)
		self.ir.gpio.stop()

	def process_fan_speed(self, fan_speed, silent):
		if silent:
			return string_inverse("0100")

		if fan_speed == "AUTO":
			return string_inverse("0000")
		elif fan_speed == "HIGH":
			return string_inverse("0001")
		elif fan_speed == "MED" or fan_speed == "MEDIUM":
			return string_inverse("0010")
		elif fan_speed == "LOW":
			return string_inverse("0011")

		print("Warning: unknown fan_speed: " + fan_speed)
		return "0000"

	def process_power(self, status):
		log.debug("Processing power: {}".format(status))
		return "01111111" if status else "01000000"

	def process_silent(self, silent):
		return "01" if silent else "00"

	def process_mode(self, mode):
		if mode == "AUTO":
			return string_inverse("0000")
		elif mode == "ICE" or mode == "COOL":
			return string_inverse("0001")
		elif mode == "DRY":
			return string_inverse("0010")
		elif mode == "FAN":
			return string_inverse("0011")

		print("Warning: unknown mode: " + mode)
		return "0000"

	def process_swing(self, swing):
		return "1000" if swing else "0000"

	def process_temperature(self, temperature, prev_status):
		result = "0000" if prev_status else "1000"

		temperature = int(temperature)
		binary = "{0:04b}".format(temperature - 16)

		result += string_inverse(binary)
		return result

	def process_checksum(self, byte9, byte10, byte11, byte15):

		result = int(string_inverse(byte9), 2) + int(string_inverse(byte10), 2) + int(string_inverse(byte11), 2) + int(string_inverse(byte15), 2)
		result = 208 - result
		result = result % 256
		return string_inverse(convert_int_to_binary_string(result)).ljust(8, '0')

