from threading import Thread
from time import sleep
from logger import log

from pyfingerprint.pyfingerprint import PyFingerprint
from neopixel import Adafruit_NeoPixel, ws, Color
from . import Button
from gpiozero import LED
from notifications import push_service, Notification

class FingerprintScanner():

	def __init__(self, model, port, baud_rate, address, password):
		self.model = model
		self.port = port
		self.baud_rate = baud_rate
		self.address = address
		self.password = password

		self.f = None
		self._scan_success_callbacks = []
		self._scan_error_callbacks = []
		self.thread = Thread(target=self._scan_worker, daemon=True) #doesn't work with green threads

	def connect(self):
		log.info('Connecting fingerprint scanner')

		self.f = PyFingerprint(self.port, self.baud_rate, self.address, self.password)

		log.debug('Verifying fingerprint scanner password')

		if (self.f.verifyPassword() == False):
			raise ValueError('The given fingerprint sensor password is wrong!')

		log.info('Fingerprint scanner successfuly connected')

	def read_image(self, buffer=0x01):
		if self.f is None:
			self.connect()

		while (self.f.readImage() == False):
			pass

		self.f.convertImage(buffer)

	def scan_fingerprint(self):
		self.read_image()

		## Searchs template
		result = self.f.searchTemplate()

		position = result[0]
		accuracy = result[1]

		return position, accuracy

	def start_scan(self):
		log.info('Starting fingerprint scan thread')
		self.thread.start()

	def stop_scan(self):
		log.info('Stoping fingerprint scan thread')
		self.thread.stop()

	def _scan_worker(self):
		log.info('Initializing fingerprint thread')
		self.connect()

		while True:
			try: 
				log.info('Fingerprint scan thread started')

				position,accuracy = self.scan_fingerprint()

				if accuracy != -1:
					log.info('Fingerprint found ' + str(position) + ' with accuracy ' + str(accuracy))
					[callback(position, accuracy) for callback in self._scan_success_callbacks]
				else:
					log.info('Fingerprint not found')
					[callback() for callback in self._scan_error_callbacks]
			except Exception as e:
				log.exception('Fingerprint scan error:')
				[callback() for callback in self._scan_error_callbacks]

				self.connect()

	@property
	def when_scan_success(self):
		return self._scan_success_callbacks

	@when_scan_success.setter
	def when_scan_success(self, callback):
		self._scan_success_callbacks.append(callback)

	@property
	def when_scan_error(self):
		return self._scan_error_callbacks

	@when_scan_error.setter
	def when_scan_error(self, callback):
		self._scan_error_callbacks.append(callback)

	def template_count(self):
		return self.f.getTemplateCount()

	def enroll(self):
		if self.thread.isAlive():
			raise Exception('Fingerprint scan is reading')

		position = self.scan_fingerprint()

		if position >= 0:
			raise Exception('Finger is already registered at position ' + str(position))

		sleep(2) #Waits for confirmation

		self.read_image(0x02)
		self.f.createTemplate()

		position = self.f.getTemplateCount() #TODO there is a bug if exists a non sequential delete and write

		if (self.f.storeTemplate(position) == True):
			return position
		else:
			raise Exception('Error storing fingerprint')

	def __repr__(self):
		return '<FingerprintScanner %r at %r>' % (self.model, self.port)

# class Doorbell(Button):
# 	def __init__(self, buzzer_pin, button_pin, pull_up=False):
# 		super(Doorbell, self).__init__(button_pin, pull_up=pull_up)

# 		self.buzzer = Dimmer(buzzer_pin)

# 		self.when_pressed = self.buzzer.on
# 		self.when_released = self.buzzer.off

# class DoorLock(Dimmer):
# 	def __init__(self, pin, time=0.2):
# 		super(DoorLock, self).__init__(pin)
# 		self.time = time

# 	def unlock(self):
# 		log.info('Unlocking door lock')
# 		self.on()
# 		sleep(self.time)
# 		self.off()


class Guardian:
	
	def __init__(self, device):
		self.device = device

		self.fingerprint_scanner = FingerprintScanner('R303', '/dev/serial0', 57600, 0xFFFFFFFF, 0x00000000)
		self.fingerprint_scanner.when_scan_success = self._when_scan_success
		self.fingerprint_scanner.when_scan_error = self._when_scan_error

		self.LED_COUNT      = 25      # Number of LED pixels.
		self.LED_PIN        = 10      # GPIO pin connected to the pixels (10 uses SPI /dev/spidev0.0).
		self.LED_FREQ_HZ    = 800000  # LED signal frequency in hertz (usually 800khz)
		self.LED_DMA        = 10      # DMA channel to use for generating signal (try 10)
		self.LED_BRIGHTNESS = 255     # Set to 0 for darkest and 255 for brightest
		self.LED_INVERT     = False   # True to invert the signal (when using NPN transistor level shift)
		self.LED_CHANNEL    = 0       # set to '1' for GPIOs 13, 19, 41, 45 or 53
		self.LED_STRIP      = ws.WS2811_STRIP_GRB   # Strip type and colour ordering

		self.led_strip = Adafruit_NeoPixel(self.LED_COUNT, self.LED_PIN, self.LED_FREQ_HZ, self.LED_DMA, self.LED_INVERT, self.LED_BRIGHTNESS, self.LED_CHANNEL, self.LED_STRIP)

		self.doorbell_button = Button(device.doorbell_button_address, pull_up=False, bounce_delay=0.2)
		self.doorbell_button.when_pressed = self._when_doorbell_pressed
		self.doorbell_button.when_released = self._when_doorbell_released

		self.doorlock = LED(device.doorlock_pin)
		self.doorbell_ring = LED(device.doorbell_ring_pin)

		self.mute_doorbell_alert = device.bell_mute_alert
		self.mute_doorbell_notifications = device.bell_mute_notifications

	def start(self):
		self.fingerprint_scanner.start_scan()
		self.led_strip.begin()

		self.clear_leds()

		#Activate doorbell led
		self.led_strip.setPixelColor(24, Color(255,255,255))
		self.led_strip.show()

	def unlock(self):
		self._unlock() #use use parameters from database?
		
		#TODO: notificar usando user
		notification = Notification(user='', category="WARNING", tag="APP_UNLOCK", message="Porta aberta pelo aplicativo.")
		push_service.push_notification(notification)

	def _unlock(self, inverted=False, duration=0.2):
		log.debug("Doorlock Unlocked!!")

		on = self.doorlock.on if not inverted else self.doorlock.off
		off = self.doorlock.off if not inverted else self.doorlock.on

		on()
		sleep(duration)
		off()

	def _when_scan_success(self, position, accuracy):
		self.do_success_led_animation() #TODO fazer animação em thread separada
		self._unlock()

		#notification = Notification(user='', category="WARNING", tag="APP_UNLOCK", message="Porta aberta pela Biometria.")
		#push_service.push_notification(notification)

	def _when_scan_error(self):
		self.do_error_led_animation()

	def _when_doorbell_pressed(self):
		log.debug("Doorbell pressed")
		if not self.mute_doorbell_alert:
			self.doorbell_ring.on()
			sleep(0.5)
			self.doorbell_ring.off()
		else:
			log.debug("Doorbell alert muted")

		#if not self.device.bell_mute_notifications:
		#	notification = Notification(user='', category="WARNING", tag="APP_UNLOCK", message="Alerta de Campainha!")
		#	push_service.push_notification(notification)

	def _when_doorbell_released(self):
		pass

	def _do_acknowledge_led_animation(self, time=0.5, color=Color(0,255,0)):
		for i in range(24):
			if i % 2 == 0:
				self.led_strip.setPixelColor(i // 2, color)
			else:
				self.led_strip.setPixelColor(24-1 - i // 2, color)
				sleep(time/24.0)

			self.led_strip.show()

		sleep(1)
		self.clear_leds(indexes=range(24))

	def do_success_led_animation(self, time=0.5):
		self._do_acknowledge_led_animation(color=Color(0,255,0))

	def do_error_led_animation(self, time=0.5):
		self._do_acknowledge_led_animation(color=Color(255,0,0))

	def clear_leds(self, indexes=range(25)):
		for i in indexes:
			self.led_strip.setPixelColor(i, Color(0, 0, 0)) #Wipe everything
		
		self.led_strip.show()

	def __repr__(self):
		return '<Guardian>'