from logger import log
from ..models import Triac, Button, ClimateIRDevice, LircIRDevice
import itertools
import re

def board_from_json(board):
	if board['code'] == 'FHTR400':
		return FHTR400(board)
	elif board['code'] == 'FHTR410':
		return FHTR410(board)
	elif board['code'] == 'FHIR100':
		return FHIR100(board)
	
	log.warning('Wrong board configuration model: ' + board['code'])


class FHTR400:
	protocol = 'rpi'
	name = 'UNIO Triac Board'
	model = 'FHTR400'
	manufacturer = 'UNIO Smart Home'
	def __init__(self, board):
		self.host = board.get('host', 'localhost')
		self.outputs = [Triac('{}.{}'.format(self.protocol, pin), host=self.host) for pin in [12, 16, 20, 21]]
		self.inputs = [Button('{}.{}'.format(self.protocol, pin), host=self.host) for pin in [4, 14, 15, 18, 23, 24]]

	@property
	def id(self):
		return '{}_{}_{}'.format(self.model, self.host, self.protocol)

	def __repr__(self):
		return '<FHTR400: protocol={}, host={}, inputs={}, outputs={}>'.format(self.protocol, self.host, self.inputs, self.outputs)

class FHTR410:
	protocol = 'i2c'
	name = 'UNIO Triac Board'
	model = 'FHTR410'
	manufacturer = 'UNIO Smart Home'
	def __init__(self, board):
		self.host = board.get('host', 'localhost')
		self.address = board['address']
		self.outputs = [Triac('{}.{}.{}'.format(self.protocol, self.address, pin)) for pin in range(9, 13)]
		self.inputs = [Button('{}.{}.{}'.format(self.protocol, self.address, pin)) for pin in range(1, 7)]

	@property
	def id(self):
		return '{}_{}_{}'.format(self.model, self.protocol, self.address)

	def __repr__(self):
		return '<FHTR410: protocol={}, inputs={}, outputs={}>'.format(self.protocol, self.inputs, self.outputs)

class FHIR100:
	protocol = 'rpi'
	name = 'UNIO Nano'
	model = 'FHIR100'
	manufacturer = 'UNIO Smart Home'
	def __init__(self, board):
		self.host = board.get('host', 'localhost')
		self.address = board['pin']
		self.pin = int(self.address.split('.')[1])
		self.devices = [self._create_device(self.pin, device, self.host) for device in board['devices']]

	@property
	def id(self):
		return '{}_{}_{}'.format(self.model, self.protocol, self.address)

	def _create_device(self, pin, device, host):
		if device['type'] == 'climate':
			return ClimateIRDevice(device['type'], host, device['conf'], pin, device['name'], \
				device.get('min_temp', 16), device.get('max_temp', 30))
		elif device['type'] == 'media':
			return LircIRDevice(device['type'], device['conf'], pin, device['name'])

	def __repr__(self):
		return '<FHIR100: devices={}>'.format(self.devices)
