from __future__ import annotations

import abc
import math
from functools import partial
from typing import TYPE_CHECKING, Any, Generic, NamedTuple, TypeVar, Union

from more_itertools import pairwise

if TYPE_CHECKING:
    from collections.abc import Callable, Iterable

    from typing_extensions import Self

PointType = TypeVar("PointType")


class Point(NamedTuple):
    x: int
    y: int

    @classmethod
    def start_point(cls, value: int = -100) -> Point:
        return cls(-192000, value)

    @classmethod
    def end_point(cls, value: int = -100) -> Point:
        return cls(1073741823, value)


def _inner_interpolate(
    data: list[Point],
    sampling_interval_tick: int,
    mapping: Callable[[int, Point, Point], float],
) -> list[Point]:
    return (
        (
            [data[0]]
            + [
                Point(x=x, y=round(mapping(x, start, end)))
                for start, end in pairwise(data)
                for x in range(start.x + 1, end.x, sampling_interval_tick)
            ]
            + [data[-1]]
        )
        if data
        else data
    )


def linear_interpolation(x: int, start: tuple[float, float], end: tuple[float, float]) -> float:
    x0, y0 = start
    x1, y1 = end
    return y0 + (x - x0) * (y1 - y0) / (x1 - x0)


interpolate_linear = partial(_inner_interpolate, mapping=linear_interpolation)


def cosine_easing_in_out_interpolation(
    x: int, start: tuple[float, float], end: tuple[float, float]
) -> float:
    x0, y0 = start
    x1, y1 = end
    return (y0 + y1) / 2 + (y0 - y1) * math.cos((x - x0) / (x1 - x0) * math.pi) / 2


interpolate_cosine_ease_in_out = partial(
    _inner_interpolate, mapping=cosine_easing_in_out_interpolation
)


def cosine_easing_in_interpolation(
    x: int, start: tuple[float, float], end: tuple[float, float]
) -> float:
    x0, y0 = start
    x1, y1 = end
    return y1 + (y0 - y1) * math.cos((x - x0) / (x1 - x0) * math.pi / 2)


interpolate_cosine_ease_in = partial(_inner_interpolate, mapping=cosine_easing_in_interpolation)


def cosine_easing_out_interpolation(
    x: int, start: tuple[float, float], end: tuple[float, float]
) -> float:
    x0, y0 = start
    x1, y1 = end
    return y0 + (y0 - y1) * math.cos((x - x0) / (x1 - x0) * math.pi / 2 + math.pi / 2)


interpolate_cosine_ease_out = partial(_inner_interpolate, mapping=cosine_easing_out_interpolation)


class PointList(abc.ABC, Generic[PointType]):
    root: list[PointType]

    def __len__(self) -> int:
        return len(self.root)

    def __getitem__(self, index: int) -> PointType:
        return self.root[index]

    def __setitem__(self, index: int, value: PointType) -> None:
        self.root[index] = value

    def __delitem__(self, index: int) -> None:
        del self.root[index]

    def __contains__(self, item: PointType) -> bool:
        return item in self.root

    def append(self, item: PointType) -> None:
        self.root.append(item)

    def insert(self, i: int, item: PointType) -> None:
        self.root.insert(i, item)

    def pop(self, i: int = -1) -> PointType:
        return self.root.pop(i)

    def remove(self, item: PointType) -> None:
        self.root.remove(item)

    def clear(self) -> None:
        self.root.clear()

    def count(self, item: PointType) -> int:
        return self.root.count(item)

    def index(self, item: PointType, *args: Any) -> int:
        return self.root.index(item, *args)

    def reverse(self) -> None:
        self.root.reverse()

    def sort(self, /, *args: Any, **kwds: Any) -> None:
        self.root.sort(*args, **kwds)

    def extend(self, other: Union[Self, Iterable[PointType]]) -> None:
        if isinstance(other, PointList):
            self.root.extend(other.root)
        else:
            self.root.extend(other)
