"""Support Peewee ORM for Muffin framework."""

from contextlib import asynccontextmanager
from typing import TYPE_CHECKING, Callable, Optional, Type, Union  # py37, py38: Type

import peewee as pw
from aio_databases.database import ConnectionContext, TransactionContext
from muffin.plugins import BasePlugin, PluginNotInstalledError
from peewee_aio.manager import Manager
from peewee_aio.model import AIOModel
from peewee_migrate import Router

from .fields import (
    Choices,
    IntEnumField,
    JSONLikeField,
    JSONPGField,
    StrEnumField,
    URLField,
)

if TYPE_CHECKING:
    from muffin import Application
    from peewee_aio.types import TVModel

__all__ = (
    "Plugin",
    "Choices",
    "StrEnumField",
    "IntEnumField",
    "EnumField",
    "URLField",
    "JSONLikeField",
    "JSONPGField",
)

EnumField = StrEnumField


class Plugin(BasePlugin):

    """Muffin Peewee Plugin."""

    name = "peewee"
    defaults = {
        # Connection params
        "connection": "sqlite:///db.sqlite",
        "connection_params": {},
        # Manage connections automatically
        "auto_connection": True,
        "auto_transaction": True,
        # Setup migration engine
        "migrations_enabled": True,
        "migrations_path": "migrations",
        # Support pytest
        "pytest_setup_db": True,
    }

    router: Router
    manager: Manager = Manager(
        "dummy://localhost",
    )  # Dummy manager for support registration

    def setup(self, app: "Application", **options):
        """Init the plugin."""
        super().setup(app, **options)

        # Init manager and rebind models
        manager = Manager(self.cfg.connection, **self.cfg.connection_params)
        for model in list(self.manager):
            manager.register(model)
        self.manager = manager

        if self.cfg.migrations_enabled:
            router = Router(manager.pw_database, migrate_dir=self.cfg.migrations_path)
            self.router = router

            # Register migration commands
            @app.manage
            def peewee_migrate(name: Optional[str] = None, *, fake: bool = False):
                """Run application's migrations.

                :param name: Choose a migration' name
                :param fake: Run as fake. Update migration history and don't touch the database
                """
                with manager.allow_sync():
                    router.run(name, fake=fake)

            @app.manage
            def peewee_create(name: str = "auto", *, auto: bool = False):
                """Create a migration.

                :param name: Set name of migration [auto]
                :param auto: Track changes and setup migrations automatically
                """
                with manager.allow_sync():
                    router.create(name, auto=auto and list(self.manager.models))

            @app.manage
            def peewee_rollback():
                """Rollback the latest migration."""
                with manager.allow_sync():
                    router.rollback()

            @app.manage
            def peewee_list():
                """List migrations."""
                with manager.allow_sync():
                    router.logger.info("Migrations are done:")
                    router.logger.info("\n".join(router.done))
                    router.logger.info("")
                    router.logger.info("Migrations are undone:")
                    router.logger.info("\n".join(router.diff))

            @app.manage
            def peewee_clear():
                """Clear migrations from DB."""
                with manager.allow_sync():
                    self.router.clear()

            @app.manage
            def peewee_merge(name: str = "initial"):
                """Merge all migrations into one."""
                with manager.allow_sync():
                    self.router.merge(name)

        if self.cfg.auto_connection:
            app.middleware(self.get_middleware(), insert_first=True)

    async def startup(self):
        """Connect to the database (initialize a pool and etc)."""
        await self.manager.connect()

    async def shutdown(self):
        """Disconnect from the database (close a pool and etc.)."""
        await self.manager.disconnect()

    async def __aenter__(self) -> "Plugin":
        """Connect the database."""
        await self.manager.connect()
        return self

    async def __aexit__(self, *exit_args):
        """Disconnect the database."""
        await self.manager.disconnect()

    def register(self, model_cls: Type["TVModel"]) -> Type["TVModel"]:
        """Register a model with self manager."""
        return self.manager.register(model_cls)

    def connection(self, *params, **opts) -> ConnectionContext:
        return self.manager.connection(*params, **opts)

    def transaction(self, *params, **opts) -> TransactionContext:
        return self.manager.transaction(*params, **opts)

    async def create_tables(self, *models_cls: Type[pw.Model]):
        """Create SQL tables."""
        await self.manager.create_tables(*(models_cls or self.manager.models))

    async def drop_tables(self, *models_cls: Type[pw.Model]):
        """Drop SQL tables."""
        await self.manager.drop_tables(*(models_cls or self.manager.models))

    def get_middleware(self) -> Callable:
        """Generate a middleware to manage connection/transaction."""

        if self.cfg.auto_transaction:

            async def middleware(handler, request, receive, send):
                async with self.connection():
                    async with self.transaction():
                        return await handler(request, receive, send)

        else:

            async def middleware(handler, request, receive, send):
                async with self.connection():
                    return await handler(request, receive, send)

        return middleware

    @property
    def Model(self) -> Type[AIOModel]:  # noqa: N802
        if self.app is None:
            raise PluginNotInstalledError

        return self.manager.Model

    @property
    def JSONField(self) -> Union[Type[JSONLikeField], Type[JSONPGField]]:  # noqa: N802
        """Return a JSON field for the current backend."""
        if self.app is None:
            raise PluginNotInstalledError

        if self.manager.backend.db_type == "postgres":
            return JSONPGField

        return JSONLikeField

    @asynccontextmanager
    async def conftest(self):
        """Initialize a database schema for pytest."""
        if self.cfg.pytest_setup_db:
            async with self:
                async with self.connection():
                    await self.create_tables()
                    yield self
                    await self.drop_tables()
        else:
            yield self
