import json

from .general import str2pathlib
from .typing import File, JsonSerializable


@str2pathlib
def read_json(filename: File, **kwargs) -> JsonSerializable:
    """Reads data from json-file.

    Args:
        filename: name of the json-file.
        **kwargs: arguments from ``json.load()`` method.

    Returns:
        Data from json-file.
    """
    with open(filename) as file:
        return json.load(file, **kwargs)


@str2pathlib
def write_json(
    data: JsonSerializable, filename: File, *, newline: bool = False, **kwargs
) -> File:
    """Writes dictionary to json-file.

    Args:
        data: dictionary to save.
        filename: name of the json-file. Empty extension will be replaced with .json
        newline: to put neline symbol at the end of the file or not
            (needed to avoid `pre-commit` trigger on autogenerated files)
        **kwargs: arguments from ``json.dump()`` method.

    Returns:
        Path to saved file (may differ in suffix from original)
    """
    if not len(filename.suffix):
        filename = filename.with_suffix(".json")
    with open(filename, "w") as file:
        json.dump(data, file, **kwargs)
        if newline:
            file.write("\n")
    return filename


try:
    import yaml

    @str2pathlib
    def read_yaml(filename: File) -> JsonSerializable:
        """Reads YAML file. analogue to `read_json`"""
        with open(filename) as file:
            return yaml.safe_load(file)

    @str2pathlib
    def write_yaml(data: JsonSerializable, filename: File, **kwargs) -> File:
        """Writes YAML file. analogue to `write_json`"""
        with open(filename, "w") as file:
            yaml.safe_dump(data, file, **kwargs)
        return filename

except ModuleNotFoundError:
    pass


try:
    import toml

    @str2pathlib
    def read_toml(filename: File) -> JsonSerializable:
        """Reads TOML file. analogue to `read_json`"""
        with open(filename) as file:
            return toml.load(file)

    @str2pathlib
    def write_toml(data: JsonSerializable, filename: File) -> File:
        """Writes YAML file. analogue to `write_json`"""
        with open(filename, "w") as file:
            toml.dump(data, file)
        return filename

except ModuleNotFoundError:
    pass
