# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gdrfile',
 'gdrfile.generations',
 'gdrfile.helpers',
 'gdrfile.sample',
 'gdrfile.sample.conftest']

package_data = \
{'': ['*']}

install_requires = \
['poetry-core>=1.0.8,<2.0.0', 'python-dotenv>=0.20.0,<0.21.0']

entry_points = \
{'console_scripts': ['gdrfile = gdrfile.main:main']}

setup_kwargs = {
    'name': 'gdrfile',
    'version': '0.0.1',
    'description': 'Generate file for django rest framework.',
    'long_description': '# Генератор файлов для django rest приложения #\n\nGdrfile - генератор начальных файлов приложения, который за основу берет модели \nиз приложения. К начальным файлам относятся: представления, сериализаторы,\nтесты, админ-панель, init файлы, роутер.\n\n# Как установить # \n\nPip:\n\n```bash\npip install gdrfile\n  ```\n\nPoetry:\n\n```bash\npoetry add gdrfile\n ```\n\n# Использование #\n\nДля генерации файлов в консоли django проекта необходимо прописать команду:\n\n```bash\ngdrfile -pm=path/to/folder/models -pa=path.to.api -an=app_name\n ```\n\nДля очистки сгенерированных файлов:\n\n```bash\ngdrfile -clear\n ```\n\nКоманда имеет следующие **обязательные** аргументы для работы.\n\n1. **-pm** - Путь до папки models из текущей директории.\n   Пример: server/apps/name_app/models\n2. **-pa** - Путь до приложения. Пример: server.apps.name_app\n   Используется для указания корректных импортов\n3. **-an** - Название приложения. Пример: app_name\n\n**Необязательные** аргументы для работы:\n\n1. **-pmc** - Названия родительских классов моделей, кроме от models.Model и\nBaseModel. Пример: BaseModel, MPTTModel\nи др. Доступны разделители \';\' и \',\'.\n2. **-mf** - Поля в моделях могут быть из сторонних пакетов. Для их корректного\nанализа необходимо указать информацию о них.Необходимо указать тип поля в\nмодели, тип поля в сериализаторе и тип поля в тестах.\nПример: PhoneField, CharField(), fake.number();\nДля корректной работы необходимо поля разделить \',\', а в конце поставить \';\'\n3. **-r** - Флаг для генерации файлов с учетом прав доступа \n(библиотеки django-rules). По умолчанию False.\n\nЧтобы не указывать в ручную параметры, существует возможность указать путь до\n.env файла.\n\n```bash\n$ gdrfile -env=settings/config\n```\n\nВ .env файле также необходимо прописать три обязательные переменные:\n\n```yaml\n# Путь до папки models из текущей директории.\nGDRFILE_PATH_TO_MODELS=server/apps/app_name/models\n\n# Путь до приложения.\nGDRFILE_PATH_TO_API=server.apps.app_name\n\n# Название приложения.\nGDRFILE_APP_NAME=app_name\n\n# Типы полей могут использоваться из сторонних библиотек.\n# Первое значение - тип поля для моделей.\n# Второе значение - тип поля в сериализаторе.\n# Третье значение - тип поля для тестов в библиотеке faker.\nGDRFILE_MODEL_FIELDS=TreeForeignKey, None, fake.pyint();\n\n# Названия родительских классов моделей, кроме models.Model и BaseModel\nGDRFILE_PARENT_MODEL_CLASS=MPTTModel, ExampleModel\n\n# Флаг для генерации файлов с учетом прав доступа\nGDRFILE_RULES=False\n```\n\nИнформация, которая будет получена из моделей:\n\n```\n{{path_to_app}} - путь до приложения\n{{app_name}} - название приложения\n{{AppName}} - название приложения в верблбюжем стиле\n{{app-name}} - название приложения с \'-\'\n{{MainClass}} - название модели в верблбюжем стиле\n{{main_class}} - название модели с \'_\'\n{{mainclass}} - слитное название модели маленькими буквами \n{{main-class}} - название приложения с \'-\'\n{{docs}} - документация модели\n{{list_main_fields}} - список полей модели\nfields_django_model - список полей модели с типами, указаннами в моделе\nfields_for_serializers - список полей модели с типами для сериализатора\nfields_for_conftest - список полей модели с типами для тестов\n```\n\nСуществует возможность переопределить шаблоны файлов для генерации. \nДля этого в текущем каталоге необходимо создать папку sample \nсо следующей структурой:\n\n```\nsample\n  ├── conftest\n  │    ├── conftest_factory.py\n  │    ├── conftest_fixture.py\n  │    └── conftest_import.py\n  ├── example_admin.py\n  ├── example_init_serializer.py\n  ├── example_init_view.py\n  ├── example_routers.py\n  ├── example_rules.py\n  ├── example_serializers.py\n  ├── example_tests.py\n  ├── example_views.py\n  └── example_views_this_rules.py\n```\n\n# Примеры сгенерированных файлов\n\n\n##### Сериализатор:\n```python\nfrom rest_framework import serializers\n\nfrom server.apps.mobile_provision.models import Account\n\n\nclass AccountSerializer(serializers.ModelSerializer):\n    """Аккаунт."""\n\n    class Meta(object):\n        model = Account\n        fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n\n\nclass AccountSerializer(serializers.Serializer):\n    """Аккаунт."""\n\n        user = serializers.None(__change_me__)\n        photo = serializers.ImageField()\n        birth_date = serializers.DateField()\n        passport_series = serializers.CharField()\n        passport_number = serializers.CharField()\n```\n\n##### Представление (доступно как с правами, так и без прав доступа)\n```python\nfrom rest_framework.viewsets import ModelViewSet\n\nfrom server.apps.mobile_provision.api.serializers import AccountSerializer\nfrom server.apps.mobile_provision.models import Account\n\n\nclass AccountViewSet(ModelViewSet):\n    """Аккаунт."""\n\n    queryset = Account.objects.all()\n    serializer_class = AccountSerializer\n    ordering_fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n    search_fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n\n\nfrom rest_framework_extensions.mixins import NestedViewSetMixin\nfrom rest_framework.viewsets import ModelViewSet\nfrom rules.contrib.rest_framework import AutoPermissionViewSetMixin\n\nfrom server.apps.mobile_provision.api.serializers import AccountSerializer\nfrom server.apps.mobile_provision.models import Account\n\nclass AccountViewSet(\n    NestedViewSetMixin,\n    AutoPermissionViewSetMixin,\n    ModelViewSet,\n):\n    """Аккаунт."""\n\n    queryset = Account.objects.all()\n    serializer_class = AccountSerializer\n    ordering_fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n    search_fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n    permission_type_map = {\n        **AutoPermissionViewSetMixin.permission_type_map,\n        \'list\': \'list\',\n        \'metadata\': None,\n    }\n```\n\n##### Права доступа\n```python\nimport rules\nfrom rules.predicates import is_authenticated\n\n\nrules.add_perm(\'mobile_provision.view_account\', is_authenticated)\nrules.add_perm(\'mobile_provision.add_account\', is_authenticated)\nrules.add_perm(\'mobile_provision.change_account\', is_authenticated)\nrules.add_perm(\'mobile_provision.delete_account\', is_authenticated)\nrules.add_perm(\'mobile_provision.list_account\', is_authenticated)\n```\n\n##### Тесты\n```python\nimport pytest\nfrom faker import Faker\nfrom rest_framework import status\nfrom rest_framework.reverse import reverse\n\nfake = Faker()\n\n\n@pytest.mark.django_db()\ndef test_account_format(\n    api_client,\n    account,\n    account_format,\n):\n    """Формат Account."""\n    url = reverse(\n        \'api:mobile-provision:account-detail\',\n        [account.pk],\n    )\n\n    json_response = api_client.get(url).json()\n\n    assert json_response == account_format(account)\n\n\n@pytest.mark.django_db()\ndef test_account_post(\n    api_client,\n):\n    """Создание Account."""\n    url = reverse(\'api:mobile-provision:account-list\')\n    json_response = api_client.post(\n        url,\n        data={},\n        format=\'json\',\n    )\n\n    assert json_response.status_code == status.HTTP_201_CREATED\n\n\n@pytest.mark.django_db()\ndef test_account_delete(\n    api_client,\n    account,\n):\n    """Удаление Account."""\n    url = reverse(\n        \'api:mobile-provision:account-detail\',\n        [account.pk],\n    )\n\n    json_response = api_client.delete(url)\n\n    assert json_response.status_code == status.HTTP_204_NO_CONTENT\n\n\n@pytest.mark.django_db()\ndef test_account_change(\n    api_client,\n    account,\n):\n    """Изменение Account."""\n    url = reverse(\n        \'api:mobile-provision:account-detail\',\n        [account.pk],\n    )\n\n    json_response = api_client.put(\n        url,\n        data={},\n        format=\'json\',\n    )\n\n    assert json_response.status_code == status.HTTP_200_OK\n```\n\n##### Админка\n```python\nfrom django.contrib import admin\nfrom server.apps.mobile_provision.model import Account\n\n\n@admin.register(Account)\nclass ConfigurableAdmin(admin.ModelAdmin[Account]):\n    """Аккаунт."""\n\n    list_filter = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n    search_fields = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n    list_display = [\'user\', \'photo\', \'birth_date\', \'passport_series\', \'passport_number\']\n```\n\n##### Файл conftest для тестов\n```python\nimport pytest\nfrom factory import LazyAttribute, SubFactory\nfrom factory.django import DjangoModelFactory\nfrom faker import Faker\nfrom pytest_factoryboy import register\nfrom rest_framework.fields import DateTimeField\nfrom rest_framework.test import APIClient\n\nfrom server.apps.mobile_provision.models import Account\n\nfake = Faker()\n\n\nclass AccountFactory(DjangoModelFactory):\n    """Фабрика для Account."""\n\n    class Meta(object):\n        model = Account\n\n    user = factory.SubFactory(__change_me__)\n    photo = factory.LazyAttribute(lambda account: fake.file_name())\n    birth_date = factory.LazyAttribute(lambda account: fake.date_between())\n    passport_series = factory.LazyAttribute(lambda account: fake.paragraph())\n    passport_number = factory.LazyAttribute(lambda account: fake.paragraph())\n\n\nregister(Account)\n\n\n@pytest.fixture\ndef account_format():\n    """Формат Account."""\n    def _account_format(account: Account):\n        return {\n            \'id\': account.pk,\n            \'user\': account.user,\n            \'photo\': account.photo,\n            \'birth_date\': account.birth_date,\n            \'passport_series\': account.passport_series,\n            \'passport_number\': account.passport_number,\n        }\n    return _account_format\n```\n\n##### Файл init в папке с сериализаторами. Аналогичный создается для представлений\n```python\nfrom server.apps.app_name.api.serializer.measure_recommendation import AccountSerializer\n\n__all__ = [\n    \'AccountSerializer\',\n]\n```\n\n##### Файл router\n```python\nfrom django.utils.translation import gettext_lazy as _\nfrom drf_nova_router.api_router import ApiRouter\nfrom rest_framework.routers import APIRootView\n\nfrom server.apps.app_name.api.views import AccountViewSet\n\n\nclass AppNameAPIRootView(APIRootView):\n    """Корневой view для app."""\n\n    __doc__ = _(\'Приложение AppName\')\n    name = _(\'app_name\')\n\n\nrouter = ApiRouter()\n\nrouter.APIRootView = AppNameAPIRootView\nrouter.register(\'accounts\', AccountViewSet, \'accounts\')\n```\n',
    'author': 'Gleb Mazur',
    'author_email': 'mazur.gleb2010@yandex.ru',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/BigBread57/gdrfile',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
