import logging
import os
import sys
from logging.handlers import TimedRotatingFileHandler

LOG_FORMAT = "%(asctime)s %(levelname)-5s - %(name)s : %(message)s"
LOG_FILE = "echoss.log"
# Set environment variable for Python IO encoding
os.environ["PYTHONIOENCODING"] = "utf-8"
# Set stdout and stderr to UTF-8 encoding
# sys.stdout = open(sys.stdout.fileno(), mode='w', encoding='utf-8', buffering=1)
# sys.stderr = open(sys.stderr.fileno(), mode='w', encoding='utf-8', buffering=1)


def _get_console_handler(formatter):
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setFormatter(formatter)
    # set utf-8 encoding
    return console_handler


def _get_file_handler(formatter, file_path, backup_count):
    file_handler = TimedRotatingFileHandler(file_path, when="midnight", backupCount=backup_count, encoding='utf-8')
    file_handler.setFormatter(formatter)
    return file_handler


def get_logger(logger_name='echoss', logger_format=LOG_FORMAT, file_path=LOG_FILE, backup_count=0, use_console=True):
    """
    create new echoss logger or replace old
    :param logger_name: logger name
    :param logger_format: logging format
    :param file_path: file path if File logging else None
    :param backup_count: if 0 keep all timed rotating log files, else only keep backup count number of files
    :param use_console: use console output or not
    :returns: generated logger
    :raises None
    """
    logger = logging.getLogger(logger_name)
    formatter = logging.Formatter(logger_format)
    logger.setLevel(logging.DEBUG)  # better to have too much log than not enough
    # with this pattern, it's rarely necessary to propagate the error up to parent
    logger.propagate = False

    # Update or add console handler
    has_console_handler = False
    for handler in logger.handlers:
        if not isinstance(handler, TimedRotatingFileHandler) and isinstance(handler, logging.StreamHandler):
            if use_console:
                handler.setFormatter(formatter)
                has_console_handler = True
            else:
                # Remove StreamHandler
                logger.removeHandler(handler)
                handler.close()

    if use_console and not has_console_handler:
        logger.addHandler(_get_console_handler(formatter))

    # Update or add file handler
    file_handler_exists = False
    for handler in logger.handlers:
        if isinstance(handler, TimedRotatingFileHandler):
            # Check if the existing handler path matches the current file_path
            if handler.baseFilename == file_path:
                handler.setFormatter(formatter)
                file_handler_exists = True
            else:
                # Remove the old handler and close it
                logger.removeHandler(handler)
                handler.close()

    if file_path and not file_handler_exists:
        logger.addHandler(_get_file_handler(formatter, file_path, backup_count))

    return logger


def set_logger_level(logger, level):
    """
    :param logger: logger generated by get_logger()
    :param level: string like "DEBUG" or "INFO" or integer logging.DEBUG or logging.INFO
    :return: None
    :raises ValueError or TypeError if level invalid
    """
    if isinstance(level, str):
        level = level.upper()
        numeric_level = getattr(logging, level, None)
        if not isinstance(numeric_level, int):
            raise ValueError(f"Invalid log level: {level}")
    elif isinstance(level, int):
        numeric_level = level
    else:
        raise TypeError("Level must be an integer or string")

    logger.setLevel(numeric_level)
