# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['optional']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'optional.py',
    'version': '1.3.2',
    'description': 'An implementation of the Optional object in Python',
    'long_description': '# Optional.py\n[![Build Status](https://img.shields.io/pypi/v/optional.py.svg)](https://pypi.org/project/optional.py/)\n[![Build Status](https://travis-ci.com/cbefus/optional.py.svg?branch=master)](https://travis-ci.com/cbefus/optional.py)\n[![Coverage Status](https://coveralls.io/repos/github/cbefus/optional.py/badge.svg?branch=master)](https://coveralls.io/github/cbefus/optional.py?branch=master)\n[![License](https://img.shields.io/pypi/l/optional.py.svg)](https://pypi.org/project/optional.py/)\n[![Python Versions](https://img.shields.io/pypi/pyversions/optional.py.svg)](https://pypi.org/project/optional.py/)\n[![Contributors](https://img.shields.io/github/contributors/cbefus/optional.py.svg)](https://pypi.org/project/optional.py/)\n[![Open Source Love](https://badges.frapsoft.com/os/v2/open-source.png?v=103)](https://github.com/open-source)\n[![Downloads](https://pepy.tech/badge/optional-py)](https://pepy.tech/project/optional-py)\n\nAn Implementation of the Optional Object for Python\n\n## Why\n\nThere is a difference between `None` as Empty and `None` as the result for an Error.  A common bad practice is to\nreturn `None` to indicate the absence of something. Doing this introduces ambiguity into you code.\n\nFor example:\n```python\nthing = stuff.getSomeThing().getAnotherThing()\n```\nWhat will happen if the result from getSomeThing returns `None`?  We will get an `AttributeError: \'NoneType\' object has\nno attribute \'getAnotherThing\'`.\n\nWhat can you do to prevent these kinds of exceptions?  You can write defensively:\n```python\nsomething = stuff.getSomeThing()\nif something is not None:\n    thing = something.getAnotherThing()\n```\nHowever, if we add to our chain, you can imagine how the nesting of defensive checks adds up quickly. These defensive checks obfuscate our actual business logic, decreasing readability.\nFurthermore, defensive checking is an error prone process, because it is easy to forget to check a required condition.\n\nSo we present you with an **Optional** object as an alternative.\n\n## Install\n\n**Compatible with both python 2 and 3!**\n\n```bash\n$ pip install optional.py\n```\n\n## Usage\n\n1. You can import it using:\n    ```python\n    from optional import Optional\n    ```\n\n2. You can set it to empty:\n\n    instead of: :scream_cat:\n    ```python\n    return None\n    ```\n    you can do: :smile_cat:\n    ```python\n    return Optional.empty()\n    ```\n    or\n    ```python\n    return Optional.of()\n    ```\n\n3. You can set it to have content:\n\n    instead of: :scream_cat:\n    ```python\n    return "thing"\n    ```\n    you can do: :smile_cat:\n    ```python\n    return Optional.of("thing")\n    ```\n\n\n4. You can check if its present:\n\n    instead of: :scream_cat:\n    ```python\n    if thing is not None:\n    ```\n    you can do: :smile_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    if thing.is_present():\n    ```\n    or, alternatively: :smirk_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    if thing:\n    ```\n5. You can check if its empty:\n\n    instead of: :scream_cat:\n    ```python\n    if thing is None:\n    ```\n    you can do: :smile_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    if thing.is_empty():\n    ```\n    or, alternatively: :smirk_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    if not thing:\n    ```\n\n\n6. You can get the value:\n\n    instead of: :scream_cat:\n    ```python\n    print(thing)\n    ```\n    you can do: :smirk_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    ...\n    print(thing.get())\n    ```\n    **but this is not the recommended way to use this library.**\n\n7. You **can\'t** get the value if its empty:\n\n    instead of: :crying_cat_face:\n    ```python\n    if thing is None:\n        print(None) # very odd\n    ```\n    you can do: :smirk_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    if thing.is_empty():\n        print(thing.get()) # **will raise an exception**\n    ```\n    **but this will raise an exception!**\n\n8. You can get_or_default which takes a default value:\n\n    instead of: :crying_cat_face:\n    ```python\n    thing = some_func_may_return_none()\n    if thing is None:\n         thing = \'23\'\n    ```\n    or: :scream_cat:\n    ```python\n    thing = Optional.of(some_func_may_return_none())\n    if thing.is_empty():\n        thing = \'23\'\n    else:\n        thing = thing.get()\n    ```\n    you can do: :smirk_cat:\n    ```python\n    thing = Optional.of(some_func_may_return_none()).get_or_default(\'23\')\n    ```\n\n9. You can get_or_raise to raise any exception on abscence:\n\n    instead of: :scream_cat:\n    ```python\n    try:\n        thing = some_func_returning_an_optional()\n        return thing.get()\n    except OptionalAccessOfEmptyException:\n        raise MyCustomException()\n    ```\n    you can do: :heart_eyes_cat:\n    ```python\n    return some_func_returning_an_optional().get_or_raise(\n        MyCustomException()\n    )\n    ```\n\n10. **Best Usage:** You can chain on presence:\n\n    instead of: :scream_cat:\n    ```python\n    if thing is not None:\n        print(thing)\n    ```\n    you can do: :heart_eyes_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    thing.if_present(lambda thing: print(thing))\n    ```\n\n\n11. **Best Usage:** You can chain on non presence:\n\n    instead of: :scream_cat:\n    ```python\n    if thing is not None:\n        print(thing)\n    else:\n        print("PANTS!")\n    ```\n    you can do: :heart_eyes_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    thing.if_present(lambda thing: print(thing)).or_else(lambda _: print("PANTS!"))\n    ```\n    Note that the lambdas here can be swapped out for actual function names.\n\n12. **Best Usage:** You can run a supplier on non presence:\n\n    instead of: :scream_cat:\n    ```python\n        thing = some_func_returning_an_empty_optional()\n        if thing.is_empty():\n            thing = Optional.of("pants")\n        print(thing.get()) # Prints "pants"\n\n    ```\n    you can do: :heart_eyes_cat:\n    ```python\n        def some_supplier():\n            return "pants"\n        thing = some_func_returning_an_empty_optional().or_else(some_supplier)\n        print(thing.get()) # Prints "pants"\n    ```\n\n12. **Best Usage:** You can raise on non presence:\n\n    instead of: :scream_cat:\n    ```python\n    if thing is None:\n        raise SomeException("Boom!")\n    ```\n    you can do: :heart_eyes_cat:\n    ```python\n    thing = some_func_returning_an_optional()\n    thing.if_present(lambda thing: print(thing)).or_else_raise(SomeException("Boom!"))\n    ```\n\n13. **Best Usage:** You can map a function: :heart_eyes_cat:\n\n    ```python\n    def mapping_func(thing):\n        return thing + "PANTS"\n\n    thing_to_map = Optional.of("thing")\n    mapped_thing = thing_to_map.map(mapping_func) # returns Optional.of("thingPANTS")\n    ```\n    **Note** that if the mapping function returns `None` then the map call will return `Optional.empty()`. Also\n    if you call `map` on an empty optional it will return `Optional.empty()`.\n\n14. **Best Usage:** You can flat map a function which **already returns an Optional**: :heart_eyes_cat:\n    ```python\n    def flat_mapping_func(thing):\n        return Optional.of(thing + "PANTS")\n\n    thing_to_map = Optional.of("thing")\n    mapped_thing = thing_to_map.map(mapping_func) # returns Optional.of("thingPANTS")\n    ```\n    **Note** that this does not return an Optional of an Optional.  __Use this for mapping functions which return optionals.__\n    If the mapping function you use with this does not return an Optional, calling `flat_map` will raise a\n    `FlatMapFunctionDoesNotReturnOptionalException`.\n\n15. You can compare two optionals: :smile_cat:\n    ```python\n    Optional.empty() == Optional.empty() # True\n    Optional.of("thing") == Optional.of("thing") # True\n    Optional.of("thing") == Optional.empty() # False\n    Optional.of("thing") == Optional.of("PANTS") # False\n    ```\n\n\n## Tests\n\nThere is complete test coverage and they pass in both python 2 and 3.\n\n### Running Unit Tests\n\nFirst, install `poetry` using the instructions located [here](https://python-poetry.org/docs/#installation).\n\nThen, install the requirements using:\n```bash\n$ poetry install\n```\n\nYou can run the tests using:\n```bash\n$ poetry run pytest\n```\n\n#### Test Coverage\nYou can check the code coverage using:\n```bash\n$ poetry run pytest --cov=optional\n```\n',
    'author': 'Chad Befus',
    'author_email': 'crbefus@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cbefus/optional.py',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
