import { JupyterFrontEnd } from '@jupyterlab/application';
import React, { createContext, useContext, useState } from 'react';

export const STUDIO_SAGEMAKER_UI_PLUGIN_ID = '@amzn/sagemaker-ui:project';

enum PluginEnvironmentType {
  LocalJL = 'local-jupyter-lab',
  Studio = 'studio',
}

class PluginEnvironment {

  public get isStudio(): boolean {
    return this.type === PluginEnvironmentType.Studio;
  }

  public get isLocalJL(): boolean {
    return this.type === PluginEnvironmentType.LocalJL;
  }

  constructor(
    public readonly type: PluginEnvironmentType,
  ) { }

}

function getPluginEnvironment(app: JupyterFrontEnd): PluginEnvironment {
  if (app.hasPlugin(STUDIO_SAGEMAKER_UI_PLUGIN_ID)) {
    return new PluginEnvironment(PluginEnvironmentType.Studio);
  }

  return new PluginEnvironment(PluginEnvironmentType.LocalJL);
}

type PluginEnvironmentValue = {
  pluginEnvironment: PluginEnvironment;
  setPluginEnvironment: (state: PluginEnvironment) => void;
}

const PluginEnvironmentContext = createContext<PluginEnvironmentValue | undefined>(undefined);

type PluginEnvironmentProviderProps = {
  app: JupyterFrontEnd;
  children: React.ReactNode;
};

function PluginEnvironmentProvider({ app, children }: PluginEnvironmentProviderProps) {
  const [pluginEnvironment, setPluginEnvironment] = useState<PluginEnvironment>(() => {
    return getPluginEnvironment(app);
  });
  const value = { pluginEnvironment, setPluginEnvironment };
  return <PluginEnvironmentContext.Provider value={value}>{children}</PluginEnvironmentContext.Provider>
}

function usePluginEnvironment(): PluginEnvironmentValue {
  const context = useContext(PluginEnvironmentContext);
  if (context === undefined) {
    throw new Error('usePluginEnvironment must be used within a PluginEnvironmentProvider');
  }
  return context;
}

export { PluginEnvironmentProvider, usePluginEnvironment, PluginEnvironment, PluginEnvironmentType };
