import os

import pytz
import json
import re

from datetime import datetime
from typing import Dict, List, Union, Optional
from amazon_sagemaker_jupyter_scheduler.providers.image_metadata import (
    get_image_metadata,
)
from amazon_sagemaker_jupyter_scheduler.providers.tags import (
    get_common_resource_tag_filters,
)
from amazon_sagemaker_jupyter_scheduler.providers.revenue import (
    get_revenue_attribution_string,
)
from amazon_sagemaker_jupyter_scheduler.utils import sanitize_string

from jupyter_scheduler.models import (
    CreateJob,
    Status,
    SortDirection,
    DescribeJob,
    ListJobsQuery,
    ListJobsResponse,
    DescribeJobDefinition,
    ListJobDefinitionsQuery,
    CreateJobDefinition,
    UpdateJobDefinition,
    DEFAULT_SORT,
)
from jupyter_scheduler.scheduler import BaseScheduler

from amazon_sagemaker_jupyter_scheduler.app_metadata import get_region_name
from amazon_sagemaker_jupyter_scheduler.clients import SageMakerAsyncBoto3Client
from amazon_sagemaker_jupyter_scheduler.cron_util import GNUCronExpressionAdapter
from amazon_sagemaker_jupyter_scheduler.file_uploader import (
    S3FileUploader,
    SM_UI_EMPTY_FILE_STRING,
)
from amazon_sagemaker_jupyter_scheduler.internal_metadata_adapter import (
    InternalMetadataAdapter,
)
from amazon_sagemaker_jupyter_scheduler.models import (
    SageMakerSearchSortOrder,
    SageMakerTrainingJobStatus,
    JobTag,
    EventBridgeRuleStatus,
    JobEnvironmentVariableName,
    RuntimeEnvironmentParameterName,
)
from amazon_sagemaker_jupyter_scheduler.runtime_environment_parameters import (
    RuntimeEnvironmentParameters,
)

CHANNEL_NAME_SAGEMAKER_SYSTEM_DRIVER = "sagemaker_headless_execution_system"
TRAINING_JOB_INPUT_FOLDER = "/opt/ml/input/data"


class ModelConverter:
    def __init__(self, logger):
        self.log = logger

    def to_job_tags(
        self,
        notebook_name: str,
        upstream_model: Union[CreateJob, CreateJobDefinition, UpdateJobDefinition],
    ):
        # TODO: Include customer tags.
        return [
            {"Key": "sagemaker:name", "Value": upstream_model.name},
            {"Key": "sagemaker:is-scheduling-notebook-job", "Value": "true"},
            {
                "Key": "sagemaker:notebook-name",
                "Value": notebook_name,
            },
            {"Key": "sagemaker:is-studio-archived", "Value": "false"},
            {"Key": "sagemaker:headless-execution-version", "Value": "false"},
            {"Key": "sagemaker:scheduling-notebook-job-source", "Value": "studiolab"},
        ]

    def generate_training_job_name(self, job_name, notebook_name: str):
        filename_parts = os.path.splitext(notebook_name)
        # To be consistent with timestamps generated by SageMaker pipelines, use ISO 8601 format with trailing Z-.
        timestamp = (
            datetime.now(pytz.utc)
            .isoformat(timespec="milliseconds")
            .replace("+00:00", "Z-")
        )
        return f"{sanitize_string(job_name, 10)}-{sanitize_string(filename_parts[0],10)}-{timestamp}{filename_parts[1]}"

    async def to_create_training_job_input(
        self,
        training_job_name: str,
        upstream_model: Union[CreateJob, CreateJobDefinition, UpdateJobDefinition],
        s3_file_uploader: S3FileUploader,
        s3_dependency_uri: str,
    ) -> Dict:
        """
        Converts a Jupyter CreateJob input into a SageMaker CreateTrainingJob input
        """
        self.log.info(f"Creating training job input...")
        runtime_environment_parameters = RuntimeEnvironmentParameters(
            upstream_model.runtime_environment_parameters
        )

        training_image_metadata = await get_image_metadata(
            runtime_environment_parameters.sm_image, get_region_name()
        )

        execution_script = f"{TRAINING_JOB_INPUT_FOLDER}/{CHANNEL_NAME_SAGEMAKER_SYSTEM_DRIVER}/headless_execution.sh"
        container_arguments = []

        if training_image_metadata.gid == "0":
            container_arguments.append(execution_script)
        else:
            # 3p images may use uid:gid values of 1000:100 which requires setting permissions for the job to succeed.
            # https://docs.aws.amazon.com/sagemaker/latest/dg/studio-byoi-create.html
            container_arguments.append("-c")
            container_arguments.append(
                "; ".join(
                    [
                        "set -x",
                        "sudo chown -R 1000:100 /opt/ml/input/data",
                        "sudo chmod -R 777 /opt/ml/input /opt/ml/output /tmp",
                        f"exec /bin/bash {execution_script}",
                    ]
                )
            )

        output = {
            "AlgorithmSpecification": {
                "TrainingImage": training_image_metadata.ecr_uri,
                "TrainingInputMode": "File",
                "ContainerEntrypoint": ["/bin/bash"],
                "ContainerArguments": container_arguments,
            },
            "Environment": {
                JobEnvironmentVariableName.SM_JOB_DEF_VERSION.value: "1.0",
                JobEnvironmentVariableName.SM_FIRST_PARTY_IMAGEOWNER.value: training_image_metadata.image_owner,
                JobEnvironmentVariableName.SM_FIRST_PARTY_IMAGE_ARN.value: training_image_metadata.image_arn,
                JobEnvironmentVariableName.SM_KERNEL_NAME.value: runtime_environment_parameters.sm_kernel,
                JobEnvironmentVariableName.SM_EFS_MOUNT_PATH.value: training_image_metadata.mount_path,
                JobEnvironmentVariableName.SM_EFS_MOUNT_UID.value: training_image_metadata.uid,
                JobEnvironmentVariableName.SM_EFS_MOUNT_GID.value: training_image_metadata.gid,
                JobEnvironmentVariableName.SM_INPUT_NOTEBOOK_NAME.value: os.path.basename(
                    upstream_model.input_uri
                ),
                JobEnvironmentVariableName.SM_OUTPUT_NOTEBOOK_NAME.value: (
                    self.generate_training_job_name(
                        upstream_model.name, s3_file_uploader.notebook_name
                    )
                ),
                JobEnvironmentVariableName.AWS_DEFAULT_REGION.value: get_region_name(),
                JobEnvironmentVariableName.SM_ENV_NAME.value: upstream_model.runtime_environment_name,
                JobEnvironmentVariableName.SM_OUTPUT_FORMATS.value: ",".join(
                    upstream_model.output_formats or []
                ),
                JobEnvironmentVariableName.SM_EXECUTION_INPUT_PATH.value: f"{TRAINING_JOB_INPUT_FOLDER}/{get_revenue_attribution_string()}",
                **runtime_environment_parameters.customer_environment_variables,
            },
            "InputDataConfig": [
                {
                    "ChannelName": get_revenue_attribution_string(),
                    "DataSource": {
                        "S3DataSource": {
                            "S3DataType": "S3Prefix",
                            "S3Uri": os.path.join(
                                runtime_environment_parameters.s3_input,
                                training_job_name,
                                "input",
                            ),
                            "S3DataDistributionType": "FullyReplicated",
                        }
                    },
                },
                {
                    "ChannelName": CHANNEL_NAME_SAGEMAKER_SYSTEM_DRIVER,
                    "DataSource": {
                        "S3DataSource": {
                            "S3DataType": "S3Prefix",
                            # TODO: read latest version
                            "S3Uri": s3_dependency_uri,
                            "S3DataDistributionType": "FullyReplicated",
                        }
                    },
                },
            ],
            "OutputDataConfig": {
                "S3OutputPath": os.path.join(
                    runtime_environment_parameters.s3_output, training_job_name
                )
                if isinstance(upstream_model, CreateJobDefinition)
                or isinstance(upstream_model, UpdateJobDefinition)
                else runtime_environment_parameters.s3_output,
            },
            "ResourceConfig": {
                "InstanceCount": 1,
                # TODO: The UpdateJobDefinition model currently does not have compute_type
                "InstanceType": "ml.m5.xlarge"
                if isinstance(upstream_model, UpdateJobDefinition)
                else upstream_model.compute_type,
                "VolumeSizeInGB": 30,
            },
            # "RetryStrategy": {"MaximumRetryAttempts": upstream_model.max_retries},
            "RetryStrategy": {
                "MaximumRetryAttempts": runtime_environment_parameters.max_retry_attempts
                if hasattr(runtime_environment_parameters, "max_retry_attempts")
                else 1
            },  # TODO: currently OSS UI is not providing an option to set this, setting it for default 1
            "RoleArn": runtime_environment_parameters.role_arn,
            "StoppingCondition": {
                # TODO: Stop hard-coding these values, when they are added back to the model.
                # 60 * 60 * 48 = 172800 seconds (48 hours)
                "MaxRuntimeInSeconds": runtime_environment_parameters.max_run_time_in_seconds
                if hasattr(runtime_environment_parameters, "max_run_time_in_seconds")
                else 172800,
                "MaxWaitTimeInSeconds": 172800,
                # "MaxRuntimeInSeconds": upstream_model.timeout_seconds,
                # "MaxWaitTimeInSeconds": upstream_model.timeout_seconds,
            },
        }

        if upstream_model.parameters is not None and len(upstream_model.parameters):
            output["HyperParameters"] = upstream_model.parameters

        if runtime_environment_parameters.sm_init_script:
            output["Environment"][
                str(JobEnvironmentVariableName.SM_INIT_SCRIPT)
            ] = s3_file_uploader.sm_init_script_name

        if (
            runtime_environment_parameters.sm_lcc_init_script_arn
            != SM_UI_EMPTY_FILE_STRING
        ):
            output["Environment"][
                str(JobEnvironmentVariableName.SM_LCC_INIT_SCRIPT)
            ] = s3_file_uploader.sm_lcc_init_script_name

            output["Environment"][
                str(JobEnvironmentVariableName.SM_LCC_INIT_SCRIPT_ARN)
            ] = s3_file_uploader.sm_lcc_init_script_arn

        if len(runtime_environment_parameters.security_group_ids) > 0 and len(
            runtime_environment_parameters.subnets
        ):
            output["VpcConfig"] = {
                "SecurityGroupIds": runtime_environment_parameters.security_group_ids.split(
                    ","
                ),
                "Subnets": runtime_environment_parameters.subnets.split(","),
            }

        if runtime_environment_parameters.sm_output_kms_key:
            output["OutputDataConfig"][
                "KmsKeyId"
            ] = runtime_environment_parameters.sm_output_kms_key

        if runtime_environment_parameters.sm_volume_kms_key:
            output["ResourceConfig"][
                "VolumeKmsKeyId"
            ] = runtime_environment_parameters.sm_volume_kms_key

        self.log.info(f"Successfully created training job input...")
        return output

    def to_jupyter_job_status(
        self, status: Union[SageMakerTrainingJobStatus, str]
    ) -> Status:
        if str(status) == SageMakerTrainingJobStatus.IN_PROGRESS.value:
            return Status.IN_PROGRESS
        if str(status) == SageMakerTrainingJobStatus.COMPLETED.value:
            return Status.COMPLETED
        if str(status) == SageMakerTrainingJobStatus.FAILED.value:
            return Status.FAILED
        if str(status) == SageMakerTrainingJobStatus.STOPPING.value:
            return Status.STOPPING
        if str(status) == SageMakerTrainingJobStatus.STOPPED.value:
            return Status.STOPPED

        raise NotImplementedError(f"Unrecognized training job status: {status}")

    def to_training_job_status(self, status: Status) -> SageMakerTrainingJobStatus:
        if status == Status.IN_PROGRESS:
            return SageMakerTrainingJobStatus.IN_PROGRESS
        if status == Status.COMPLETED:
            return SageMakerTrainingJobStatus.COMPLETED
        if status == Status.FAILED:
            return SageMakerTrainingJobStatus.FAILED
        if status == Status.STOPPING:
            return SageMakerTrainingJobStatus.STOPPING
        if status == Status.STOPPED:
            return SageMakerTrainingJobStatus.STOPPED

        raise NotImplementedError(f"Unrecognized job status: {status}")

    def to_sagemaker_sort_order(
        self, sort_direction: SortDirection
    ) -> SageMakerSearchSortOrder:
        if sort_direction == SortDirection.asc:
            return SageMakerSearchSortOrder.ASCENDING
        if sort_direction == SortDirection.desc:
            return SageMakerSearchSortOrder.DESCENDING

        raise NotImplementedError(f"Unrecognized sort direction: {sort_direction}")

    def _to_jupyter_runtime_environment_parameters(
        self,
        training_job_name: str,
        training_job_response: Dict,
        s3_uri: str,
        s3_output: str,
    ) -> Dict:
        training_job_environment = training_job_response.get("Environment", {})

        return {
            **RuntimeEnvironmentParameters(
                training_job_environment
            ).customer_environment_variables,
            RuntimeEnvironmentParameterName.SM_IMAGE.value: training_job_environment.get(
                JobEnvironmentVariableName.SM_FIRST_PARTY_IMAGE_ARN.value, ""
            ),
            RuntimeEnvironmentParameterName.S3_INPUT.value: s3_uri.split(
                f"/{training_job_name}"
            )[0],
            RuntimeEnvironmentParameterName.MAX_RETRY_ATTEMPTS.value: training_job_response.get("RetryStrategy", {}).get(
                "MaximumRetryAttempts", 0
            ),
            RuntimeEnvironmentParameterName.MAX_RUN_TIME_IN_SECONDS.value: training_job_response.get("StoppingCondition", {}).get(
                "MaxRuntimeInSeconds", 172800
            ),
            RuntimeEnvironmentParameterName.S3_OUTPUT.value: s3_output,
            RuntimeEnvironmentParameterName.SM_KERNEL.value: training_job_environment.get(
                JobEnvironmentVariableName.SM_KERNEL_NAME.value, ""
            ),
            RuntimeEnvironmentParameterName.SM_INIT_SCRIPT.value: training_job_environment.get(
                JobEnvironmentVariableName.SM_INIT_SCRIPT.value, ""
            ),
            RuntimeEnvironmentParameterName.SM_LCC_INIT_SCRIPT_ARN.value: training_job_environment.get(
                JobEnvironmentVariableName.SM_LCC_INIT_SCRIPT_ARN.value, ""
            ),
            RuntimeEnvironmentParameterName.ROLE_ARN.value: training_job_response[
                "RoleArn"
            ],
            RuntimeEnvironmentParameterName.VPC_SECURITY_GROUP_IDS.value: ",".join(
                training_job_response.get("VpcConfig", {}).get("SecurityGroupIds", [])
            ),
            RuntimeEnvironmentParameterName.VPC_SUBNETS.value: ",".join(
                training_job_response.get("VpcConfig", {}).get("Subnets", [])
            ),
            RuntimeEnvironmentParameterName.SM_OUTPUT_KMS_KEY.value: training_job_response.get(
                "OutputDataConfig", {}
            ).get(
                "KmsKeyId", ""
            ),
            RuntimeEnvironmentParameterName.SM_VOLUME_KMS_KEY.value: training_job_response.get(
                "ResourceConfig", {}
            ).get(
                "VolumeKmsKeyId", ""
            ),
        }

    def determine_available_output_formats_and_failure_reason(
        self, status: Status, failure_reason: Optional[str]
    ) -> (List[str], str):
        """
        If a failed job has a special code like "[SM-XXX]", that indicates which output formats are available.
        When that code is not present, we will assume no output formats are available.
        When that code is present, 0 represents no output, 1 represents available output.
        Each "X" will be either a 0 or a 1. 0 represents no output. 1 represents available output.
        For example, "100" means only the Input file is available.
        """

        # This order corresponds to the execution driver "[SM-XXX]" code.
        known_output_formats = ["input", "ipynb", "log"]

        if status != Status.FAILED or not failure_reason:
            return known_output_formats, failure_reason

        match = re.search(r"^(.*?)\[SM-([0-9]+)]\s*(.*)", failure_reason, re.DOTALL)

        # If failure reason does not match "[SM-XXX]", assume no outputs are available.
        if not match:
            return [], failure_reason

        failure_code = match.group(2)
        available_output_formats = []

        for index, output_format in enumerate(known_output_formats):
            if len(failure_code) > index and failure_code[index] == "1":
                available_output_formats.append(output_format)

        return available_output_formats, (match.group(1) + match.group(3))

    def to_jupyter_describe_job_output(
        self,
        scheduler: BaseScheduler,
        outputs: Optional[bool],
        training_job_response: Dict,
        tag_dict: Dict,
    ) -> DescribeJob:
        """
        Combines DescribeTrainingJob and ListTags responses to produce a DescribeJob instance.
        """

        start_time = training_job_response.get("TrainingStartTime")
        end_time = training_job_response.get("TrainingEndTime")

        # TODO: What if there are multiple InputDataConfig entries? (Use "headless_execution" channel.)
        s3_uri = (
            training_job_response.get("InputDataConfig", [{}])[0]
            .get("DataSource", {})
            .get("S3DataSource", {})
            .get("S3Uri")
        )

        s3_output = training_job_response.get("OutputDataConfig", {}).get(
            "S3OutputPath"
        )

        environment = training_job_response.get("Environment", {})

        output_formats = environment.get(
            JobEnvironmentVariableName.SM_OUTPUT_FORMATS.value, "ipynb"
        ).split(",") + ["log"]

        status = self.to_jupyter_job_status(training_job_response["TrainingJobStatus"])

        (
            available_output_formats,
            failure_reason,
        ) = self.determine_available_output_formats_and_failure_reason(
            status, training_job_response.get("FailureReason")
        )

        # Reduce output formats to those that are available, while preserving original order.
        output_formats = [
            output_format
            for output_format in output_formats
            if output_format in available_output_formats
        ]
        secondary_status = training_job_response.get("SecondaryStatus", '')
        training_job_status = training_job_response["TrainingJobStatus"]

        describe_job_response = DescribeJob(
            job_id=training_job_response.get("TrainingJobName"),
            output_formats=output_formats,
            input_filename=environment.get(
                JobEnvironmentVariableName.SM_INPUT_NOTEBOOK_NAME.value, ""
            ),
            runtime_environment_name=environment.get(
                JobEnvironmentVariableName.SM_ENV_NAME.value, "sagemaker-default-env"
            ),
            name=tag_dict.get(JobTag.NAME.value, "Untitled Job"),
            job_definition_id=tag_dict.get(JobTag.JOB_DEFINITION_ID.value),
            parameters=training_job_response.get("HyperParameters"),
            runtime_environment_parameters=self._to_jupyter_runtime_environment_parameters(
                training_job_name=training_job_response["TrainingJobName"],
                training_job_response=training_job_response,
                s3_uri=s3_uri,
                s3_output=s3_output,
            ),
            timeout_seconds=training_job_response.get("StoppingCondition", {}).get(
                "MaxRuntimeInSeconds"
            ),
            retry_on_timeout=(
                training_job_response.get("RetryStrategy", {}).get(
                    "MaximumRetryAttempts", 0
                )
                > 0
            ),
            max_retries=training_job_response.get("RetryStrategy", {}).get(
                "MaximumRetryAttempts", 0
            ),
            compute_type=training_job_response.get("ResourceConfig", {}).get(
                "InstanceType"
            ),
            create_time=training_job_response["CreationTime"].timestamp() * 1000,
            update_time=training_job_response["LastModifiedTime"].timestamp() * 1000,
            start_time=start_time.timestamp() * 1000 if start_time else None,
            end_time=end_time.timestamp() * 1000 if end_time else None,
            url = "https://amazon.com",
            status_message = secondary_status if secondary_status == "MaxRuntimeExceeded" and training_job_status == "Stopped" else failure_reason,
            status = Status.FAILED if secondary_status == "MaxRuntimeExceeded" and training_job_status == "Stopped" else status,
            # idempotency_token = TODO: TBD OSS
            # tags = TODO: How should we translate list of key:value pairs to list of strings?
        )

        if outputs:
            # This indicates to the UI where the output files are located, if they have been downloaded.
            scheduler.add_job_files(describe_job_response)

        return describe_job_response

    def get_common_resource_tags_filters(self):
        return [
            {
                "Name": f"Tags.{JobTag.IS_SCHEDULING_NOTEBOOK_JOB}",
                "Operator": "Exists",
            },
            {
                "Name": f"Tags.{JobTag.IS_STUDIO_ARCHIVED}",
                "Operator": "Equals",
                "Value": "false",
            },
        ]

    def to_sagemaker_search_pipeline_sort_field(self, sort_field: str):
        """
        Converts a Jupyter sort field name to a field name recognized by SageMaker Search for Pipelines.

        Example inputs: "name", "input_filename", "create_time"
        """

        if not sort_field:
            return "LastModifiedTime"

        if sort_field == "name":
            return f"Tags.{JobTag.NAME}"
        if sort_field == "input_filename":
            # Limitation: We cannot sort by input_filename because it's embedded in the pipeline definition.
            return "LastModifiedTime"
        if sort_field == "create_time":
            return "CreationTime"

        raise RuntimeError(f"Unrecognized sort field: {sort_field}")

    def to_pipeline_seach_input(
        self, query: ListJobDefinitionsQuery, default_filters=None
    ) -> Dict:
        # Limitation: SageMaker Search can sort by only one field, so use the last one the user clicked (0).
        sort_by = query.sort_by[0] if query.sort_by else DEFAULT_SORT

        filters = get_common_resource_tag_filters()

        if query.name:
            filters.append(
                {
                    "Name": f"Tags.{JobTag.NAME}",
                    "Operator": "Contains",
                    "Value": query.name,
                }
            )

        if query.create_time:
            filters.append(
                {
                    "Name": "CreationTime",
                    "Operator": "GreaterThanOrEqualTo",
                    "Value": datetime.fromtimestamp(
                        query.create_time / 1000
                    ).isoformat(),
                }
            )

        return {
            "MaxResults": min(
                # Don't go over 50 pipeline results, since Events:DescribeRule has a default rate of 50 TPS
                # https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-quota.html
                query.max_items or 50,
                50,
            ),  # Valid Range: Minimum value of 1. Maximum value of 100.
            "NextToken": query.next_token or "",
            "Resource": "Pipeline",
            "SearchExpression": {
                "Filters": filters,
            },
            "SortBy": self.to_sagemaker_search_pipeline_sort_field(sort_by.name),
            "SortOrder": self.to_sagemaker_sort_order(sort_by.direction).value,
        }

    def to_sagemaker_search_training_job_sort_field(self, sort_field: str):
        """
        Converts a Jupyter sort field name to a field name recognized by SageMaker Search for Training Jobs.

        Example inputs: "name", "input_filename", "create_time", "status".
        """

        if not sort_field:
            return "CreationTime"

        if sort_field == "name":
            return f"Tags.{JobTag.NAME}"
        if sort_field == "input_filename":
            return f"Environment.{JobEnvironmentVariableName.SM_INPUT_NOTEBOOK_NAME}"
        if sort_field == "create_time":
            return "CreationTime"
        if sort_field == "status":
            return "TrainingJobStatus"

        raise RuntimeError(f"Unrecognized sort field: {sort_field}")

    def to_training_job_search_input(self, query: ListJobsQuery) -> Dict:
        # Limitation: SageMaker Search can sort by only one field, so use the last one the user clicked (0).
        sort_by = query.sort_by[0] if query.sort_by else DEFAULT_SORT

        # TODO: Translate tags from List[str] to List[Dict]

        filters = get_common_resource_tag_filters()

        if query.status:
            filters.append(
                {
                    "Name": "TrainingJobStatus",
                    "Operator": "Equals",
                    "Value": self.to_training_job_status(query.status),
                }
            )

        if query.start_time:
            filters.append(
                {
                    "Name": "TrainingStartTime",
                    "Operator": "GreaterThanOrEqualTo",
                    "Value": datetime.fromtimestamp(query.start_time, pytz.utc),
                }
            )

        if query.name:
            filters.append(
                {
                    "Name": f"Tags.{JobTag.NAME}",
                    "Operator": "Contains",
                    "Value": query.name,
                }
            )

        if query.job_definition_id:
            filters.append(
                {
                    "Name": f"Tags.{JobTag.JOB_DEFINITION_ID}",
                    "Operator": "Equals",
                    "Value": query.job_definition_id,
                }
            )

        return {
            "MaxResults": min(query.max_items or 100, 100),
            "NextToken": query.next_token or "",
            "Resource": "TrainingJob",
            "SearchExpression": {
                "Filters": filters,
            },
            "SortBy": self.to_sagemaker_search_training_job_sort_field(sort_by.name),
            "SortOrder": self.to_sagemaker_sort_order(sort_by.direction).value,
        }

    def to_jupyter_list_jobs_response(
        self,
        scheduler: BaseScheduler,
        training_job_list_response: Dict,
    ) -> ListJobsResponse:
        job_results: List[DescribeJob] = []

        for item in training_job_list_response["Results"]:
            training_job = item["TrainingJob"]
            job_results.append(
                self.to_jupyter_describe_job_output(
                    scheduler=scheduler,
                    outputs=True,
                    training_job_response=training_job,
                    tag_dict=self.to_tag_dict(training_job["Tags"]),
                )
            )

        return ListJobsResponse(
            jobs=job_results,
            next_token=training_job_list_response.get("NextToken", None),
            total_count=-1,
        )

    def to_tag_dict(self, tag_list: List[Dict]):
        tag_dict = {}
        for tag_item in tag_list:
            tag_dict[tag_item["Key"]] = tag_item["Value"]

        return tag_dict

    def to_tag_list(self, tag_dict: Dict) -> List[Dict]:
        return [{"Key": key, "Value": value} for (key, value) in tag_dict.items()]

    def is_sagemaker_tag(self, tag_key: str) -> bool:
        return tag_key.startswith("sagemaker:")

    def extract_customer_tags_in_jupyter_format(self, tags: List[Dict]) -> List[str]:
        return [
            f"{tag['Key']}:{tag['Value']}"
            for tag in tags
            if not self.is_sagemaker_tag(tag["Key"])
        ]

    def to_jupyter_describe_job_definition_output(
        self,
        job_definition_id: str,
        describe_pipeline_response: Dict,
        describe_event_rule_response: Dict,
        list_tags_response: Dict,
    ) -> DescribeJobDefinition:
        training_job_details = json.loads(
            describe_pipeline_response["PipelineDefinition"]
        )["Steps"][0]["Arguments"]

        tag_dict = self.to_tag_dict(list_tags_response["Tags"])

        # TODO: OSS implementation is not extending CreateJob for CreateJobDefinition,
        # for some reason it wants to keep them separate. So not combining this code with DescribeJob

        s3_uri = (
            training_job_details.get("InputDataConfig", [{}])[0]
            .get("DataSource", {})
            .get("S3DataSource", {})
            .get("S3Uri")
        )

        s3_output = training_job_details.get("OutputDataConfig", {}).get("S3OutputPath")

        environment = training_job_details.get("Environment", {})

        return DescribeJobDefinition(
            output_formats=environment.get(
                JobEnvironmentVariableName.SM_OUTPUT_FORMATS.value, "ipynb"
            ).split(","),
            input_filename=environment.get(
                JobEnvironmentVariableName.SM_INPUT_NOTEBOOK_NAME.value, ""
            ),
            runtime_environment_name=environment.get(
                JobEnvironmentVariableName.SM_ENV_NAME.value, "sagemaker-default-env"
            ),
            name=tag_dict.get(JobTag.NAME.value, "Untitled Job"),
            parameters=training_job_details.get("HyperParameters"),
            runtime_environment_parameters=self._to_jupyter_runtime_environment_parameters(
                training_job_name=job_definition_id,
                training_job_response=training_job_details,
                s3_uri=s3_uri,
                s3_output=s3_output,
            ),
            tags=self.extract_customer_tags_in_jupyter_format(
                list_tags_response["Tags"]
            ),
            compute_type=training_job_details.get("ResourceConfig", {}).get(
                "InstanceType"
            ),
            schedule=str(
                GNUCronExpressionAdapter(
                    describe_event_rule_response["ScheduleExpression"]
                ).cron_expression
            ),
            timezone="UTC",
            job_definition_id=describe_pipeline_response["PipelineName"],
            create_time=describe_pipeline_response["CreationTime"].timestamp() * 1000,
            update_time=describe_pipeline_response["LastModifiedTime"].timestamp()
            * 1000,
            active=(
                describe_event_rule_response["State"]
                == EventBridgeRuleStatus.ENABLED.value
            ),
        )
