"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestCheck = void 0;
const path = require("path");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
/**
 * Represents a reference to a PullRequestCheck.
 */
class PullRequestCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { repository, buildSpec, buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_4_0, computeType = buildImage.defaultComputeType, privileged = false, updateApprovalState = true, postComment = true, projectName = `${repository.repositoryName}-pull-request`, role, } = props;
        this.pullRequestProject = new aws_codebuild_1.Project(this, 'PullRequestProject', {
            projectName,
            source: aws_codebuild_1.Source.codeCommit({
                repository,
            }),
            environment: {
                buildImage,
                computeType,
                privileged,
            },
            buildSpec,
            role,
        });
        if (updateApprovalState || postComment) {
            const codeBuildResultFunction = new aws_lambda_1.Function(this, 'CodeBuildResultFunction', {
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'code-build-result')),
                handler: 'index.handler',
                environment: {
                    UPDATE_APPROVAL_STATE: updateApprovalState ? 'TRUE' : 'FALSE',
                    POST_COMMENT: postComment ? 'TRUE' : 'FALSE',
                },
            });
            codeBuildResultFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: [repository.repositoryArn],
                actions: ['codecommit:PostCommentForPullRequest', 'codecommit:UpdatePullRequestApprovalState'],
            }));
            this.pullRequestProject.onStateChange('PullRequestValidationRule', {
                target: new aws_events_targets_1.LambdaFunction(codeBuildResultFunction),
            });
        }
        const rule = repository.onPullRequestStateChange(`${this.node.uniqueId}Rule`, {
            eventPattern: {
                detail: {
                    event: ['pullRequestSourceBranchUpdated', 'pullRequestCreated'],
                },
            },
        });
        rule.addTarget(new aws_events_targets_1.CodeBuildProject(this.pullRequestProject, {
            event: aws_events_1.RuleTargetInput.fromObject({
                sourceVersion: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                artifactsOverride: { type: 'NO_ARTIFACTS' },
                environmentVariablesOverride: [
                    {
                        name: 'pullRequestId',
                        value: aws_events_1.EventField.fromPath('$.detail.pullRequestId'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'repositoryName',
                        value: aws_events_1.EventField.fromPath('$.detail.repositoryNames[0]'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'sourceCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'destinationCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.destinationCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'revisionId',
                        value: aws_events_1.EventField.fromPath('$.detail.revisionId'),
                        type: 'PLAINTEXT',
                    },
                ],
            }),
        }));
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.pullRequestProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.pullRequestProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.pullRequestProject.onBuildSucceeded(id, options);
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        this.pullRequestProject.addToRolePolicy(statement);
    }
}
exports.PullRequestCheck = PullRequestCheck;
//# sourceMappingURL=data:application/json;base64,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