"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const assert_1 = require("./vendored/assert");
/**
 * (experimental) Partial and special matching during template assertions.
 *
 * @experimental
 */
class Match {
    /**
     * (experimental) Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @experimental
     */
    static absentProperty() {
        return assert_1.ABSENT;
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * (experimental) Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * (experimental) Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @experimental
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.123.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.push(this, [], `Expected type ${typeof this.pattern} but received ${getType(actual)}`);
            return result;
        }
        if (this.pattern === assert_1.ABSENT) {
            throw new Error('absentProperty() can only be used in an object matcher');
        }
        if (actual !== this.pattern) {
            result.push(this, [], `Expected ${this.pattern} but received ${actual}`);
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type array but received ${getType(actual)}`);
        }
        if (!this.partial && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected array of length ${this.pattern.length} but received ${actual.length}`);
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.partial || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.push(this, [], `Missing element${element}at pattern index ${patternIdx}`);
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type object but received ${getType(actual)}`);
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.push(this, [`/${a}`], 'Unexpected key');
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (patternVal === assert_1.ABSENT) {
                if (patternKey in actual) {
                    result.push(this, [`/${patternKey}`], 'Key should be absent');
                }
                continue;
            }
            if (!(patternKey in actual)) {
                result.push(this, [`/${patternKey}`], 'Missing key');
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.push(this, [], `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`);
        }
        return result;
    }
}
function getType(obj) {
    return Array.isArray(obj) ? 'array' : typeof obj;
}
//# sourceMappingURL=data:application/json;base64,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