// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import Input from '../../elements/Input';
import MultiToggleBox from '../../elements/MultiToggleBox';
import MultiToggleItem from '../../elements/MultiToggleItem';
import Select from '../../elements/Select';
import TextButton from '../../elements/TextButton';
import DropdownItem from '../../elements/DropdownItem';
import { StepType } from '../../../types';
import useSendEditOnClick from '../../../hooks/useSendEditOnClick';
import { toggleInArray } from '../../../utils/arrays';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import DefaultTaskpaneFooter from '../DefaultTaskpane/DefaultTaskpaneFooter';
import Spacer from '../../spacing/Spacer';
const getDefaultParams = (fileName) => {
    return {
        file_name: fileName,
        sheet_names: [],
        has_headers: true,
        skiprows: 0,
    };
};
const getButtonMessage = (params, loading) => {
    if (loading) {
        return `Importing...`;
    }
    else if (params.sheet_names.length === 0) {
        return `Select sheets to import them`;
    }
    return `Import ${params.sheet_names.length} Selected Sheet${params.sheet_names.length === 1 ? '' : 's'}`;
};
function getSuccessMessage(params) {
    return `Imported ${params.sheet_names.length} sheet${params.sheet_names.length === 1 ? '' : 's'}.`;
}
/*
    Allows a user to import an XLSX file with the given name, and
    in turn allows them to configure how to import sheets in this
    file.
*/
function XLSXImport(props) {
    // NOTE: this loading state is just for getting the metadata about the sheets
    // and not for importing the file
    const [fileMetadata, setFileMetadata] = useState({ sheet_names: [], size: 0, loading: true });
    const { params, setParams, loading, edit, editApplied } = useSendEditOnClick(() => getDefaultParams(props.fileName), StepType.ExcelImport, props.mitoAPI, props.analysisData, { allowSameParamsToReapplyTwice: true });
    useEffect(() => {
        const loadSheets = async () => {
            const loadedFileMetadata = await props.mitoAPI.getExcelFileMetadata(props.fileName) || { sheet_names: [], size: 0, loading: false };
            setFileMetadata(loadedFileMetadata);
            setParams(prevParams => {
                return Object.assign(Object.assign({}, prevParams), { sheet_names: loadedFileMetadata.sheet_names });
            });
        };
        void loadSheets();
    }, []);
    if (params === undefined) {
        return (React.createElement("div", { className: 'text-body-1' }, "There has been an error loading your Excel file metadata. Please try again, or contact support."));
    }
    const numSelectedSheets = params === null || params === void 0 ? void 0 : params.sheet_names.length;
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: `Import ${props.fileForImportWizard}`, setUIState: props.setUIState, backCallback: () => props.setFileForImportWizard(undefined) }),
        React.createElement(DefaultTaskpaneBody, { noScroll: true },
            React.createElement("div", null,
                React.createElement(MultiToggleBox, { loading: fileMetadata.loading, searchable: true, height: 'medium', toggleAllIndexes: (indexesToToggle) => {
                        setParams(prevParams => {
                            const newSheetNames = [...prevParams.sheet_names];
                            const sheetsToToggle = indexesToToggle.map(index => fileMetadata.sheet_names[index]);
                            sheetsToToggle.forEach(sheetName => {
                                toggleInArray(newSheetNames, sheetName);
                            });
                            return Object.assign(Object.assign({}, prevParams), { sheet_names: newSheetNames });
                        });
                    } }, fileMetadata.sheet_names.map((sheetName, idx) => {
                    return (React.createElement(MultiToggleItem, { key: idx, title: sheetName, toggled: params.sheet_names.includes(sheetName), onToggle: () => {
                            setParams(prevParams => {
                                const newSheetNames = [...prevParams.sheet_names];
                                toggleInArray(newSheetNames, sheetName);
                                return Object.assign(Object.assign({}, prevParams), { sheet_names: newSheetNames });
                            });
                        }, index: idx }));
                })),
                React.createElement("p", { className: 'text-body-1 mt-20px' }, "Has Header Row"),
                React.createElement(Select, { value: params.has_headers ? 'Yes' : 'No', onChange: (newValue) => setParams(prevParams => {
                        return Object.assign(Object.assign({}, prevParams), { has_headers: newValue === 'Yes' });
                    }) },
                    React.createElement(DropdownItem, { title: 'Yes' }),
                    React.createElement(DropdownItem, { title: 'No' })),
                React.createElement("p", { className: 'text-body-1 mt-20px' }, "Number of Rows to Skip"),
                React.createElement(Input, { value: "" + params.skiprows, type: 'number', onChange: (e) => {
                        const newValue = e.target.value;
                        setParams(prevParams => {
                            return Object.assign(Object.assign({}, prevParams), { skiprows: newValue });
                        });
                    } }),
                fileMetadata.size >= 100000 && fileMetadata.size < 10000000 &&
                    React.createElement("p", { className: "text-body-2 mt-20px" }, "Due to Python limitations, large Excel files take minutes to import."),
                fileMetadata.size >= 10000000 &&
                    React.createElement("p", { className: "text-body-2 mt-20px" }, "Due to Python limitations, massive Excel files take many minutes to import. If possible, save the Excel file as a CSV before importing."))),
        React.createElement(DefaultTaskpaneFooter, null,
            React.createElement(TextButton, { variant: 'dark', width: 'block', onClick: () => edit((params) => {
                    // Do a final parsing to make sure that the int is a valid number
                    const parsedSkipRows = parseInt("" + params.skiprows);
                    return Object.assign(Object.assign({}, params), { skiprows: parsedSkipRows });
                }), disabled: numSelectedSheets === 0, autoFocus: true }, getButtonMessage(params, loading)),
            editApplied && !loading &&
                React.createElement("p", { className: 'text-subtext-1' }, getSuccessMessage(params)),
            !editApplied &&
                React.createElement(Spacer, { px: 18 }))));
}
export default XLSXImport;
//# sourceMappingURL=XLSXImport.js.map