// Copyright (c) Mito
// Distributed under the terms of the Modified BSD License.
import React, { useEffect } from 'react';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import { StepType } from '../../../types';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import Row from '../../spacing/Row';
import Col from '../../spacing/Col';
import Select from '../../elements/Select';
import DefaultEmptyTaskpane from '../DefaultTaskpane/DefaultEmptyTaskpane';
import DropdownItem from '../../elements/DropdownItem';
import MultiToggleBox from '../../elements/MultiToggleBox';
import MultiToggleItem from '../../elements/MultiToggleItem';
import { getDisplayColumnHeader, getFirstCharactersOfColumnHeaders } from '../../../utils/columnHeaders';
import { getDtypeValue } from '../ControlPanel/FilterAndSortTab/DtypeCard';
import { addIfAbsent, intersection, removeIfPresent } from '../../../utils/arrays';
import Spacer from '../../spacing/Spacer';
import Input from '../../elements/Input';
import { isDatetimeDtype, isNumberDtype, isTimedeltaDtype } from '../../../utils/dtypes';
import useSendEditOnClick from '../../../hooks/useSendEditOnClick';
import TextButton from '../../elements/TextButton';
import { isOnlyNumberString } from '../../../utils/numbers';
// If the user inputs these, we cast them as booleans
const BOOLEAN_STRINGS = ['True', 'true', 'False', 'false'];
const getDefaultParams = (sheetDataArray, sheetIndex, defaultFillMethod, startingColumnIDs) => {
    if (sheetDataArray.length === 0 || sheetDataArray[sheetIndex] === undefined) {
        return undefined;
    }
    const sheetData = sheetDataArray[sheetIndex];
    let finalFillMethod = defaultFillMethod || { type: 'value', 'value': 0 };
    // We make sure that the default fill method is valid for the new dataframe we're selecting
    // which is only an issue if these are mean or median values
    if (finalFillMethod.type === 'mean' || finalFillMethod.type === 'median') {
        const onlyMeanAndMedianColumnSelected = Object.values(sheetData.columnDtypeMap)
            .map(columnDtype => isNumberDtype(columnDtype) || isDatetimeDtype(columnDtype) || isTimedeltaDtype(columnDtype))
            .every(hasDefinedMeanAndMedian => hasDefinedMeanAndMedian === true);
        if (!onlyMeanAndMedianColumnSelected) {
            finalFillMethod = { type: 'value', 'value': 0 };
        }
    }
    const columnIDs = startingColumnIDs === undefined ? Object.keys(sheetData.columnIDsMap) : intersection(Object.keys(sheetData.columnIDsMap), startingColumnIDs);
    return {
        sheet_index: sheetIndex,
        column_ids: columnIDs,
        fill_method: finalFillMethod
    };
};
/*
    Constructs a message in the case an edit is applied telling users
    fill na was successful on some columns
*/
const getButtonMessage = (sheetData, columnIDs) => {
    if (columnIDs.length === 0) {
        return 'Select columns to fill NaN values';
    }
    const columnHeaders = columnIDs.map(columnID => sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnIDsMap[columnID]).filter(columnHeader => columnHeader !== undefined);
    const [columnHeadersString, numOtherColumnHeaders] = getFirstCharactersOfColumnHeaders(columnHeaders, 25);
    if (numOtherColumnHeaders === 0) {
        return `Fill NaNs in ${columnHeadersString}`;
    }
    else {
        return `Fill NaNs in ${columnHeadersString} and ${numOtherColumnHeaders} others`;
    }
};
/*
    Constructs a message in the case an edit is applied telling users
    fill na was successful on some columns
*/
const getSuccessMessage = (sheetData, columnIDs) => {
    const columnHeaders = columnIDs.map(columnID => sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnIDsMap[columnID]).filter(columnHeader => columnHeader !== undefined);
    const [columnHeadersString, numOtherColumnHeaders] = getFirstCharactersOfColumnHeaders(columnHeaders, 25);
    if (numOtherColumnHeaders === 0) {
        return (React.createElement("p", null,
            "Filled NaNs in ",
            React.createElement("span", { className: 'text-color-gray-important' }, columnHeadersString),
            "."));
    }
    else {
        return (React.createElement("p", null,
            "Filled NaNs in ",
            React.createElement("span", { className: 'text-color-gray-important' }, columnHeadersString),
            " and ",
            React.createElement("span", { className: 'text-color-gray-important' }, numOtherColumnHeaders),
            " other columns."));
    }
};
/*
    A taskpane that allows users to fill NaN values in the sheet
*/
const FillNaTaskpane = (props) => {
    var _a;
    const { params, setParams, loading, edit, editApplied } = useSendEditOnClick(() => getDefaultParams(props.sheetDataArray, props.selectedSheetIndex, undefined, props.startingColumnIDs), StepType.FillNa, props.mitoAPI, props.analysisData);
    // If we change the starting column ids from outside the taskpane, then we 
    // update which columns are selected to fill nan in
    useEffect(() => {
        setParams(prevParams => {
            const newParams = getDefaultParams(props.sheetDataArray, props.selectedSheetIndex, prevParams.fill_method, props.startingColumnIDs);
            if (newParams) {
                return newParams;
            }
            return prevParams;
        });
    }, [props.startingColumnIDs]);
    if (params === undefined) {
        return (React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState, message: "Import a dataset before filling NaN values." }));
    }
    const sheetData = props.sheetDataArray[params.sheet_index];
    const columnIDsMap = (sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnIDsMap) || {};
    const columnDtypeMap = (sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnDtypeMap) || {};
    const onlyMeanAndMedianColumnSelected = params.column_ids.length === 0 || params.column_ids
        .map(columnID => columnDtypeMap[columnID])
        .filter(columnDtype => columnDtype !== undefined)
        .map(columnDtype => isNumberDtype(columnDtype) || isDatetimeDtype(columnDtype) || isTimedeltaDtype(columnDtype))
        .every(hasDefinedMeanAndMedian => hasDefinedMeanAndMedian === true);
    const toggleIndexes = (indexes, newToggle) => {
        var _a;
        const columnIds = Object.keys((_a = props.sheetDataArray[params.sheet_index]) === null || _a === void 0 ? void 0 : _a.columnIDsMap) || [];
        const columnIdsToToggle = indexes.map(index => columnIds[index]);
        const newColumnIds = [...params.column_ids];
        columnIdsToToggle.forEach(columnID => {
            if (newToggle) {
                addIfAbsent(newColumnIds, columnID);
            }
            else {
                removeIfPresent(newColumnIds, columnID);
            }
        });
        setParams(prevParams => {
            return Object.assign(Object.assign({}, prevParams), { column_ids: newColumnIds });
        });
    };
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: 'Fill NaN Values', setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, null,
            React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Select the dataframe to fill nan values in.' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Dataframe")),
                React.createElement(Col, null,
                    React.createElement(Select, { value: (_a = props.sheetDataArray[params.sheet_index]) === null || _a === void 0 ? void 0 : _a.dfName, onChange: (newDfName) => {
                            const newSheetIndex = props.sheetDataArray.findIndex((sheetData) => {
                                return sheetData.dfName == newDfName;
                            });
                            setParams(prevParams => {
                                const newParams = getDefaultParams(props.sheetDataArray, newSheetIndex, prevParams.fill_method);
                                if (newParams) {
                                    return newParams;
                                }
                                return Object.assign(Object.assign({}, prevParams), { sheet_index: newSheetIndex });
                            });
                        }, width: 'medium' }, props.sheetDataArray.map(sheetData => {
                        return (React.createElement(DropdownItem, { key: sheetData.dfName, title: sheetData.dfName }));
                    })))),
            React.createElement(Spacer, { px: 15 }),
            React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Select the columns to fill nan values in.' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Columns to Fill NaN Values In"))),
            React.createElement(MultiToggleBox, { searchable: true, toggleAllIndexes: toggleIndexes, height: 'medium' }, Object.entries(columnDtypeMap).map(([columnID, columnDtype], index) => {
                const columnHeader = columnIDsMap[columnID];
                const toggle = params.column_ids.includes(columnID);
                const disabled = (params.fill_method.type === 'mean' || params.fill_method.type === 'median') &&
                    !(isNumberDtype(columnDtype) || isTimedeltaDtype(columnDtype) || isDatetimeDtype(columnDtype));
                return (React.createElement(MultiToggleItem, { key: index, index: index, disabled: disabled, title: getDisplayColumnHeader(columnHeader), rightText: getDtypeValue(columnDtype), toggled: toggle, onToggle: () => {
                        toggleIndexes([index], !toggle);
                    } }));
            })),
            React.createElement(Spacer, { px: 15 }),
            React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Select the method for filling nan values' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Fill Method")),
                React.createElement(Col, null,
                    React.createElement(Select, { value: params.fill_method.type, onChange: (newFillMethodType) => {
                            setParams(prevConcatParams => {
                                let newFillMethod = { type: 'bfill' };
                                if (newFillMethodType === 'value') {
                                    newFillMethod = { type: 'value', value: 0 };
                                }
                                else {
                                    newFillMethod = { type: newFillMethodType };
                                }
                                return Object.assign(Object.assign({}, prevConcatParams), { fill_method: newFillMethod });
                            });
                        }, width: 'medium' },
                        React.createElement(DropdownItem, { id: 'value', title: 'Value', subtext: "Replaces NaN values with a specific value that you input." }),
                        React.createElement(DropdownItem, { id: 'ffill', title: "Forward Fill", subtext: "Replaces NaNs in the column with the value in the row before." }),
                        React.createElement(DropdownItem, { id: 'bfill', title: "Back Fill", subtext: "Replaces NaNs in the column with the value in the row after." }),
                        React.createElement(DropdownItem, { id: 'mean', title: "Column Mean", subtext: !onlyMeanAndMedianColumnSelected
                                ? "Only number, datetime, and timedetla columns support fill with mean."
                                : "Replaces NaN values in number columns with the average of the column.", disabled: !onlyMeanAndMedianColumnSelected }),
                        React.createElement(DropdownItem, { id: 'median', title: "Column Median", subtext: !onlyMeanAndMedianColumnSelected
                                ? "Only number, datetime, and timedetla columns support fill with median."
                                : "Replaces NaN values in number columns with the median of the column.", disabled: !onlyMeanAndMedianColumnSelected })))),
            params.fill_method.type === 'value' &&
                React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Select the dataframe to fill nan values in.' },
                    React.createElement(Col, null,
                        React.createElement("p", { className: 'text-header-3' }, "Fill Value")),
                    React.createElement(Col, null,
                        React.createElement(Input, { autoFocus: true, width: 'medium', value: '' + params.fill_method.value, onChange: (e) => {
                                const newValue = e.target.value;
                                setParams(prevParams => {
                                    return Object.assign(Object.assign({}, prevParams), { fill_method: {
                                            type: 'value',
                                            value: newValue
                                        } });
                                });
                            } }))),
            React.createElement(Spacer, { px: 10 + (params.fill_method.type === 'value' ? 0 : 38) }),
            React.createElement(TextButton, { variant: 'dark', width: 'block', onClick: () => {
                    // We check if the params have a string stored in them that could be a number,
                    // and if so we parse it to a number. This is a final tranform before the edit
                    edit((prevParams) => {
                        if (prevParams.fill_method.type === 'value' && typeof prevParams.fill_method.value === 'string') {
                            let finalValue = prevParams.fill_method.value;
                            if (BOOLEAN_STRINGS.includes(finalValue)) {
                                finalValue = finalValue.toLowerCase().startsWith('t') ? true : false;
                            }
                            else if (isOnlyNumberString(prevParams.fill_method.value)) {
                                finalValue = parseFloat(prevParams.fill_method.value);
                            }
                            return Object.assign(Object.assign({}, prevParams), { fill_method: { type: 'value', value: finalValue } });
                        }
                        return prevParams;
                    });
                }, disabled: params.column_ids.length === 0, disabledTooltip: "Select at least one column to fill NaN values in" }, getButtonMessage(sheetData, params.column_ids)),
            editApplied && !loading &&
                React.createElement(Row, { className: 'mt-5' },
                    React.createElement("p", { className: 'text-subtext-1' }, getSuccessMessage(sheetData, params.column_ids))))));
};
export default FillNaTaskpane;
//# sourceMappingURL=FillNaTaskpane.js.map