"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryNotificationEvents = exports.RepositoryEventTrigger = exports.Repository = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const codecommit_generated_1 = require("./codecommit.generated");
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.codecommit'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     *
     * @stability stable
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeCommit Repository State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     *
     * @stability stable
     */
    onReferenceCreated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     *
     * @stability stable
     */
    onReferenceUpdated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated', 'referenceUpdated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     *
     * @stability stable
     */
    onReferenceDeleted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceDeleted'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     *
     * @stability stable
     */
    onPullRequestStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Pull Request State Change'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    onCommentOnPullRequest(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Pull Request'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     *
     * @stability stable
     */
    onCommentOnCommit(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Commit'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     *
     * @stability stable
     */
    onCommit(id, options = {}) {
        const rule = this.onReferenceUpdated(id, options);
        if (options.branches) {
            rule.addEventPattern({ detail: { referenceName: options.branches } });
        }
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
        });
    }
    /**
     * Grant the given identity permissions to pull this repository.
     *
     * @stability stable
     */
    grantPull(grantee) {
        return this.grant(grantee, 'codecommit:GitPull');
    }
    /**
     * Grant the given identity permissions to pull and push this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:GitPush');
    }
    /**
     * Grant the given identity permissions to read this repository.
     *
     * @stability stable
     */
    grantRead(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:EvaluatePullRequestApprovalRules', 'codecommit:Get*', 'codecommit:Describe*');
    }
    /**
     * Defines a CodeStar Notification rule triggered when the project events specified by you are emitted. Similar to `onEvent` API.
     *
     * You can also use the methods to define rules for the specific event emitted.
     * eg: `notifyOnPullRequstCreated`.
     *
     * @stability stable
     */
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    notifyOnPullRequestComment(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_COMMENT],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when an approval status is changed.
     *
     * @stability stable
     */
    notifyOnApprovalStatusChanged(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.APPROVAL_STATUS_CHANGED],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when an approval rule is overridden.
     *
     * @stability stable
     */
    notifyOnApprovalRuleOverridden(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.APPROVAL_RULE_OVERRIDDEN],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is created.
     *
     * @stability stable
     */
    notifyOnPullRequestCreated(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_CREATED],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is merged.
     *
     * @stability stable
     */
    notifiyOnPullRequestMerged(id, target, options) {
        return this.notifyOnPullRequestMerged(id, target, options);
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is merged.
     *
     * @stability stable
     */
    notifyOnPullRequestMerged(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_MERGED],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a new branch or tag is created.
     *
     * @stability stable
     */
    notifyOnBranchOrTagCreated(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.BRANCH_OR_TAG_CREATED],
        });
    }
    /**
     * Defines a CodeStar Notification rule which triggers when a branch or tag is deleted.
     *
     * @stability stable
     */
    notifyOnBranchOrTagDeleted(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.BRANCH_OR_TAG_DELETED],
        });
    }
    /**
     * Returns a source configuration for notification rule.
     *
     * @stability stable
     */
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.repositoryArn,
        };
    }
}
/**
 * Provides a CodeCommit Repository.
 *
 * @stability stable
 */
class Repository extends RepositoryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.triggers = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_codecommit_RepositoryProps(props);
        const repository = new codecommit_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: props.repositoryName,
            repositoryDescription: props.description,
            triggers: core_1.Lazy.any({ produce: () => this.triggers }, { omitEmptyArray: true }),
            code: (_c = ((_b = props.code) === null || _b === void 0 ? void 0 : _b.bind(this))) === null || _c === void 0 ? void 0 : _c.code,
        });
        this.repositoryName = this.getResourceNameAttribute(repository.attrName);
        this.repositoryArn = this.getResourceArnAttribute(repository.attrArn, {
            service: 'codecommit',
            resource: this.physicalName,
        });
        this.repositoryCloneUrlHttp = repository.attrCloneUrlHttp;
        this.repositoryCloneUrlSsh = repository.attrCloneUrlSsh;
        this.repositoryCloneUrlGrc = makeCloneUrl(core_1.Stack.of(this), this.repositoryName, 'grc');
    }
    /**
     * Imports a codecommit repository.
     *
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`).
     * @stability stable
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.splitArn(repositoryArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const repositoryName = arn.resource;
        const region = arn.region;
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryArn = repositoryArn;
                this.repositoryName = repositoryName;
                this.repositoryCloneUrlHttp = makeCloneUrl(stack, repositoryName, 'https', region);
                this.repositoryCloneUrlSsh = makeCloneUrl(stack, repositoryName, 'ssh', region);
                this.repositoryCloneUrlGrc = makeCloneUrl(stack, repositoryName, 'grc', region);
            }
        }
        return new Import(scope, id, {
            account: arn.account,
            region,
        });
    }
    /**
     * @stability stable
     */
    static fromRepositoryName(scope, id, repositoryName) {
        const stack = core_1.Stack.of(scope);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = core_1.Stack.of(scope).formatArn({
                    service: 'codecommit',
                    resource: repositoryName,
                });
                this.repositoryCloneUrlHttp = makeCloneUrl(stack, repositoryName, 'https');
                this.repositoryCloneUrlSsh = makeCloneUrl(stack, repositoryName, 'ssh');
                this.repositoryCloneUrlGrc = makeCloneUrl(stack, repositoryName, 'grc');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Create a trigger to notify another service to run actions on repository events.
     *
     * @param arn Arn of the resource that repository events will notify.
     * @param options Trigger options to run actions.
     * @stability stable
     */
    notify(arn, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_codecommit_RepositoryTriggerOptions(options);
        let evt = options && options.events;
        if (evt && evt.length > 1 && evt.indexOf(RepositoryEventTrigger.ALL) > -1) {
            evt = [RepositoryEventTrigger.ALL];
        }
        const customData = options && options.customData;
        const branches = options && options.branches;
        let name = options && options.name;
        if (!name) {
            name = this.node.path + '/' + arn;
        }
        if (this.triggers.find(prop => prop.name === name)) {
            throw new Error(`Unable to set repository trigger named ${name} because trigger names must be unique`);
        }
        this.triggers.push({
            destinationArn: arn,
            name,
            customData,
            branches,
            events: evt || [RepositoryEventTrigger.ALL],
        });
        return this;
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "@aws-cdk/aws-codecommit.Repository", version: "1.140.0" };
/**
 * Repository events that will cause the trigger to run actions in another service.
 *
 * @stability stable
 */
var RepositoryEventTrigger;
(function (RepositoryEventTrigger) {
    RepositoryEventTrigger["ALL"] = "all";
    RepositoryEventTrigger["UPDATE_REF"] = "updateReference";
    RepositoryEventTrigger["CREATE_REF"] = "createReference";
    RepositoryEventTrigger["DELETE_REF"] = "deleteReference";
})(RepositoryEventTrigger = exports.RepositoryEventTrigger || (exports.RepositoryEventTrigger = {}));
/**
 * Returns the clone URL for a protocol.
 */
function makeCloneUrl(stack, repositoryName, protocol, region) {
    switch (protocol) {
        case 'https':
        case 'ssh':
            return `${protocol}://git-codecommit.${region !== null && region !== void 0 ? region : stack.region}.${stack.urlSuffix}/v1/repos/${repositoryName}`;
        case 'grc':
            return `codecommit::${region !== null && region !== void 0 ? region : stack.region}://${repositoryName}`;
    }
}
/**
 * List of event types for AWS CodeCommit.
 *
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-repositories
 * @stability stable
 */
var RepositoryNotificationEvents;
(function (RepositoryNotificationEvents) {
    RepositoryNotificationEvents["COMMIT_COMMENT"] = "codecommit-repository-comments-on-commits";
    RepositoryNotificationEvents["PULL_REQUEST_COMMENT"] = "codecommit-repository-comments-on-pull-requests";
    RepositoryNotificationEvents["APPROVAL_STATUS_CHANGED"] = "codecommit-repository-approvals-status-changed";
    RepositoryNotificationEvents["APPROVAL_RULE_OVERRIDDEN"] = "codecommit-repository-approvals-rule-override";
    RepositoryNotificationEvents["PULL_REQUEST_CREATED"] = "codecommit-repository-pull-request-created";
    RepositoryNotificationEvents["PULL_REQUEST_SOURCE_UPDATED"] = "codecommit-repository-pull-request-source-updated";
    RepositoryNotificationEvents["PULL_REQUEST_STATUS_CHANGED"] = "codecommit-repository-pull-request-status-changed";
    RepositoryNotificationEvents["PULL_REQUEST_MERGED"] = "codecommit-repository-pull-request-merged";
    RepositoryNotificationEvents["BRANCH_OR_TAG_CREATED"] = "codecommit-repository-branches-and-tags-created";
    RepositoryNotificationEvents["BRANCH_OR_TAG_DELETED"] = "codecommit-repository-branches-and-tags-deleted";
    RepositoryNotificationEvents["BRANCH_OR_TAG_UPDATED"] = "codecommit-repository-branches-and-tags-updated";
})(RepositoryNotificationEvents = exports.RepositoryNotificationEvents || (exports.RepositoryNotificationEvents = {}));
//# sourceMappingURL=data:application/json;base64,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