"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("assert");
const assert_2 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const glue = require("../lib");
test('default database does not create a bucket', () => {
    const stack = new core_1.Stack();
    new glue.Database(stack, 'Database', {
        databaseName: 'test_database',
    });
    assert_2.expect(stack).toMatch({
        Resources: {
            DatabaseB269D8BB: {
                Type: 'AWS::Glue::Database',
                Properties: {
                    CatalogId: {
                        Ref: 'AWS::AccountId',
                    },
                    DatabaseInput: {
                        Name: 'test_database',
                    },
                },
            },
        },
    });
});
test('explicit locationURI', () => {
    const stack = new core_1.Stack();
    new glue.Database(stack, 'Database', {
        databaseName: 'test_database',
        locationUri: 's3://my-uri/',
    });
    assert_2.expect(stack).toMatch({
        Resources: {
            DatabaseB269D8BB: {
                Type: 'AWS::Glue::Database',
                Properties: {
                    CatalogId: {
                        Ref: 'AWS::AccountId',
                    },
                    DatabaseInput: {
                        LocationUri: 's3://my-uri/',
                        Name: 'test_database',
                    },
                },
            },
        },
    });
});
test('fromDatabase', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const database = glue.Database.fromDatabaseArn(stack, 'import', 'arn:aws:glue:us-east-1:123456789012:database/db1');
    // THEN
    assert_1.deepEqual(database.databaseArn, 'arn:aws:glue:us-east-1:123456789012:database/db1');
    assert_1.deepEqual(database.databaseName, 'db1');
    assert_1.deepEqual(stack.resolve(database.catalogArn), {
        'Fn::Join': ['',
            ['arn:', { Ref: 'AWS::Partition' }, ':glue:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':catalog']],
    });
    assert_1.deepEqual(stack.resolve(database.catalogId), { Ref: 'AWS::AccountId' });
});
test('locationUri length must be >= 1', () => {
    const stack = new core_1.Stack();
    assert_1.throws(() => new glue.Database(stack, 'Database', {
        databaseName: 'test_database',
        locationUri: '',
    }));
});
test('locationUri length must be <= 1024', () => {
    const stack = new core_1.Stack();
    assert_1.throws(() => new glue.Database(stack, 'Database', {
        databaseName: 'test_database',
        locationUri: 'a'.repeat(1025),
    }));
});
//# sourceMappingURL=data:application/json;base64,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