/****************************************************************************
 *
 *   Copyright (c) 2013-2016 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/**
 * @file fw_pos_control_l1_params.c
 *
 * Parameters defined by the L1 position control task
 *
 * @author Lorenz Meier <lorenz@px4.io>
 */

/*
 * Controller parameters, accessible via MAVLink
 */

/**
 * L1 period
 *
 * Used to determine the L1 gain and controller time constant. This parameter is
 * proportional to the L1 distance (which points ahead of the aircraft on the path
 * it is following). A value of 18-25 seconds works for most aircraft. Shorten
 * slowly during tuning until response is sharp without oscillation.
 *
 * @unit s
 * @min 7.0
 * @max 50.0
 * @decimal 1
 * @increment 0.5
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_L1_PERIOD, 20.0f);

/**
 * L1 damping
 *
 * Damping factor for L1 control.
 *
 * @min 0.6
 * @max 0.9
 * @decimal 2
 * @increment 0.05
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_L1_DAMPING, 0.75f);

/**
 * L1 controller roll slew rate limit.
 *
 * The maximum change in roll angle setpoint per second.
 *
 * @unit deg/s
 * @min 0
 * @decimal 0
 * @increment 1
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_L1_R_SLEW_MAX, 90.0f);

/**
 * Use NPFG as lateral-directional guidance law for fixed-wing vehicles
 *
 * Replaces L1.
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(FW_USE_NPFG, 0);

/**
 * NPFG period
 *
 * Period of the NPFG control law.
 *
 * @unit s
 * @min 1.0
 * @max 100.0
 * @decimal 1
 * @increment 0.1
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_PERIOD, 10.0f);

/**
 * NPFG damping ratio
 *
 * Damping ratio of the NPFG control law.
 *
 * @min 0.10
 * @max 1.00
 * @decimal 2
 * @increment 0.01
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_DAMPING, 0.7f);

/**
 * Enable automatic lower bound on the NPFG period
 *
 * Avoids limit cycling from a too aggressively tuned period/damping combination.
 * If set to false, also disables the upper bound NPFG_PERIOD_UB.
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(NPFG_LB_PERIOD, 1);

/**
 * Enable automatic upper bound on the NPFG period
 *
 * Adapts period to maintain track keeping in variable winds and path curvature.
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(NPFG_UB_PERIOD, 1);

/**
 * Enable track keeping excess wind handling logic.
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(NPFG_TRACK_KEEP, 1);

/**
 * Enable minimum forward ground speed maintaining excess wind handling logic
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(NPFG_EN_MIN_GSP, 1);

/**
 * Enable wind excess regulation.
 *
 * Disabling this parameter further disables all other airspeed incrementation options.
 *
 * @boolean
 * @group FW NPFG Control
 */
PARAM_DEFINE_INT32(NPFG_WIND_REG, 1);

/**
 * Maximum, minimum forward ground speed for track keeping in excess wind
 *
 * The maximum value of the minimum forward ground speed that may be commanded
 * by the track keeping excess wind handling logic. Commanded in full at the normalized
 * track error fraction of the track error boundary and reduced to zero on track.
 *
 * @unit m/s
 * @min 0.0
 * @max 10.0
 * @decimal 1
 * @increment 0.5
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_GSP_MAX_TK, 5.0f);

/**
 * Roll time constant
 *
 * Time constant of roll controller command / response, modeled as first order delay.
 * Used to determine lower period bound. Setting zero disables automatic period bounding.
 *
 * @unit s
 * @min 0.00
 * @max 2.00
 * @decimal 2
 * @increment 0.05
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_ROLL_TC, 0.5f);

/**
 * NPFG switch distance multiplier
 *
 * Multiplied by the track error boundary to determine when the aircraft switches
 * to the next waypoint and/or path segment. Should be less than 1. 1/pi (0.32)
 * sets the switch distance equivalent to that of the L1 controller.
 *
 * @min 0.1
 * @max 1.0
 * @decimal 2
 * @increment 0.01
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_SW_DST_MLT, 0.32f);

/**
 * Period safety factor
 *
 * Multiplied by period for conservative minimum period bounding (when period lower
 * bounding is enabled). 1.0 bounds at marginal stability.
 *
 * @min 1.0
 * @max 10.0
 * @decimal 1
 * @increment 0.1
 * @group FW NPFG Control
 */
PARAM_DEFINE_FLOAT(NPFG_PERIOD_SF, 1.5f);

/**
 * Trim throttle
 *
 * This is the throttle setting required to achieve FW_AIRSPD_TRIM during level flight. Most airframes have a value of 0.5-0.7.
 *
 * @unit norm
 * @min 0.0
 * @max 1.0
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_THR_TRIM, 0.6f);

/**
 * Throttle max slew rate
 *
 * Maximum slew rate for the commanded throttle
 *
 * @min 0.0
 * @max 1.0
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_THR_SLEW_MAX, 0.0f);

/**
 * Minimum pitch angle
 *
 * The minimum pitch angle setpoint for autonomous modes including altitude and position control.
 *
 * @unit deg
 * @min -60.0
 * @max 0.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_P_LIM_MIN, -30.0f);

/**
 * Maximum pitch angle
 *
 * The maximum pitch angle setpoint for autonomous modes including altitude and position control.
 *
 * @unit deg
 * @min 0.0
 * @max 60.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_P_LIM_MAX, 30.0f);

/**
 * Maximum roll angle
 *
 * The maximum roll angle setpoint for autonomous modes including altitude and position control.
 *
 * @unit deg
 * @min 35.0
 * @max 65.0
 * @decimal 1
 * @increment 0.5
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_R_LIM, 50.0f);

/**
 * Throttle limit max
 *
 * This is the maximum throttle % that can be used by the controller.
 * For overpowered aircraft, this should be reduced to a value that
 * provides sufficient thrust to climb at the maximum pitch angle PTCH_MAX.
 *
 * @unit norm
 * @min 0.0
 * @max 1.0
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_THR_MAX, 1.0f);

/**
 * Throttle limit min
 *
 * This is the minimum throttle % that can be used by the controller.
 * For electric aircraft this will normally be set to zero, but can be set
 * to a small non-zero value if a folding prop is fitted to prevent the
 * prop from folding and unfolding repeatedly in-flight or to provide
 * some aerodynamic drag from a turning prop to improve the descent rate.
 *
 * For aircraft with internal combustion engine this parameter should be set
 * for desired idle rpm.
 *
 * @unit norm
 * @min 0.0
 * @max 1.0
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_THR_MIN, 0.0f);

/**
 * Idle throttle
 *
 * This is the minimum throttle while on the ground
 *
 * For aircraft with internal combustion engines, this parameter should be set
 * above the desired idle rpm. For electric motors, idle should typically be set
 * to zero.
 *
 * Note that in automatic modes, "landed" conditions will engage idle throttle.
 *
 * @unit norm
 * @min 0.0
 * @max 0.4
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_THR_IDLE, 0.0f);

/**
 * Climbout Altitude difference
 *
 * If the altitude error exceeds this parameter, the system will climb out
 * with maximum throttle and minimum airspeed until it is closer than this
 * distance to the desired altitude. Mostly used for takeoff waypoints / modes.
 * Set to 0 to disable climbout mode (not recommended).
 *
 * @unit m
 * @min 0.0
 * @max 150.0
 * @decimal 1
 * @increment 0.5
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_CLMBOUT_DIFF, 10.0f);

/**
 * Maximum landing slope angle
 *
 * Typically the desired landing slope angle when landing configuration (flaps, airspeed) is enabled.
 * Set this value within the vehicle's performance limits.
 *
 * @unit deg
 * @min 1.0
 * @max 15.0
 * @decimal 1
 * @increment 0.5
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_ANG, 5.0f);

/**
 * Minimum pitch during takeoff.
 *
 * @unit deg
 * @min -5.0
 * @max 30.0
 * @decimal 1
 * @increment 0.5
 * @group FW L1 Control
 */
PARAM_DEFINE_FLOAT(FW_TKO_PITCH_MIN, 10.0f);

/**
 * Landing flare altitude (relative to landing altitude)
 *
 * NOTE: max(FW_LND_FLALT, FW_LND_FL_TIME * |z-velocity|) is taken as the flare altitude
 *
 * @unit m
 * @min 0.0
 * @decimal 1
 * @increment 0.5
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_FLALT, 0.5f);

/**
 * Use terrain estimation during landing. This is critical for detecting when to flare, and should be enabled if possible.
 *
 * NOTE: terrain estimate is currently solely derived from a distance sensor.
 *
 * If enabled and no measurement is found within a given timeout, the landing waypoint altitude will be used OR the landing
 * will be aborted, depending on the criteria set in FW_LND_ABORT.
 *
 * If disabled, FW_LND_ABORT terrain based criteria are ignored.
 *
 * @min 0
 * @max 2
 * @value 0 Disable the terrain estimate
 * @value 1 Use the terrain estimate to trigger the flare (only)
 * @value 2 Calculate landing glide slope relative to the terrain estimate
 * @group FW Auto Landing
 */
PARAM_DEFINE_INT32(FW_LND_USETER, 1);

/**
 * Early landing configuration deployment
 *
 * When disabled, the landing configuration (flaps, landing airspeed, etc.) is only activated
 * on the final approach to landing. When enabled, it is already activated when entering the
 * final loiter-down (loiter-to-alt) waypoint before the landing approach. This shifts the (often large)
 * altitude and airspeed errors caused by the configuration change away from the ground such that
 * these are not so critical. It also gives the controller enough time to adapt to the new
 * configuration such that the landing approach starts with a cleaner initial state.
 *
 * @boolean
 *
 * @group FW Auto Landing
 */
PARAM_DEFINE_INT32(FW_LND_EARLYCFG, 0);

/**
 * Flare, minimum pitch
 *
 * Minimum pitch during flare, a positive sign means nose up
 * Applied once flaring is triggered
 *
 * @unit deg
 * @min 0
 * @max 15.0
 * @decimal 1
 * @increment 0.5
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_FL_PMIN, 2.5f);

/**
 * Flare, maximum pitch
 *
 * Maximum pitch during flare, a positive sign means nose up
 * Applied once flaring is triggered
 *
 * @unit deg
 * @min 0
 * @max 45.0
 * @decimal 1
 * @increment 0.5
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_FL_PMAX, 15.0f);

/**
 * Min. airspeed scaling factor for landing
 *
 * Multiplying this factor with the minimum airspeed of the plane
 * gives the target airspeed the landing approach.
 * FW_AIRSPD_MIN * FW_LND_AIRSPD_SC
 *
 * @unit norm
 * @min 1.0
 * @max 1.5
 * @decimal 2
 * @increment 0.01
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_AIRSPD_SC, 1.3f);

/**
 * Altitude time constant factor for landing
 *
 * Set this parameter to less than 1.0 to make TECS react faster to altitude errors during
 * landing than during normal flight (i.e. giving efficiency and low motor wear at
 * high altitudes but control accuracy during landing). During landing, the TECS
 * altitude time constant (FW_T_ALT_TC) is multiplied by this value.
 *
 * @unit
 * @min 0.2
 * @max 1.0
 * @increment 0.1
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_THRTC_SC, 1.0f);



/*
 * TECS parameters
 *
 */


/**
 * Minimum Airspeed (CAS)
 *
 * The minimal airspeed (calibrated airspeed) the user is able to command.
 * Further, if the airspeed falls below this value, the TECS controller will try to
 * increase airspeed more aggressively.
 * Has to be set according to the vehicle's stall speed (which should be set in FW_AIRSPD_STALL),
 * with some margin between the stall speed and minimum airspeed.
 * This value corresponds to the desired minimum speed with the default load factor (level flight, default weight),
 * and is automatically adpated to the current load factor (calculated from roll setpoint and WEIGHT_GROSS/WEIGHT_BASE).
 *
 * @unit m/s
 * @min 0.5
 * @max 40
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_AIRSPD_MIN, 10.0f);

/**
 * Maximum Airspeed (CAS)
 *
 * The maximal airspeed (calibrated airspeed) the user is able to command.
 * Further, if the airspeed is above this value, the TECS controller will try to decrease
 * airspeed more aggressively.
 *
 * @unit m/s
 * @min 0.5
 * @max 40
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_AIRSPD_MAX, 20.0f);

/**
 * Cruise Airspeed (CAS)
 *
 * The trim CAS (calibrated airspeed) of the vehicle. If an airspeed controller is active,
 * this is the default airspeed setpoint that the controller will try to achieve if
 * no other airspeed setpoint sources are present (e.g. through non-centered RC sticks).
 *
 * @unit m/s
 * @min 0.5
 * @max 40
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_AIRSPD_TRIM, 15.0f);

/**
 * Stall Airspeed (CAS)
 *
 * The stall airspeed (calibrated airspeed) of the vehicle.
 * It is used for airspeed sensor failure detection and for the control
 * surface scaling airspeed limits.
 *
 * @unit m/s
 * @min 0.5
 * @max 40
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_AIRSPD_STALL, 7.0f);

/**
 * Maximum climb rate
 *
 * This is the best climb rate that the aircraft can achieve with
 * the throttle set to THR_MAX and the airspeed set to the
 * default value. For electric aircraft make sure this number can be
 * achieved towards the end of flight when the battery voltage has reduced.
 * The setting of this parameter can be checked by commanding a positive
 * altitude change of 100m in loiter, RTL or guided mode. If the throttle
 * required to climb is close to THR_MAX and the aircraft is maintaining
 * airspeed, then this parameter is set correctly. If the airspeed starts
 * to reduce, then the parameter is set to high, and if the throttle
 * demand required to climb and maintain speed is noticeably less than
 * FW_THR_MAX, then either FW_T_CLMB_MAX should be increased or
 * FW_THR_MAX reduced.
 *
 * @unit m/s
 * @min 1.0
 * @max 15.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_CLMB_MAX, 5.0f);

/**
 * Minimum descent rate
 *
 * This is the sink rate of the aircraft with the throttle
 * set to THR_MIN and flown at the same airspeed as used
 * to measure FW_T_CLMB_MAX.
 *
 * @unit m/s
 * @min 1.0
 * @max 5.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SINK_MIN, 2.0f);

/**
 * Maximum descent rate
 *
 * This sets the maximum descent rate that the controller will use.
 * If this value is too large, the aircraft can over-speed on descent.
 * This should be set to a value that can be achieved without
 * exceeding the lower pitch angle limit and without over-speeding
 * the aircraft.
 *
 * @unit m/s
 * @min 1.0
 * @max 15.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SINK_MAX, 5.0f);

/**
 * Throttle damping factor
 *
 * This is the damping gain for the throttle demand loop.
 * Increase to add damping to correct for oscillations in speed and height.
 *
 * @min 0.0
 * @max 2.0
 * @decimal 2
 * @increment 0.1
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_THR_DAMP, 0.1f);

/**
 * Integrator gain throttle
 *
 * This is the integrator gain on the throttle part of the control loop.
 * Increasing this gain increases the speed at which speed
 * and height offsets are trimmed out, but reduces damping and
 * increases overshoot. Set this value to zero to completely
 * disable all integrator action.
 *
 * @min 0.0
 * @max 2.0
 * @decimal 2
 * @increment 0.05
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_I_GAIN_THR, 0.3f);

/**
 * Integrator gain pitch
 *
 * This is the integrator gain on the pitch part of the control loop.
 * Increasing this gain increases the speed at which speed
 * and height offsets are trimmed out, but reduces damping and
 * increases overshoot. Set this value to zero to completely
 * disable all integrator action.
 *
 * @min 0.0
 * @max 2.0
 * @decimal 2
 * @increment 0.05
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_I_GAIN_PIT, 0.1f);

/**
 * Maximum vertical acceleration
 *
 * This is the maximum vertical acceleration (in m/s/s)
 * either up or down that the controller will use to correct speed
 * or height errors. The default value of 7 m/s/s (equivalent to +- 0.7 g)
 * allows for reasonably aggressive pitch changes if required to recover
 * from under-speed conditions.
 *
 * @unit m/s^2
 * @min 1.0
 * @max 10.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_VERT_ACC, 7.0f);

/**
 * Complementary filter "omega" parameter for speed
 *
 * This is the cross-over frequency (in radians/second) of the complementary
 * filter used to fuse longitudinal acceleration and airspeed to obtain an
 * improved airspeed estimate. Increasing this frequency weights the solution
 * more towards use of the airspeed sensor, whilst reducing it weights the
 * solution more towards use of the accelerometer data.
 *
 * @unit rad/s
 * @min 1.0
 * @max 10.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SPD_OMEGA, 2.0f);

/**
 * Roll -> Throttle feedforward
 *
 * Increasing this gain turn increases the amount of throttle that will
 * be used to compensate for the additional drag created by turning.
 * Ideally this should be set to  approximately 10 x the extra sink rate
 * in m/s created by a 45 degree bank turn. Increase this gain if
 * the aircraft initially loses energy in turns and reduce if the
 * aircraft initially gains energy in turns. Efficient high aspect-ratio
 * aircraft (eg powered sailplanes) can use a lower value, whereas
 * inefficient low aspect-ratio models (eg delta wings) can use a higher value.
 *
 * @min 0.0
 * @max 20.0
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_RLL2THR, 15.0f);

/**
 * Speed <--> Altitude priority
 *
 * This parameter adjusts the amount of weighting that the pitch control
 * applies to speed vs height errors. Setting it to 0.0 will cause the
 * pitch control to control height and ignore speed errors. This will
 * normally improve height accuracy but give larger airspeed errors.
 * Setting it to 2.0 will cause the pitch control loop to control speed
 * and ignore height errors. This will normally reduce airspeed errors,
 * but give larger height errors. The default value of 1.0 allows the pitch
 * control to simultaneously control height and speed.
 * Note to Glider Pilots - set this parameter to 2.0 (The glider will
 * adjust its pitch angle to maintain airspeed, ignoring changes in height).
 *
 * @min 0.0
 * @max 2.0
 * @decimal 1
 * @increment 1.0
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SPDWEIGHT, 1.0f);

/**
 * Pitch damping factor
 *
 * This is the damping gain for the pitch demand loop. Increase to add
 * damping to correct for oscillations in height. The default value of 0.0
 * will work well provided the pitch to servo controller has been tuned
 * properly.
 *
 * @min 0.0
 * @max 2.0
 * @decimal 2
 * @increment 0.1
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_PTCH_DAMP, 0.1f);

/**
 * Altitude error time constant.
 *
 * @min 2.0
 * @decimal 2
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_ALT_TC, 5.0f);

/**
 * Height rate feed forward
 *
 * @min 0.0
 * @max 1.0
 * @decimal 2
 * @increment 0.05
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_HRATE_FF, 0.3f);

/**
 * True airspeed error time constant.
 *
 * @min 2.0
 * @decimal 2
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_TAS_TC, 5.0f);

/**
 * Minimum groundspeed
 *
 * The controller will increase the commanded airspeed to maintain
 * this minimum groundspeed to the next waypoint.
 *
 * @unit m/s
 * @min 0.0
 * @max 40
 * @decimal 1
 * @increment 0.5
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_GND_SPD_MIN, 5.0f);

/**
 * RC stick configuration fixed-wing.
 *
 * Set RC/joystick configuration for fixed-wing manual position and altitude controlled flight.
 *
 * @min 0
 * @max 3
 * @bit 0 Alternative stick configuration (height rate on throttle stick, airspeed on pitch stick)
 * @bit 1 Enable airspeed setpoint via sticks in altitude and position flight mode
 * @group FW L1 Control
 */
PARAM_DEFINE_INT32(FW_POS_STK_CONF, 2);

/**
 * Specific total energy rate first order filter time constant.
 *
 * This filter is applied to the specific total energy rate used for throttle damping.
 *
 * @min 0.0
 * @max 2
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_STE_R_TC, 0.4f);


/**
 * True airspeed rate first order filter time constant.
 *
 * This filter is applied to the true airspeed rate.
 *
 * @min 0.0
 * @max 2
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_TAS_R_TC, 0.2f);


/**
 * Specific total energy balance rate feedforward gain.
 *
 *
 * @min 0.5
 * @max 3
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SEB_R_FF, 1.0f);

/**
 * Default target climbrate.
 *
 *
 * The default rate at which the vehicle will climb in autonomous modes to achieve altitude setpoints.
 * In manual modes this defines the maximum rate at which the altitude setpoint can be increased.
 *
 *
 * @unit m/s
 * @min 0.5
 * @max 15
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_CLMB_R_SP, 3.0f);

/**
 * Default target sinkrate.
 *
 *
 * The default rate at which the vehicle will sink in autonomous modes to achieve altitude setpoints.
 * In manual modes this defines the maximum rate at which the altitude setpoint can be decreased.
 *
 * @unit m/s
 * @min 0.5
 * @max 15
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_T_SINK_R_SP, 2.0f);

/**
 * GPS failure loiter time
 *
 * The time in seconds the system should do open loop loiter and wait for GPS recovery
 * before it starts descending. Set to 0 to disable. Roll angle is set to FW_GPSF_R.
 * Does only apply for fixed-wing vehicles or VTOLs with NAV_FORCE_VT set to 0.
 *
 * @unit s
 * @min 0
 * @max 3600
 * @group Mission
 */
PARAM_DEFINE_INT32(FW_GPSF_LT, 30);

/**
 * GPS failure fixed roll angle
 *
 * Roll in degrees during the loiter after the vehicle has lost GPS in an auto mode (e.g. mission or loiter).
 * Does only apply for fixed-wing vehicles or VTOLs with NAV_FORCE_VT set to 0.
 *
 * @unit deg
 * @min 0.0
 * @max 30.0
 * @decimal 1
 * @increment 0.5
 * @group Mission
 */
PARAM_DEFINE_FLOAT(FW_GPSF_R, 15.0f);

/**
 * Vehicle base weight.
 *
 * This is the weight of the vehicle at which it's performance limits were derived. A zero or negative value
 * disables trim throttle and minimum airspeed compensation based on weight.
 *
 * @unit kg
 * @decimal 1
 * @increment 0.5
 * @group Mission
 */
PARAM_DEFINE_FLOAT(WEIGHT_BASE, -1.0f);

/**
 * Vehicle gross weight.
 *
 * This is the actual weight of the vehicle at any time. This value will differ from WEIGHT_BASE in case weight was added
 * or removed from the base weight. Examples are the addition of payloads or larger batteries. A zero or negative value
 * disables trim throttle and minimum airspeed compensation based on weight.
 *
 * @unit kg
 * @decimal 1
 * @increment 0.1
 * @group Mission
 */
PARAM_DEFINE_FLOAT(WEIGHT_GROSS, -1.0f);

/**
 * The aircraft's wing span (length from tip to tip).
 *
 * This is used for limiting the roll setpoint near the ground. (if multiple wings, take the longest span)
 *
 * @unit m
 * @min 0.1
 * @decimal 1
 * @increment 0.1
 * @group FW Geometry
 */
PARAM_DEFINE_FLOAT(FW_WING_SPAN, 3.0);

/**
 * Height (AGL) of the wings when the aircraft is on the ground.
 *
 * This is used to constrain a minimum altitude below which we keep wings level to avoid wing tip strike. It's safer
 * to give a slight margin here (> 0m)
 *
 * @unit m
 * @min 0.0
 * @decimal 1
 * @increment 1
 * @group FW Geometry
 */
PARAM_DEFINE_FLOAT(FW_WING_HEIGHT, 0.5);

/**
 * Landing flare time
 *
 * Multiplied by the descent rate to calculate a dynamic altitude at which
 * to trigger the flare.
 *
 * NOTE: max(FW_LND_FLALT, FW_LND_FL_TIME * descent rate) is taken as the flare altitude
 *
 * @unit s
 * @min 0.0
 * @max 5.0
 * @decimal 1
 * @increment 0.1
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_FL_TIME, 1.0f);

/**
 * Landing flare sink rate
 *
 * TECS will attempt to control the aircraft to this sink rate via pitch angle (throttle killed during flare)
 *
 * @unit m/s
 * @min 0.0
 * @max 1.0
 * @decimal 1
 * @increment 0.1
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_FL_SINK, 0.25f);

/**
 * Maximum lateral position offset for the touchdown point
 *
 * @unit m
 * @min 0.0
 * @max 10.0
 * @decimal 1
 * @increment 1
 * @group FW Auto Landing
 */
PARAM_DEFINE_FLOAT(FW_LND_TD_OFF, 3.0);

/**
 * Landing touchdown nudging option.
 *
 * Approach angle nudging: shifts the touchdown point laterally while keeping the approach entrance point constant
 * Approach path nudging: shifts the touchdown point laterally along with the entire approach path
 *
 * This is useful for manually adjusting the landing point in real time when map or GNSS errors cause an offset from the
 * desired landing vector. Nuding is done with yaw stick, constrained to FW_LND_TD_OFF (in meters) and the direction is
 * relative to the vehicle heading (stick deflection to the right = land point moves to the right as seen by the vehicle).
 *
 * @min 0
 * @max 2
 * @value 0 Disable nudging
 * @value 1 Nudge approach angle
 * @value 2 Nudge approach path
 * @group FW Auto Landing
 */
PARAM_DEFINE_INT32(FW_LND_NUDGE, 2);

/**
 * Bit mask to set the automatic landing abort conditions.
 *
 * Terrain estimation:
 * bit 0: Abort if terrain is not found
 * bit 1: Abort if terrain times out (after a first successful measurement)
 *
 * The last estimate is always used as ground, whether the last valid measurement or the land waypoint, depending on the
 * selected abort criteria, until an abort condition is entered. If FW_LND_USETER == 0, these bits are ignored.
 *
 * TODO: Extend automatic abort conditions
 * e.g. glide slope tracking error (horizontal and vertical)
 *
 * @min 0
 * @max 3
 * @bit 0 Abort if terrain is not found
 * @bit 1 Abort if terrain times out (after a first successful measurement)
 * @group FW Auto Landing
 */
PARAM_DEFINE_INT32(FW_LND_ABORT, 3);

/**
 * Wind-based airspeed scaling factor
 *
 * Multiplying this factor with the current absolute wind estimate gives the airspeed offset
 * added to the minimum airspeed setpoint limit. This helps to make the
 * system more robust against disturbances (turbulence) in high wind.
 * Only applies to AUTO flight mode.
 *
 * airspeed_min_adjusted = FW_AIRSPD_MIN + FW_WIND_ARSP_SC * wind.length()
 *
 * @min 0
 * @decimal 2
 * @increment 0.01
 * @group FW TECS
 */
PARAM_DEFINE_FLOAT(FW_WIND_ARSP_SC, 0.f);
