/****************************************************************************
 *
 *   Copyright (c) 2015-2021 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#ifdef __PX4_NUTTX

#include <sys/ioctl.h>
#include <lib/mathlib/mathlib.h>
#include <parameters/param.h>


#include "drivers/drv_pwm_trigger.h"
#include "pwm.h"

CameraInterfacePWM::CameraInterfacePWM():
	CameraInterface()
{
	param_get(param_find("TRIG_PWM_SHOOT"), &_pwm_camera_shoot);
	param_get(param_find("TRIG_PWM_NEUTRAL"), &_pwm_camera_neutral);
	get_pins();
	setup();
}

CameraInterfacePWM::~CameraInterfacePWM()
{
	// Deinitialise trigger channels
	up_pwm_trigger_deinit();
}

void CameraInterfacePWM::setup()
{
	// Precompute the bitmask for enabled channels
	uint32_t pin_bitmask = 0;

	for (unsigned i = 0; i < arraySize(_pins); i++) {
		if (_pins[i] >= 0) {
			pin_bitmask |= (1 << _pins[i]);
		}
	}

	// Initialize and arm channels
	int ret = up_pwm_trigger_init(pin_bitmask);

	if (ret < 0) {
		PX4_ERR("up_pwm_trigger_init failed (%i)", ret);
		pin_bitmask = 0;

	} else {
		pin_bitmask = ret;
	}

	// Clear pins that could not be initialized
	for (unsigned i = 0; i < arraySize(_pins); i++) {
		if (_pins[i] >= 0 && ((1 << _pins[i]) & pin_bitmask) == 0) {
			_pins[i] = -1;
		}
	}

	// Set neutral pulsewidths
	for (unsigned i = 0; i < arraySize(_pins); i++) {
		if (_pins[i] >= 0) {
			up_pwm_trigger_set(_pins[i], math::constrain(_pwm_camera_neutral, (int32_t) 0, (int32_t) 2000));
		}
	}

}

void CameraInterfacePWM::trigger(bool trigger_on_true)
{
	for (unsigned i = 0; i < arraySize(_pins); i++) {
		if (_pins[i] >= 0) {
			// Set all valid pins to shoot or neutral levels
			up_pwm_trigger_set(_pins[i], math::constrain(trigger_on_true ? _pwm_camera_shoot : _pwm_camera_neutral, (int32_t) 0,
					   (int32_t) 2000));
		}
	}
}

void CameraInterfacePWM::info()
{
	PX4_INFO_RAW("PWM trigger mode, pins enabled: ");

	for (unsigned i = 0; i < arraySize(_pins); ++i) {
		if (_pins[i] < 0) {
			continue;
		}

		PX4_INFO_RAW("[%d]", _pins[i] + 1);
	}

	PX4_INFO_RAW("\n");
}

#endif /* ifdef __PX4_NUTTX */
