from typing import Optional, Any

import requests
from .exceptions import AbstractionFailure, CelsiusNetworkHTTPError
from .utils import get_key, filter_transactions


class CelsiusNetwork:
    def __init__(self,
                 partner_token: str,
                 api_key: str,
                 enviroment: str = 'production',
                 silent: bool = False):
        """Initializes pyCelsiusNetwork

        Args:
            partner_token (str): A partner token provided by Celsius Network
            api_key (str): An API Key generated by the user on the app
            enviroment (str): Optional. Can be either 'production' or 'staging'. Changes API calls' base URL to match provided enviroment. Defaults to 'production'
            silent (bool): Global. If True silently returns None instead of raising custom Exceptions. Can be overriden on a per-function basis.
        """
        self._token = partner_token
        self._key = api_key

        if enviroment.lower() == 'production':
            self._base_url = "https://wallet-api.celsius.network"
        elif enviroment.lower() == 'staging':
            self._base_url = "https://wallet-api.staging.celsius.network"
        else:
            self._base_url = "https://wallet-api.celsius.network"

        self.headers = {
            'X-Cel-Partner-Token': self._token,
            'X-Cel-Api-Key': self._key
        }

        self.silent = silent

    def get_interest_rate(self,
                          coin: str = None,
                          raw: bool = False,
                          silent: bool = None):
        """Fetch interest rates

        Args:
            coin (str): Optional. A 3-letter code representing a cryptocoin
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            A dict with interest rates for each coin
            i.e. {'ETH': '0.0445', 'BTC': '0.0441'}

            If coin is given, will return a float with the interest rate for that coin

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        silent = silent if silent is not None else self.silent
        coin = coin.upper() if coin else None

        url = f"{self._base_url}" \
              "/util" \
              "/interest" \
              "/rates"

        response = requests.request("GET", url)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        else:
            rates = get_key('interestRates', json=json, silent=silent)
            rates_list = [{'coin': x['coin'], 'rate': x['rate']} for x in rates]
            rates_dict = {item.pop("coin"): item['rate'] for item in rates_list}
            if coin:
                return float(rates_dict[coin])
            else:
                return rates_dict

    def get_wallet_balance(self,
                           raw: bool = False,
                           silent: bool = None):
        """Fetch account balance

        Args:
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            A dict with a balance for each coin, even empty ones.

            i.e. {'eth': '0', 'btc': '0.00315111', 'dash': '0'}

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        silent = silent if silent is not None else self.silent

        url = f"{self._base_url}" \
              "/wallet" \
              "/balance"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()
        if raw:
            return json
        else:
            return get_key('balance', json=json, silent=silent)

    def get_coin_balance(self,
                         coin: str,
                         return_type: str = 'both',
                         raw: bool = False,
                         silent: bool = None):
        """Fetch account balance for specific coin

        Args:
            coin (str): A 3-letter code representing a cryptocoin
            return_type (str): Specify what you want to get. Can be: 'in_coin' for amount in coin, 'in_usd', for amount in usd and 'both', for a dict containing both values
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            Either a number with balance for specified coin in usd or the coin itself.
            Or a dict with both values.

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        coin = coin.upper()
        silent = silent if silent is not None else self.silent
        return_type = return_type.lower()

        url = f"{self._base_url}" \
              f"/wallet" \
              f"/{coin}" \
              f"/balance"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        else:
            if return_type == 'in_coin':
                in_coin = get_key('amount', json=json, silent=silent)
                return in_coin
            elif return_type == 'in_usd':
                in_usd = get_key('amount_in_usd', json=json, silent=silent)
                return in_usd
            elif return_type == 'both':
                in_coin = get_key('amount', json=json, silent=silent)
                in_usd = get_key('amount_in_usd', json=json, silent=silent)
                return {'in_coin': in_coin,
                        'in_usd': in_usd}

    def get_transactions(self,
                         raw: bool = False,
                         depaginate: bool = True,
                         reverse: bool = False,
                         silent: bool = None,
                         **kwargs):
        """Fetch all transactions on a account

        Args:
            depaginate (bool): Will automatically fetch all results in the next pages of the response. Defaults to True
            reverse (bool): Will reverse the results. From newest first to oldest first. Defaults to False
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Keyword Args:
            page (int): The page you want to fetch or start depagination from. Defaults to 1.
            per_page (int): The amount of results you want to see in a page. Only works if depaginate is False or raw is True. Defaults to 100

            dt_from (str/datetime): Optional. Inclusive. ISO compliant date string or datetime object. Return results after or equal to this date.
            dt_to (str/datetime): Optional. Inclusive. ISO compliant date string or datetime object. Only return results before or equal to this date.
            amount_bigger_than (float/int): Optional. Inclusive. Only return results with amounts bigger or equal to this
            amount_lower_than (float/int): Optional. Inclusive. Only return results with amounts lower or equal to this
            state (str): Optional. Only return results with a 'state' value equals to this
            nature (str): Optional. Only return results with a 'nature' value equals to this

        Returns:
            A list of dicts containing transaction information

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        page = kwargs.get('page') or 1
        per_page = kwargs.get('per_page') or 100
        silent = silent if silent is not None else self.silent

        # Filter options
        dt_from = kwargs.get('dt_from')
        dt_to = kwargs.get('dt_to')
        amount_bigger_than = kwargs.get('amount_bigger_than')
        amount_lower_than = kwargs.get('amount_lower_than')
        state = kwargs.get('state')
        nature = kwargs.get('nature')

        url = f"{self._base_url}" \
              f"/wallet" \
              f"/transactions?page={page}&per_page={per_page}"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        elif depaginate:
            # Depaginate results and return then as one list
            result = []
            try:
                result += json['record']

                pagination = json['pagination']
                if pagination['pages'] > page:
                    for next_page in range(
                            pagination['current'] + 1, pagination['pages'] + 1):
                        url = f"{self._base_url}" \
                              f"/wallet" \
                              f"/transactions?page={next_page}&per_page={per_page}"
                        response = requests.request("GET", url,
                                                    headers=self.headers)

                        json = response.json()
                        result += json['record']
            except KeyError:
                if silent:
                    return None
                else:
                    raise AbstractionFailure(json=json)

            if reverse:
                result.reverse()

            return filter_transactions(result,
                                       dt_from,
                                       dt_to,
                                       amount_bigger_than,
                                       amount_lower_than,
                                       state,
                                       nature)

        else:
            return filter_transactions(get_key(
                                        'record', json=json, silent=silent),
                                       dt_from,
                                       dt_to,
                                       amount_bigger_than,
                                       amount_lower_than,
                                       state,
                                       nature)

    def get_transactions_for_coin(self,
                                  coin: str,
                                  raw: bool = False,
                                  depaginate: bool = True,
                                  reverse: bool = False,
                                  silent: bool = None,
                                  **kwargs):
        """Fetch all transactions for a specific coin

        Args:
            coin (str): A 3-letter code representing a cryptocoin
            depaginate (bool): Will automatically fetch all results in the next pages of the response. Defaults to True
            reverse (bool): Will reverse the results. From newest first to oldest first. Defaults to False
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Keyword Args:
            page (int): The page you want to fetch or start depagination from. Defaults to 1.
            per_page (int): The amount of results you want to see in a page. Only works if depaginate is False or raw is True. Defaults to 100

            dt_from (str/datetime): Optional. Inclusive. ISO compliant date string or datetime object. Return results after or equal to this date.
            dt_to (str/datetime): Optional. Inclusive. ISO compliant date string or datetime object. Only return results before or equal to this date.
            amount_bigger_than (float/int): Optional. Inclusive. Only return results with amounts bigger or equal to this
            amount_lower_than (float/int): Optional. Inclusive. Only return results with amounts lower or equal to this
            state (str): Optional. Only return results with a 'state' value equals to this
            nature (str): Optional. Only return results with a 'nature' value equals to this

        Returns:
            A list of dicts containing transaction information

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        coin = coin.upper()
        page = kwargs.get('page') or 1
        per_page = kwargs.get('per_page') or 100
        silent = silent if silent is not None else self.silent

        # Filter options
        dt_from = kwargs.get('dt_from')
        dt_to = kwargs.get('dt_to')
        amount_bigger_than = kwargs.get('amount_bigger_than')
        amount_lower_than = kwargs.get('amount_lower_than')
        state = kwargs.get('state')
        nature = kwargs.get('nature')

        url = f"{self._base_url}" \
              f"/wallet" \
              f"/{coin}" \
              f"/transactions?page={page}&per_page={per_page}"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()
        if raw:
            return json
        elif depaginate:
            # Depaginate results and return then as one list
            result = []
            try:
                result += json['record']

                pagination = json['pagination']
                if pagination['pages'] > page:
                    for next_page in range(
                            pagination['current'] + 1, pagination['pages'] + 1):
                        url = f"{self._base_url}" \
                              f"/wallet" \
                              f"/{coin}" \
                              f"/transactions?page={next_page}&per_page=" \
                              f"{per_page}"
                        response = requests.request("GET", url,
                                                    headers=self.headers)

                        json = response.json()
                        result += json['record']
            except KeyError:
                if silent:
                    return None
                else:
                    raise AbstractionFailure(json=json)

            if reverse:
                result.reverse()

            return filter_transactions(result,
                                       dt_from,
                                       dt_to,
                                       amount_bigger_than,
                                       amount_lower_than,
                                       state,
                                       nature)

        else:
            return filter_transactions(get_key(
                                        'record', json=json, silent=silent),
                                       dt_from,
                                       dt_to,
                                       amount_bigger_than,
                                       amount_lower_than,
                                       state,
                                       nature)

    def get_deposit_adress_for_coin(self,
                                    coin: str,
                                    raw: bool = False,
                                    silent: bool = None):
        """Fetch the deposit address for a specific coin

        Args:
            coin (str): A 3-letter code representing a cryptocoin
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            A string representing the deposit address for adding the specified coin funds to Celsius Wallet

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        coin = coin.upper()
        silent = silent if silent is not None else self.silent

        url = f"{self._base_url}" \
              "/wallet" \
              f"/{coin}" \
              "/deposit"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        else:
            return get_key('address', json=json, silent=silent)

    def get_interest_summary(self,
                             coin: str = None,
                             raw: bool = False,
                             silent: bool = None):
        """Fetch a summary of all interest gained on Celsius Network by coin

        Args:
            coin (str): Optional. A 3-letter code representing a cryptocoin
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            A dict of dicts with all interest gained divided by coin.
            Includes 0 interest.

            i.e. {'BTC': {'amount': '0.00002348', 'amount_usd': 0.27939308701579496, 'amount_cel': 0},
                  'ETH': {'amount': 0, 'amount_usd': 0, 'amount_cel': 0, 'coin': 'ETH'}}

            If a coin argumenth is given, only the dictionary for that coin is returned.

            i.e. >> get_interest_summary('EHT')
                {'amount': 0, 'amount_usd': 0, 'amount_cel': 0, 'coin': 'ETH'}

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        url = f"{self._base_url}" \
              "/wallet" \
              f"/interest"

        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        elif coin:
            return get_key('interest', coin, json=json, silent=silent)
        else:
            return get_key('interest', json=json, silent=silent)

    def get_kyc_status(self,
                       raw: bool = False,
                       silent: bool = None):
        """Fetch KYC status for API Key owner (A.K.A. User)

        Args:
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            An upper case string informing the status.
            Can be:
                PENDING | Waiting on user to provide documents for verification
                COMPLETED | User has provided documents and is waiting to be verified
                PASSED | User was successfully verified
                REJECTED | User has failed verification

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        url = f"{self._base_url}" \
              "/kyc"
        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        else:
            return get_key('status', json=json, silent=silent)

    def get_supported_coins(self,
                            raw: bool = False,
                            silent: bool = None):
        """Fetch a list of coins supported by Celsius Network

        Args:
            raw (bool): If True returns the raw JSON response given by the server
            silent (bool): If True silently returns None instead of raising custom Exceptions

        Returns:
            A list cointaing 3 digit codes for all cryptocoins supported by Celsius Network

            i.e. ['ETH', 'BTC', 'USDC']

        Raises:
            CelsiusNetworkHTTPError
            AbstractionFailure
        """

        url = f"{self._base_url}" \
              "/util" \
              "/supported_currencies"
        response = requests.request("GET", url, headers=self.headers)

        if silent and not response.ok:
            return None
        elif not silent and not response.ok:
            raise CelsiusNetworkHTTPError(response)

        json = response.json()

        if raw:
            return json
        else:
            return get_key('currencies', json=json, silent=silent)
