from __future__ import annotations

import json
import time
from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, List, Tuple


@dataclass
class AppConfig:
    write_log: bool = False
    logs_path: Path = Path("logs")
    start_pages: int = 6
    end_pages: int = 3
    llm_certainty_threshold: float = 0.7
    request_timeout: int = 300
    min_total_pages: int = 20
    min_keywords_hit: int = 3


@dataclass
class APIConfig:
    name: str
    api_base: str
    api_key: str
    model: str
    activity: bool = False
    blocked_until: float = field(default=0.0)

    @property
    def headers(self) -> Dict[str, str]:
        return {
            "Authorization": f"Bearer {self.api_key}",
            "Content-Type": "application/json",
        }

    def is_available(self) -> bool:
        return time.time() >= self.blocked_until

    def block(self, cooldown_seconds: float) -> None:
        self.blocked_until = time.time() + cooldown_seconds


class ConfigLoadError(RuntimeError):
    pass


def load_configs(config_path: Path) -> Tuple[AppConfig, List[APIConfig]]:
    if not config_path.exists():
        raise ConfigLoadError(f"Config file not found: {config_path}")
    with config_path.open("r", encoding="utf-8") as fp:
        data = json.load(fp)

    app_config_data = data.get("options", {})
    app_config = AppConfig(
        write_log=bool(app_config_data.get("write_log", False)),
        logs_path=Path(app_config_data.get("logs", "logs")),
        start_pages=int(app_config_data.get("start_pages", 6)),
        end_pages=int(app_config_data.get("end_pages", 3)),
        llm_certainty_threshold=float(app_config_data.get("llm_certainty_threshold", 0.7)),
        request_timeout=int(app_config_data.get("request-timeout", 300)),
        min_total_pages=int(app_config_data.get("min_total_pages", 20)),
        min_keywords_hit=int(app_config_data.get("min_keywords_hit", 3)),
    )

    configs_data = data.get("configs")
    if not isinstance(configs_data, list):
        raise ConfigLoadError("Invalid config file: missing 'configs' list")
    configs: List[APIConfig] = []
    for entry in configs_data:
        try:
            configs.append(
                APIConfig(
                    name=entry["name"],
                    api_base=entry["api_base"].rstrip("/"),
                    api_key=entry["api_key"],
                    model=entry["model"],
                    activity=bool(entry.get("activity", False)),
                )
            )
        except KeyError as exc:  # pragma: no cover - configuration guard
            raise ConfigLoadError(f"Missing key {exc} in config entry: {entry}") from exc
    if not configs:
        raise ConfigLoadError("No API configs provided")
    return app_config, configs
