"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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