"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : getOrCreateVpc(this),
            this.service = [];
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            if (!t.internalOnly) {
                this.hasExternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
            vpc: this.vpc,
            internetFacing: false,
        });
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            // not internalOnly
            if (!t.internalOnly) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.listenerPort}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                protocol: elbv2.ApplicationProtocol.HTTP,
                vpc: this.vpc,
            });
            // listener for the internal ALB
            new elbv2.ApplicationListener(this, `IntAlbListener${t.listenerPort}`, {
                loadBalancer: this.internalAlb,
                open: true,
                port: t.listenerPort,
                protocol: elbv2.ApplicationProtocol.HTTP,
                defaultTargetGroups: [inttg],
            });
            // extra scaling policy
            scaling.scaleOnRequestCount('RequestScaling2', {
                requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                targetGroup: inttg,
            });
            inttg.addTarget(svc);
        });
        // Route53
        const zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        const externalAlbRecordName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.externalAlbRecordName) !== null && _f !== void 0 ? _f : 'external';
        const internalAlbRecordName = (_h = (_g = props.route53Ops) === null || _g === void 0 ? void 0 : _g.internalAlbRecordName) !== null && _h !== void 0 ? _h : 'internal';
        const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
            zoneName,
            vpc: this.vpc,
        });
        new route53.ARecord(this, 'InternalAlbAlias', {
            zone,
            recordName: internalAlbRecordName,
            target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
        });
        if (this.externalAlb) {
            new route53.ARecord(this, 'ExternalAlbAlias', {
                zone,
                recordName: externalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
            });
        }
        if (this.externalAlb) {
            new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
        }
        new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
        new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.0.11" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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