import { Capture } from './capture';
/**
 * Represents a matcher that can perform special data matching capabilities between a given pattern and a target.
 *
 * @stability stable
 */
export declare abstract class Matcher {
    /**
     * Check whether the provided object is a subtype of the `IMatcher`.
     *
     * @stability stable
     */
    static isMatcher(x: any): x is Matcher;
    /**
     * A name for the matcher.
     *
     * This is collected as part of the result and may be presented to the user.
     *
     * @stability stable
     */
    abstract readonly name: string;
    /**
     * Test whether a target matches the provided pattern.
     *
     * Every Matcher must implement this method.
     * This method will be invoked by the assertions framework. Do not call this method directly.
     *
     * @param actual the target to match.
     * @returns the list of match failures. An empty array denotes a successful match.
     * @stability stable
     */
    abstract test(actual: any): MatchResult;
}
/**
 * Match failure details.
 *
 * @stability stable
 */
export interface MatchFailure {
    /**
     * The matcher that had the failure.
     *
     * @stability stable
     */
    readonly matcher: Matcher;
    /**
     * The relative path in the target where the failure occurred.
     *
     * If the failure occurred at root of the match tree, set the path to an empty list.
     * If it occurs in the 5th index of an array nested within the 'foo' key of an object,
     * set the path as `['/foo', '[5]']`.
     *
     * @stability stable
     */
    readonly path: string[];
    /**
     * Failure message.
     *
     * @stability stable
     */
    readonly message: string;
}
/**
 * Information about a value captured during match.
 *
 * @stability stable
 */
export interface MatchCapture {
    /**
     * The instance of Capture class to which this capture is associated with.
     *
     * @stability stable
     */
    readonly capture: Capture;
    /**
     * The value that was captured.
     *
     * @stability stable
     */
    readonly value: any;
}
/**
 * The result of `Match.test()`.
 *
 * @stability stable
 */
export declare class MatchResult {
    /**
     * The target for which this result was generated.
     *
     * @stability stable
     */
    readonly target: any;
    private readonly failures;
    private readonly captures;
    private finalized;
    /**
     * @stability stable
     */
    constructor(target: any);
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use recordFailure()
     */
    push(matcher: Matcher, path: string[], message: string): this;
    /**
     * Record a new failure into this result at a specific path.
     *
     * @stability stable
     */
    recordFailure(failure: MatchFailure): this;
    /**
     * Does the result contain any failures.
     *
     * If not, the result is a success
     *
     * @stability stable
     */
    hasFailed(): boolean;
    /**
     * The number of failures.
     *
     * @stability stable
     */
    get failCount(): number;
    /**
     * Compose the results of a previous match as a subtree.
     *
     * @param id the id of the parent tree.
     * @stability stable
     */
    compose(id: string, inner: MatchResult): this;
    /**
     * Prepare the result to be analyzed.
     *
     * This API *must* be called prior to analyzing these results.
     *
     * @stability stable
     */
    finished(): this;
    /**
     * Get the list of failures as human readable strings.
     *
     * @stability stable
     */
    toHumanStrings(): string[];
    /**
     * Record a capture against in this match result.
     *
     * @stability stable
     */
    recordCapture(options: MatchCapture): void;
}
