import { Construct } from '@aws-cdk/core';
import { IRepository } from '@aws-cdk/aws-codecommit';
import { IPipeline } from '@aws-cdk/aws-codepipeline';
import { IProject } from '@aws-cdk/aws-codebuild';
import { IServerApplication, ILambdaApplication, IEcsApplication } from '@aws-cdk/aws-codedeploy';
import { INotificationTarget } from './notification-targets';
export interface INotificationRule {
    readonly notificationRuleArn: string;
}
export interface CommonNotificationRuleProps {
    /**
     * The name for the notification rule. Notification rule names
     * must be unique in your AWS account.
     */
    readonly name: string;
    /**
     * SNS topics or AWS Chatbot clients to associate with the notification rule.
     */
    readonly targets?: INotificationTarget[];
    /**
     * The level of detail to include in the notifications for this
     * resource. BASIC will include only the contents of the event
     * as it would appear in AWS CloudWatch. FULL will include any
     * supplemental information provided by AWS CodeStar Notifications
     * and/or the service for the resource for which the notification
     * is created.
     *
     * @default FULL
     */
    readonly detailType?: DetailType;
    /**
     * The status of the notification rule. The default value is ENABLED.
     * If the status is set to DISABLED, notifications aren't sent for
     * the notification rule.
     *
     * @default ENABLED
     */
    readonly status?: Status;
}
export interface NotificationRuleProps extends CommonNotificationRuleProps {
    /**
     * A list of events associated with this notification rule.
     */
    readonly events: Events[];
    /**
     * The Amazon Resource Name (ARN) of the resource to associate with
     * the notification rule. Supported resources include pipelines in
     * AWS CodePipeline, repositories in AWS CodeCommit, and build
     * projects in AWS CodeBuild.
     */
    readonly resource: string;
}
export declare class NotificationRule extends Construct implements INotificationRule {
    readonly notificationRuleArn: string;
    private readonly targets;
    constructor(scope: Construct, id: string, props: NotificationRuleProps);
    addTarget(target: INotificationTarget): void;
    protected validate(): string[];
}
export interface RepositoryNotificationRuleProps extends CommonNotificationRuleProps {
    readonly repository: IRepository;
    readonly events: RepositoryEvent[];
}
export declare class RepositoryNotificationRule extends NotificationRule {
    constructor(scope: Construct, id: string, props: RepositoryNotificationRuleProps);
}
export interface PipelineNotificationRuleProps extends CommonNotificationRuleProps {
    readonly pipeline: IPipeline;
    readonly events: PipelineEvent[];
}
export declare class PipelineNotificationRule extends NotificationRule {
    constructor(scope: Construct, id: string, props: PipelineNotificationRuleProps);
}
export interface ProjectNotificationRuleProps extends CommonNotificationRuleProps {
    readonly project: IProject;
    readonly events: ProjectEvent[];
}
export declare class ProjectNotificationRule extends NotificationRule {
    constructor(scope: Construct, id: string, props: ProjectNotificationRuleProps);
}
export interface ApplicationNotificationRuleProps extends CommonNotificationRuleProps {
    readonly application: IServerApplication | ILambdaApplication | IEcsApplication;
    readonly events: ApplicationEvent[];
}
export declare class ApplicationNotificationRule extends NotificationRule {
    constructor(scope: Construct, id: string, props: ApplicationNotificationRuleProps);
}
export declare enum RepositoryEvent {
    COMMENTS_ON_COMMITS = "codecommit-repository-comments-on-commits",
    COMMENTS_ON_PULL_REQUEST = "codecommit-repository-comments-on-pull-requests",
    APPROVAL_STATUS_CHANGED = "codecommit-repository-approvals-status-changed",
    APPROVAL_RULE_OVERRIDE = "codecommit-repository-approvals-rule-override",
    PULL_REQUEST_CREATED = "codecommit-repository-pull-request-created",
    PULL_REQUEST_SOURCE_UPDATED = "codecommit-repository-pull-request-source-updated",
    PULL_REQUEST_STATUS_CHANGED = "codecommit-repository-pull-request-status-changed",
    PULL_REQUEST_MERGED = "codecommit-repository-pull-request-merged",
    BRANCHES_AND_TAGS_CREATED = "codecommit-repository-branches-and-tags-created",
    BRANCHES_AND_TAGS_DELETED = "codecommit-repository-branches-and-tags-deleted",
    BRANCHES_AND_TAGS_UPDATED = "codecommit-repository-branches-and-tags-updated"
}
export declare enum ProjectEvent {
    BUILD_STATE_FAILED = "codebuild-project-build-state-failed",
    BUILD_STATE_SUCCEEDED = "codebuild-project-build-state-succeeded",
    BUILD_STATE_IN_PROGRESS = "codebuild-project-build-state-in-progress",
    BUILD_STATE_STOPPED = "codebuild-project-build-state-stopped",
    BUILD_PHASE_FAILRE = "codebuild-project-build-phase-failure",
    BUILD_PHASE_SUCCESS = "codebuild-project-build-phase-success"
}
export declare enum ApplicationEvent {
    DEPLOYMENT_FAILED = "codedeploy-application-deployment-failed",
    DEPLOYMENT_SUCCEEDED = "codedeploy-application-deployment-succeeded",
    DEPLOYMENT_STARTED = "codedeploy-application-deployment-started"
}
export declare enum PipelineEvent {
    ACTION_EXECUTION_SUCCEEDED = "codepipeline-pipeline-action-execution-succeeded",
    ACTION_EXECUTION_FAILED = "codepipeline-pipeline-action-execution-failed",
    ACTION_EXECUTION_CANCELED = "codepipeline-pipeline-action-execution-canceled",
    ACTION_EXECUTION_STARTED = "codepipeline-pipeline-action-execution-started",
    STAGE_EXECUTION_STARTED = "codepipeline-pipeline-stage-execution-started",
    STAGE_EXECUTION_SUCCEEDED = "codepipeline-pipeline-stage-execution-succeeded",
    STAGE_EXECUTION_RESUMED = "codepipeline-pipeline-stage-execution-resumed",
    STAGE_EXECUTION_CANCELED = "codepipeline-pipeline-stage-execution-canceled",
    STAGE_EXECUTION_FAILED = "codepipeline-pipeline-stage-execution-failed",
    PIPELINE_EXECUTION_FAILED = "codepipeline-pipeline-pipeline-execution-failed",
    PIPELINE_EXECUTION_CANCELED = "codepipeline-pipeline-pipeline-execution-canceled",
    PIPELINE_EXECUTION_STARTED = "codepipeline-pipeline-pipeline-execution-started",
    PIPELINE_EXECUTION_RESUMED = "codepipeline-pipeline-pipeline-execution-resumed",
    PIPELINE_EXECUTION_SUCCEEDED = "codepipeline-pipeline-pipeline-execution-succeeded",
    PIPELINE_EXECUTION_SUPERSEDED = "codepipeline-pipeline-pipeline-execution-superseded",
    MANUAL_APPROVAL_FAILED = "codepipeline-pipeline-manual-approval-failed",
    MANUAL_APPROVAL_NEEDED = "codepipeline-pipeline-manual-approval-needed",
    MANUAL_APPROVAL_SUCCEEDED = "codepipeline-pipeline-manual-approval-succeeded"
}
declare type Events = RepositoryEvent | PipelineEvent | ProjectEvent | ApplicationEvent;
export declare enum DetailType {
    FULL = "FULL",
    BASIC = "BASIC"
}
export declare enum Status {
    DISABLED = "DISABLED",
    ENABLED = "ENABLED"
}
export {};
