# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['babelgrid']

package_data = \
{'': ['*']}

install_requires = \
['h3==3.6.3', 'pygeotile', 'pyproj==2.6.1', 's2sphere==0.2.5', 'shapely==1.7.0']

setup_kwargs = {
    'name': 'babelgrid',
    'version': '0.1.3',
    'description': 'A common python API to work with different established grid systems.',
    'long_description': '# BabelGrid\n\n**BabelGrid is a common python API to work with different established grid systems.**\n\nCurrently, it supports [H3](https://h3geo.org/), [S2](https://s2geometry.io/) and [Bing](https://docs.microsoft.com/en-us/bingmaps/articles/bing-maps-tile-system) grid systems. \nBabelGrid does not have the intention to replace any of the existing APIs, but\nto create a common framework that geo-folks can use to easily switch between grids. Also, \nit generates methods around the tiles that ease the data analysis pipeline with seamlessly integration\nwith well knonw libraries such as Shapely and GeoPandas.\n\n## Start Using It\n\nInstall with\n```bash\npip install babelgrid\n```\n\n- Get a h3 tile with an area of 1km2 in São Paulo, Brasil.\n```python\n>>> from babelgrid import Babel\n\n>>> tile = Babel(\'h3\').geo_to_tile(lat=-23, lon=-43, area_km=1)\n>>> tile\nTile: grid_type "h3", resolution 8, tile_id 88a8a2b66dfffff\n```\n\n- Access the geojson, wkt and shapely descriptions of the tile:\n```python\n>>> tile.geometry.geojson\n{\'type\': \'Polygon\',\n \'coordinates\': (((-42.99741709893686, -23.004282833594505),\n   (-42.9932470321478, -23.00127887552568),\n   (-42.994161748920796, -22.996608473771282),\n   (-42.99924646130203, -22.994942061847414),\n   (-43.00341650043048, -22.997946087213307),\n   (-43.002501854850166, -23.002616457194414),\n   (-42.99741709893686, -23.004282833594505)),)}\n>>> tile.geometry.wkt\n\'POLYGON ((-42.9974170989368574 -23.0042828335945053, -42.9932470321477993 -23.0012788755256814, -42.9941617489207957 -22.9966084737712819, -42.9992464613020289 -22.9949420618474143, -43.0034165004304825 -22.9979460872133075, -43.0025018548501663 -23.0026164571944136, -42.9974170989368574 -23.0042828335945053))\'\n>>> tile.geometry.shapely\n```\n![][shapely]\n\n-  Fill a geometry with s2 tiles of resolution 10\n```python\n>>> tiles = Babel(\'s2\').polyfill(geometry, resolution=10)\n>>> tiles\n[Tile: grid_type "s2", resolution 10, tile_id 94d28d,... ,Tile: grid_type "s2", resolution 10, tile_id 94d28f]\n```\n- Load a geopandas dataframe with the selected tiles\n```python\n>>> import geopandas as gpd\n>>> gpd.GeoDataFrame([t.to_dict() for t in tiles], geometry=\'shapely\')\n```\n\n\n## Quick Documentation\n\n### Babel\n\nYou have to initialize the Babel object with any of the available grids. \n\n```python\n>>> Babel.available_grids()\n[\'s2\', \'h3\', \'bing\']\n\n>>> grid = Babel(\'s2\') # example\n```\n\n\n### geo_to_tile\n\nIt receives a coordinate pair (lat, lon) and either the native grid resolution or an area in km2.\nIf it receives an area, it automatically finds what is the resolution for that tile system and latitute \nthat best approximates the given area.\n\n```python\n>>> Babel(\'s2\').geo_to_tile(2, 3, resolution=10)\nTile: grid_type "s2", resolution 10, tile_id 100fb1\n\n>>> Babel(\'bing\').geo_to_tile(2, 3, area_km=0.1)\nTile: grid_type "bing", resolution 17, tile_id 12222230201200322\n\n>>> Babel(\'bing\').geo_to_tile(2, 3, area_km=0.1).area_km\n0.0934819087\n```\n\n### id_to_tile\n\nIt receives a tile id and converts it to a Tile Object.\n\n```python\n>>> Babel(\'s2\').id_to_tile(\'100fb1\')\nTile: grid_type "s2", resolution 10, tile_id 100fb1\n```\n\n### Polyfill\nOne of the most common uses to grid systems is to fill up a geometry. This function receives\na geometry and returns a list of Tile Objects.\n\n```python\n>>> tiles = Babel(\'s2\').polyfill(geometry, resolution=10)\n>>> tiles\n[Tile: grid_type "s2", resolution 10, tile_id 94d28d,... ,Tile: grid_type "s2", resolution 10, tile_id 94d28f]\n```\n\nYou can also pass a \'desired\' grid area using the parameter `grid_km`.\n\n```python\n>>> tiles = Babel(\'bing\').polyfill(geometry, area_km=10)\n>>> tiles\n[Tile: grid_type "bing", resolution 14, tile_id 21031113121331, ..., Tile: grid_type "bing", resolution 14, tile_id 21031113121333]\n```\n\nThe image below shows `polyfill` being applied for the same geometry for different grid types and sizes.\n\n![][polyfill]\n\n### The Tile Object\n\nThe Tile Object is a central piece of the package. This is the object that is returned by most of the methods implemented. It is good because it \nhas some handy features that speed-up the analysis process.\n\n- Easy access to wkt, geojson and shapely geometries\n\n```python\n>>> tile.geometry.wkt\n>>> tile.geometry.geojson\n>>> tile.geometry.shapely\n```\n\n- Child and parent transformation\n\n```python\n>>> tile.to_parent()\n>>> tile.to_children()\n```\n\n- Area in km2 already calculated\n\n```python\n>>> tile.area_km\n```\n\n- To dictonary export of all properties\n\n```python\n>>> tile.to_dict()\n```\n\n## Grid Systems\n\n\n\n|                        | H3                                     | S2                                                   | BING/QuadTree                                                                                     |\n|------------------------|----------------------------------------|------------------------------------------------------|---------------------------------------------------------------------------------------------------|\n| Tile Shape             | Hexagonal                              | Square                                               | Square                                                                                            |\n| Resolution Range       | 0 - 15                                 | 0 - 30                                               | 1 - 23 (infinite)                                                                                 |\n| API Reference          | [h3-py](https://github.com/uber/h3-py) | [s2sphere](https://github.com/sidewalklabs/s2sphere) | [pygeotile](https://github.com/geometalab/pyGeoTile)                                              |\n| Original Documentation | [H3](https://h3geo.org/)               | [S2 Geometry](https://s2geometry.io/)                | [Bing Maps Tile System](https://docs.microsoft.com/en-us/bingmaps/articles/bing-maps-tile-system) |\n\n:star: Kudos to all developer of H3, S2 and Bing/QuadTree systems.\n\n### Resolution Reference Table and Plot\n\n\nLookup table with grid resolutions at equator by area in km2. \nNote that the area is written in scientific notation (10^x) and x \nis the index of the table.\n\n| Area (10^x km2) | H3    | S2    | BING/QuadTree   |\n|-----------:|:------|:------|:----------------|\n|          9 | -     | -     | 1               |\n|          8 | -     | 0     | 2               |\n|          7 | -     | 1,2   | 3,4             |\n|          6 | 0,1   | 3,4   | 5,6             |\n|          5 | 2     | 5     | 7               |\n|          4 | 3     | 6,7   | 8,9             |\n|          3 | 4     | 8     | 10,11           |\n|          2 | 5     | 9,10  | 12              |\n|          1 | 6,7   | 11,12 | 13,14           |\n|          0 | 8     | 13    | 15,16           |\n|         -1 | 9     | 14,15 | 17              |\n|         -2 | 10    | 16,17 | 18,19           |\n|         -3 | 11    | 18    | 20,21           |\n|         -4 | 12,13 | 19,20 | 22              |\n|         -5 | 14    | 21,22 | 23              |\n|         -6 | 15    | 23    | -               |\n|         -7 | -     | 24,25 | -               |\n|         -8 | -     | 26,27 | -               |\n|         -9 | -     | 28    | -               |\n|        -10 | -     | 29,30 | -               |\n\n\n![][area-res]\n\n### Tile Area Distortion by Latitude\n\nDepending on how the tile system is built, the area of the tile varies given the latitude. \nFor inter-region comparissons, this behaviour can affect the analysis. \n\nThe figure below shows the tile area distortion by grid system. The distortion is defined as\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=D&space;=&space;A_l&space;/&space;A_e" target="_blank"><img src="https://latex.codecogs.com/gif.latex?D&space;=&space;A_l&space;/&space;A_e" title="D = A_l / A_e" /></a>\n\nwhere <a href="https://www.codecogs.com/eqnedit.php?latex=\\inline&space;A" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\inline&space;A" title="A" /></a> is the tile area and <a href="https://www.codecogs.com/eqnedit.php?latex=\\inline&space;l" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\inline&space;l" title="l" /></a> the area given a latitude and <a href="https://www.codecogs.com/eqnedit.php?latex=\\inline&space;e" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\inline&space;e" title="e" /></a> the equator area. The figure\nshows the mean distortion given all resolutions and the error bar is the standard deviation.\n\n![][area-distortion]\n\n[shapely]: https://github.com/EL-BID/BabelGrid/blob/master/imgs/Screen%20Shot%202020-07-20%20at%2019.47.58.png\n[polyfill]: https://github.com/EL-BID/BabelGrid/blob/master/imgs/polyfill.png\n[area-res]: https://github.com/EL-BID/BabelGrid/blob/master/imgs/gridtype-area-res.png?raw=true\n[area-distortion]: https://github.com/EL-BID/BabelGrid/blob/master/imgs/gridtype-distortion.png\n\n## Developing\n\n',
    'author': 'Joao Carabetta',
    'author_email': 'joao.carabetta@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/EL-BID/BabelGrid',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
