"""Headscale API endpoint information."""

__authors__ = ["Marek Pikuła <marek@serenitycode.dev>"]

from dataclasses import dataclass
from typing import Any, Dict, Generic, Literal, Optional, Type, TypeVar

from betterproto import Message

from .schema.headscale import v1 as schema

RequestT = TypeVar("RequestT", bound=Message)
ResponseT = TypeVar("ResponseT", bound=Message)


@dataclass
class Endpoint(Generic[RequestT, ResponseT]):
    """Endpoint information.

    Used to translate the protobuf route to API information.
    """

    request_schema: Type[RequestT]
    """Request message schema."""

    response_schema: Type[ResponseT]
    """Response message schema."""

    request_type: Literal["GET", "OPTIONS", "HEAD", "POST", "PUT", "PATCH", "DELETE"]
    """Type of request to be served to `requests.request()`."""

    api_url: str
    """API URL stub (e.g. "/api/v1/...")"""

    logger_start_message: str
    """Message to log (INFO) on request start.

    Can have formatter tags, which are replaced with request fields.
    """

    logger_success_message: Optional[str] = None
    """Optional message to log (INFO) on request success.

    Can have formatter tags, which are replaced with request and response fields. If set
    to None and `logger_start_message` is set it's tried to be autogenerated if a
    suitable word word replacement is found.
    """

    logger_fail_message: Optional[str] = None
    """Optional message to log (ERROR) on request failure.

    Can have formatter tags, which are replaced with request fields. If set to None and
    `logger_start_message` is set it's tried to be autogenerated if a suitable word word
    replacement is found.
    """

    def __post_init__(self):
        """Post-initialize dataclass.

        Tries to fill in missing logger messages from `logger_start_message`.
        """
        replacements = {
            "Adding": ("Added", "Failed to add"),
            "Creating": ("Created", "Failed to create"),
            "Deleting": ("Deleted", "Failed to delete"),
            "Disabling": ("Disabled", "Failed to disable"),
            "Enabling": ("Enabled", "Failed to enable"),
            "Expiring": ("Expired", "Failed to expire"),
            "Getting": ("Got", "Failed to get"),
            "Moving": ("Moved", "Failed to move"),
            "Registering": ("Registered", "Failed to register"),
            "Renaming": ("Renamed", "Failed to rename"),
            "Setting": ("Set", "Failed to set"),
        }
        reference = self.logger_start_message
        for old, new in replacements.items():
            if reference.startswith(old):
                self.logger_success_message = reference.replace(old, new[0])
                self.logger_fail_message = reference.replace(old, new[1])

    def check_logger_format(self):
        """Check logger format for wrong keys.

        Raises:
            KeyError: if unsupported key is detected in the message.
        """
        request_dict: Dict[str, Any] = self.request_schema().to_dict(  # type: ignore
            include_default_values=True
        )
        response_dict: Dict[str, Any] = self.response_schema().to_dict(  # type: ignore
            include_default_values=True
        )

        self.api_url.format_map(request_dict)
        self.logger_start_message.format_map(request_dict)

        if self.logger_success_message is not None:
            self.logger_success_message.format_map(dict(request_dict, **response_dict))

        if self.logger_fail_message is not None:
            self.logger_fail_message.format_map(request_dict)


ENDPOINTS = {
    # Users API.
    "/headscale.v1.HeadscaleService/GetUser": Endpoint(
        schema.GetUserRequest,
        schema.GetUserResponse,
        "GET",
        "/api/v1/user/{name}",
        'Getting user "{name}".',
    ),
    "/headscale.v1.HeadscaleService/CreateUser": Endpoint(
        schema.CreateUserRequest,
        schema.CreateApiKeyResponse,
        "POST",
        "/api/v1/user",
        'Creating user "{name}".',
    ),
    "/headscale.v1.HeadscaleService/RenameUser": Endpoint(
        schema.RenameUserRequest,
        schema.RegisterMachineResponse,
        "POST",
        "/api/v1/user/{oldName}/rename/{newName}",
        'Renaming user from "{oldName} to "{newName}".',
    ),
    "/headscale.v1.HeadscaleService/DeleteUser": Endpoint(
        schema.DeleteUserRequest,
        schema.DeleteDeviceResponse,
        "DELETE",
        "/api/v1/user/{name}",
        'Deleting a user "{name}".',
    ),
    "/headscale.v1.HeadscaleService/ListUsers": Endpoint(
        schema.ListUsersRequest,
        schema.ListUsersResponse,
        "GET",
        "/api/v1/user",
        "Getting all users.",
    ),
    # PreAuth keys API.
    "/headscale.v1.HeadscaleService/CreatePreAuthKey": Endpoint(
        schema.CreatePreAuthKeyRequest,
        schema.CreateApiKeyResponse,
        "POST",
        "/api/v1/preauthkey",
        'Adding PreAuth key for user "{user}".',
    ),
    "/headscale.v1.HeadscaleService/ExpirePreAuthKey": Endpoint(
        schema.ExpirePreAuthKeyRequest,
        schema.ExpirePreAuthKeyResponse,
        "POST",
        "/api/v1/preauthkey/expire",
        "Expiring PreAuth key for user {user}.",
    ),
    "/headscale.v1.HeadscaleService/ListPreAuthKeys": Endpoint(
        schema.ListPreAuthKeysRequest,
        schema.ListPreAuthKeysResponse,
        "GET",
        "/api/v1/preauthkey",
        'Getting PreAuth keys for user "{user}".',
    ),
    # Machines API.
    "/headscale.v1.HeadscaleService/DebugCreateMachine": Endpoint(
        schema.DebugCreateMachineRequest,
        schema.DebugCreateMachineResponse,
        "POST",
        "/api/v1/debug/machine",
        'Creating machine "{name}" for user "{user}".',
    ),
    "/headscale.v1.HeadscaleService/GetMachine": Endpoint(
        schema.GetMachineRequest,
        schema.GetMachineResponse,
        "GET",
        "/api/v1/machine/{machineId}",
        'Getting machine "{machineId}".',
    ),
    "/headscale.v1.HeadscaleService/SetTags": Endpoint(
        schema.SetTagsRequest,
        schema.SetTagsResponse,
        "POST",
        "/api/v1/machine/{machineId}/tags",
        'Setting tags for machine "{machineId}".',
    ),
    "/headscale.v1.HeadscaleService/RegisterMachine": Endpoint(
        schema.RegisterMachineRequest,
        schema.RegisterMachineResponse,
        "POST",
        "/api/v1/machine/register",
        'Registering machine for user "{user}".',
    ),
    "/headscale.v1.HeadscaleService/DeleteMachine": Endpoint(
        schema.DeleteMachineRequest,
        schema.DeleteDeviceResponse,
        "DELETE",
        "/api/v1/machine/{machineId}",
        'Deleting machine "{machineId}".',
    ),
    "/headscale.v1.HeadscaleService/ExpireMachine": Endpoint(
        schema.ExpireMachineRequest,
        schema.ExpireMachineResponse,
        "POST",
        "/api/v1/machine/{machineId}/expire",
        'Expiring machine "{machineId}".',
    ),
    "/headscale.v1.HeadscaleService/RenameMachine": Endpoint(
        schema.RenameMachineRequest,
        schema.RenameMachineResponse,
        "POST",
        "/api/v1/machine/{machineId}/rename/{newName}",
        'Renaming machine "{machineId}" to "{newName}".',
    ),
    "/headscale.v1.HeadscaleService/ListMachines": Endpoint(
        schema.ListMachinesRequest,
        schema.ListMachinesResponse,
        "GET",
        "/api/v1/machine",
        "Getting all machines.",
    ),
    "/headscale.v1.HeadscaleService/MoveMachine": Endpoint(
        schema.MoveMachineRequest,
        schema.MoveMachineResponse,
        "POST",
        "/api/v1/machine/{machineId}/user",
        'Moving machine "{machineId}" to user "{user}".',
    ),
    "/headscale.v1.HeadscaleService/GetRoutes": Endpoint(
        schema.GetRoutesRequest,
        schema.GetRoutesResponse,
        "GET",
        "/api/v1/routes",
        "Getting all routes.",
    ),
    "/headscale.v1.HeadscaleService/EnableRoute": Endpoint(
        schema.EnableRouteRequest,
        schema.EnableRouteResponse,
        "POST",
        "/api/v1/routes/{routeId}/enable",
        'Enabling route "{routeId}".',
    ),
    "/headscale.v1.HeadscaleService/DisableRoute": Endpoint(
        schema.DisableRouteRequest,
        schema.DisableRouteResponse,
        "POST",
        "/api/v1/routes/{routeId}/enable",
        'Disabling route "{routeId}".',
    ),
    "/headscale.v1.HeadscaleService/GetMachineRoutes": Endpoint(
        schema.GetMachineRoutesRequest,
        schema.GetMachineRoutesResponse,
        "GET",
        "/api/v1/machine/{machineId}/routes",
        'Getting routes for machine "{machineId}".',
    ),
    "/headscale.v1.HeadscaleService/DeleteRoute": Endpoint(
        schema.DeleteRouteRequest,
        schema.DeleteRouteResponse,
        "DELETE",
        "/api/v1/routes/{routeId}",
        'Deleting route "{routeId}".',
    ),
    # API key API.
    "/headscale.v1.HeadscaleService/CreateApiKey": Endpoint(
        schema.CreateApiKeyRequest,
        schema.CreateApiKeyResponse,
        "POST",
        "/api/v1/apikey",
        "Creating API key.",
    ),
    "/headscale.v1.HeadscaleService/ExpireApiKey": Endpoint(
        schema.ExpireApiKeyRequest,
        schema.ExpireApiKeyResponse,
        "POST",
        "/api/v1/apikey/expire",
        "Expiring API key.",
    ),
    "/headscale.v1.HeadscaleService/ListApiKeys": Endpoint(
        schema.ListApiKeysRequest,
        schema.ListApiKeysResponse,
        "GET",
        "/api/v1/apikey",
        "Getting all API keys.",
    ),
}
