"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(stack, 'runners', {});
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(stack, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(stack, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(stack, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   stack, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   stack,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props ?? {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        this.orchestrator = this.stateMachine();
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine() {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: this.deleteRunner(),
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${provider.label}`), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        return new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const providers = this.providers.map(provider => {
            return {
                type: provider.constructor.name,
                label: provider.label,
                vpcArn: provider.vpc && provider.vpc.vpcArn,
                securityGroup: provider.securityGroup && provider.securityGroup.securityGroupId,
                roleArn: provider.grantPrincipal.grantPrincipal.roleArn,
            };
        });
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                PROVIDERS: JSON.stringify(providers),
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                SETUP_FUNCTION_URL: this.setupUrl,
            },
            timeout: cdk.Duration.minutes(3),
        });
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${cdk.Stack.of(this).region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
            },
            timeout: cdk.Duration.minutes(3),
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        // this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.3.0" };
//# sourceMappingURL=data:application/json;base64,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