"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const utils_1 = require("../utils");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * GitHub Actions runner provider using Lambda to execute the actions.
 *
 * Creates a Docker-based function that gets executed for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class LambdaRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.label = props.label || 'lambda';
        this.vpc = props.vpc;
        this.securityGroup = props.securityGroup;
        const imageBuilder = props.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: LambdaRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = imageBuilder.bind();
        let architecture;
        if (image.os.is(common_1.Os.LINUX)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.X86_64;
            }
            if (image.architecture.is(common_1.Architecture.ARM64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.ARM_64;
            }
        }
        if (!architecture) {
            throw new Error(`Unable to find support Lambda architecture for ${image.os.name}/${image.architecture.name}`);
        }
        this.function = new aws_cdk_lib_1.aws_lambda.DockerImageFunction(this, 'Function', {
            description: `GitHub Actions runner for "${this.label}" label`,
            // CDK requires "sha256:" literal prefix -- https://github.com/aws/aws-cdk/blob/ba91ca45ad759ab5db6da17a62333e2bc11e1075/packages/%40aws-cdk/aws-ecr/lib/repository.ts#L184
            code: aws_cdk_lib_1.aws_lambda.DockerImageCode.fromEcr(image.imageRepository, { tagOrDigest: `sha256:${image.imageDigest}` }),
            architecture,
            vpc: this.vpc,
            securityGroups: this.securityGroup && [this.securityGroup],
            vpcSubnets: props.subnetSelection,
            timeout: props.timeout || cdk.Duration.minutes(15),
            memorySize: props.memorySize || 2048,
            ephemeralStorageSize: props.ephemeralStorageSize || cdk.Size.gibibytes(10),
            logRetention: props.logRetention || aws_logs_1.RetentionDays.ONE_MONTH,
        });
        this.grantPrincipal = this.function.grantPrincipal;
        this.addImageUpdater(image);
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.function.connections;
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, this.label, {
            lambdaFunction: this.function,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                token: parameters.runnerTokenPath,
                runnerName: parameters.runnerNamePath,
                label: this.label,
                githubDomain: parameters.githubDomainPath,
                owner: parameters.ownerPath,
                repo: parameters.repoPath,
            }),
        });
    }
    addImageUpdater(image) {
        // Lambda needs to be pointing to a specific image digest and not just a tag.
        // Whenever we update the tag to a new digest, we need to update the lambda.
        let stack = cdk.Stack.of(this);
        const updater = utils_1.BundledNodejsFunction.singleton(this, 'update-lambda', {
            description: 'Function that updates a GitHub Actions runner function with the latest image digest after the image has been rebuilt',
            timeout: cdk.Duration.seconds(30),
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['lambda:UpdateFunctionCode'],
                    resources: [this.function.functionArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['cloudformation:DescribeStacks'],
                    resources: [stack.formatArn({
                            service: 'cloudformation',
                            resource: 'stack',
                            resourceName: `${stack.stackName}/*`,
                        })],
                }),
            ],
        });
        let lambdaTarget = new aws_cdk_lib_1.aws_events_targets.LambdaFunction(updater, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                lambdaName: this.function.functionName,
                repositoryUri: image.imageRepository.repositoryUri,
                repositoryTag: image.imageTag,
                stackName: stack.stackName,
            }),
        });
        const rule = image.imageRepository.onEvent('Push rule', {
            description: 'Update GitHub Actions runner Lambda on ECR image push',
            eventPattern: {
                detailType: ['ECR Image Action'],
                detail: {
                    'action-type': ['PUSH'],
                    'repository-name': [image.imageRepository.repositoryName],
                    'image-tag': ['latest'],
                    'result': ['SUCCESS'],
                },
            },
            target: lambdaTarget,
        });
        // the event never triggers without this - not sure why
        rule.node.defaultChild.addDeletionOverride('Properties.EventPattern.resources');
    }
}
exports.LambdaRunner = LambdaRunner;
_a = JSII_RTTI_SYMBOL_1;
LambdaRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LambdaRunner", version: "0.3.0" };
/**
 * Path to Dockerfile for Linux x64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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