"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * An image builder that uses CodeBuild to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Each provider has its own requirements for what an image should do. That's why they each provide their own Dockerfile.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new CodeBuildImageBuilder(this, 'Builder', {
 *     dockerfilePath: FargateProvider.LINUX_X64_DOCKERFILE_PATH,
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.setBuildArg('EXTRA_PACKAGES', 'nginx xz-utils');
 * new FargateProvider(this, 'Fargate provider', {
 *     label: 'customized-fargate',
 *     imageBuilder: builder,
 * });
 * ```
 */
class CodeBuildImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.preBuild = [];
        this.postBuild = [];
        this.buildArgs = new Map();
        this.policyStatements = [];
        this.secondaryAssets = new Map();
        if (props.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // set platform
        this.architecture = props.architecture ?? common_1.Architecture.X86_64;
        this.os = props.os ?? common_1.Os.LINUX;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
        // upload Dockerfile to S3 as an asset
        this.dockerfile = new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Dockerfile', {
            path: props.dockerfilePath,
        });
        // choose build image
        this.buildImage = this.getBuildImage();
    }
    /**
     * Uploads a folder to the build server at a given folder name.
     *
     * @param sourcePath path to source directory
     * @param destName name of destination folder
     */
    addFiles(sourcePath, destName) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, destName, { path: sourcePath });
        this.secondaryAssets.set(destName, asset);
        this.preBuild.push(`ln -s "$CODEBUILD_SRC_DIR_${destName}" "${destName}"`);
    }
    /**
     * Adds a command that runs before `docker build`.
     *
     * @param command command to add
     */
    addPreBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.preBuild.push(command);
    }
    /**
     * Adds a command that runs after `docker build` and `docker push`.
     *
     * @param command command to add
     */
    addPostBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.postBuild.push(command);
    }
    /**
     * Adds a build argument for Docker. See the documentation for the Dockerfile you're using for a list of supported build arguments.
     *
     * @param name build argument name
     * @param value build argument value
     */
    setBuildArg(name, value) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.buildArgs.set(name, value);
    }
    /**
     * Add a policy statement to the builder to access resources required to the image build.
     *
     * @param statement IAM policy statement
     */
    addPolicyStatement(statement) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.policyStatements.push(statement);
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.props.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const buildSpec = this.getBuildSpec(this.repository, logGroup, this.props.runnerVersion);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            source: aws_cdk_lib_1.aws_codebuild.Source.s3({
                bucket: this.dockerfile.bucket,
                path: this.dockerfile.s3ObjectKey,
            }),
            vpc: this.props.vpc,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            subnetSelection: this.props.subnetSelection,
            timeout: this.props.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage: this.buildImage,
                computeType: this.props.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        this.policyStatements.forEach(project.addToRolePolicy);
        // call CodeBuild during deployment and delete all images from repository during destruction
        const cr = this.customResource(project);
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.props.rebuildInterval);
        for (const [assetPath, asset] of this.secondaryAssets.entries()) {
            project.addSecondarySource(aws_cdk_lib_1.aws_codebuild.Source.s3({
                identifier: assetPath,
                bucket: asset.bucket,
                path: asset.s3ObjectKey,
            }));
        }
        this.boundImage = {
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryAttributes(this, 'Dependable Image', {
                repositoryName: this.repository.repositoryName,
                // There are simpler ways to get the ARN, but we want an image object that depends on the custom resource.
                // We want whoever is using this image to automatically wait for CodeBuild to start and finish through the custom resource.
                repositoryArn: cr.ref,
            }),
            imageTag: 'latest',
            imageDigest: cr.getAtt('Digest').toString(),
            architecture: this.architecture,
            os: this.os,
        };
        return this.boundImage;
    }
    getBuildImage() {
        if (this.os.is(common_1.Os.LINUX)) {
            if (this.architecture.is(common_1.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_4_0;
            }
            else if (this.architecture.is(common_1.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
            }
        }
        if (this.os.is(common_1.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getBuildSpec(repository, logGroup, runnerVersion) {
        // don't forget to change BUILDSPEC_VERSION when the buildSpec changes, and you want to trigger a rebuild on deploy
        let buildArgs = '';
        for (const [name, value] of this.buildArgs.entries()) {
            buildArgs += ` --build-arg "${name}"="${value}"`;
        }
        buildArgs += ` --build-arg RUNNER_VERSION="${runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version}"`;
        return {
            version: '0.2',
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    STACK_ID: 'unspecified',
                    REQUEST_ID: 'unspecified',
                    LOGICAL_RESOURCE_ID: 'unspecified',
                    RESPONSE_URL: 'unspecified',
                    RUNNER_VERSION: runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version,
                },
            },
            phases: {
                pre_build: {
                    commands: this.preBuild.concat([
                        '$(aws ecr get-login --no-include-email --region "$AWS_DEFAULT_REGION")',
                    ]),
                },
                build: {
                    commands: [
                        `docker build . -t "$REPO_URI" ${buildArgs}`,
                        'docker push "$REPO_URI"',
                    ],
                },
                post_build: {
                    commands: this.postBuild.concat([
                        'STATUS="SUCCESS"',
                        'DIGEST="UNKNOWN"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILED"; else DIGEST=`docker inspect "$REPO_URI" | jq -r \'.[0].RepoDigests[0] | split("@")[1] | split(":")[1]\'`; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "StackId": "$STACK_ID",\n' +
                            '  "RequestId": "$REQUEST_ID",\n' +
                            '  "LogicalResourceId": "$LOGICAL_RESOURCE_ID",\n' +
                            '  "PhysicalResourceId": "$REPO_ARN",\n' +
                            '  "Status": "$STATUS",\n' +
                            `  "Reason": "See logs in ${logGroup.logGroupName}/$CODEBUILD_LOG_PATH (deploy again with \'cdk deploy -R\' or logRemovalPolicy=RemovalPolicy.RETAIN if they are already deleted)",\n` +
                            '  "Data": {"Digest": "$DIGEST"}\n' + // include the digest to mark the resource updated so the runner providers get updated with the latest digest too (specifically Lambda)
                            '}\n' +
                            'EOF',
                        'if [ "$RESPONSE_URL" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$RESPONSE_URL"; fi',
                    ]),
                },
            },
        };
    }
    customResource(project) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.seconds(30),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                // We include a hash so the image is built immediately on changes, and we don't have to wait for its scheduled build.
                // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
                BuildHash: this.hashBuildSettings(),
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
    /**
     * Return hash of all settings that can affect the result image so we can trigger the build when it changes.
     * @private
     */
    hashBuildSettings() {
        // main Dockerfile
        let components = [this.dockerfile.assetHash];
        // all additional files
        for (const [name, asset] of this.secondaryAssets.entries()) {
            components.push(name);
            components.push(asset.assetHash);
        }
        // buildspec.yml version
        components.push(`v${CodeBuildImageBuilder.BUILDSPEC_VERSION}`);
        // user commands
        components = components.concat(this.preBuild);
        components = components.concat(this.postBuild);
        for (const [name, value] of this.buildArgs.entries()) {
            components.push(name);
            components.push(value);
        }
        // hash it
        const all = components.join('-');
        return crypto.createHash('md5').update(all).digest('hex');
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
}
exports.CodeBuildImageBuilder = CodeBuildImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildImageBuilder", version: "0.3.0" };
CodeBuildImageBuilder.BUILDSPEC_VERSION = 1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZWJ1aWxkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Byb3ZpZGVycy9pbWFnZS1idWlsZGVycy9jb2RlYnVpbGQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMsbUNBQW1DO0FBQ25DLDZDQWFxQjtBQUNyQiw2REFBd0Q7QUFDeEQsaURBQStEO0FBQy9ELG1EQUFxRDtBQUNyRCwyQ0FBdUM7QUFDdkMsdUNBQW9EO0FBQ3BELHNDQUF3RjtBQXlHeEY7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXFCRztBQUNILE1BQWEscUJBQXNCLFNBQVEsc0JBQVM7SUFlbEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBVyxLQUFpQztRQUNsRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRGdDLFVBQUssR0FBTCxLQUFLLENBQTRCO1FBUjVFLGFBQVEsR0FBYSxFQUFFLENBQUM7UUFDeEIsY0FBUyxHQUFhLEVBQUUsQ0FBQztRQUN6QixjQUFTLEdBQXdCLElBQUksR0FBRyxFQUFFLENBQUM7UUFDM0MscUJBQWdCLEdBQTBCLEVBQUUsQ0FBQztRQUM3QyxvQkFBZSxHQUFpQyxJQUFJLEdBQUcsRUFBRSxDQUFDO1FBT2hFLElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRSxVQUFVLElBQUkscUJBQUcsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLEVBQUU7WUFDeEUseUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDLDhGQUE4RjtnQkFDMUgsMkRBQTJELENBQUMsQ0FBQztTQUNsRTtRQUVELGVBQWU7UUFDZixJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUkscUJBQVksQ0FBQyxNQUFNLENBQUM7UUFDOUQsSUFBSSxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxJQUFJLFdBQUUsQ0FBQyxLQUFLLENBQUM7UUFFL0IsNENBQTRDO1FBQzVDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxxQkFBRyxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3ZELGVBQWUsRUFBRSxJQUFJO1lBQ3JCLGtCQUFrQixFQUFFLHVCQUFhLENBQUMsT0FBTztZQUN6QyxhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1lBQ3BDLGNBQWMsRUFBRTtnQkFDZDtvQkFDRSxXQUFXLEVBQUUsNkRBQTZEO29CQUMxRSxTQUFTLEVBQUUsbUJBQVMsQ0FBQyxRQUFRO29CQUM3QixXQUFXLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2lCQUM5QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsc0NBQXNDO1FBQ3RDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSwyQkFBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3hELElBQUksRUFBRSxLQUFLLENBQUMsY0FBYztTQUMzQixDQUFDLENBQUM7UUFFSCxxQkFBcUI7UUFDckIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksUUFBUSxDQUFDLFVBQWtCLEVBQUUsUUFBZ0I7UUFDbEQsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztRQUVELE1BQU0sS0FBSyxHQUFHLElBQUksMkJBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUMxQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyw2QkFBNkIsUUFBUSxNQUFNLFFBQVEsR0FBRyxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxrQkFBa0IsQ0FBQyxPQUFlO1FBQ3ZDLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLDBGQUEwRixDQUFDLENBQUM7U0FDN0c7UUFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLE9BQWU7UUFDeEMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztRQUNELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFdBQVcsQ0FBQyxJQUFZLEVBQUUsS0FBYTtRQUM1QyxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbkIsTUFBTSxJQUFJLEtBQUssQ0FBQywwRkFBMEYsQ0FBQyxDQUFDO1NBQzdHO1FBQ0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0JBQWtCLENBQUMsU0FBOEI7UUFDdEQsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztRQUNELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSTtRQUNULElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDeEI7UUFFRCxpQ0FBaUM7UUFDakMsTUFBTSxRQUFRLEdBQUcsSUFBSSxzQkFBSSxDQUFDLFFBQVEsQ0FDaEMsSUFBSSxFQUNKLE1BQU0sRUFDTjtZQUNFLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSx3QkFBYSxDQUFDLFNBQVM7WUFDN0QsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLElBQUksMkJBQWEsQ0FBQyxPQUFPO1NBQ3BFLENBQ0YsQ0FBQztRQUVGLHFCQUFxQjtRQUNyQixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFekYsMkVBQTJFO1FBQzNFLE1BQU0sT0FBTyxHQUFHLElBQUksMkJBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN2RCxXQUFXLEVBQUUsb0RBQW9ELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxHQUFHO1lBQzdILFNBQVMsRUFBRSwyQkFBUyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDO1lBQ3BELE1BQU0sRUFBRSwyQkFBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7Z0JBQzFCLE1BQU0sRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU07Z0JBQzlCLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVc7YUFDbEMsQ0FBQztZQUNGLEdBQUcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUc7WUFDbkIsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDakYsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZTtZQUMzQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksc0JBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ2hELFdBQVcsRUFBRTtnQkFDWCxVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQzNCLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSwyQkFBVyxDQUFDLEtBQUs7Z0JBQ3hELFVBQVUsRUFBRSxJQUFJO2FBQ2pCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLFVBQVUsRUFBRTtvQkFDVixRQUFRO2lCQUNUO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxjQUFjO1FBQ2QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFdkQsNEZBQTRGO1FBQzVGLE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFeEMsOEJBQThCO1FBQzlCLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUVqRSxLQUFLLE1BQU0sQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUMvRCxPQUFPLENBQUMsa0JBQWtCLENBQUMsMkJBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO2dCQUM3QyxVQUFVLEVBQUUsU0FBUztnQkFDckIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO2dCQUNwQixJQUFJLEVBQUUsS0FBSyxDQUFDLFdBQVc7YUFDeEIsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUVELElBQUksQ0FBQyxVQUFVLEdBQUc7WUFDaEIsZUFBZSxFQUFFLHFCQUFHLENBQUMsVUFBVSxDQUFDLHdCQUF3QixDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDakYsY0FBYyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDOUMsMEdBQTBHO2dCQUMxRywySEFBMkg7Z0JBQzNILGFBQWEsRUFBRSxFQUFFLENBQUMsR0FBRzthQUN0QixDQUFDO1lBQ0YsUUFBUSxFQUFFLFFBQVE7WUFDbEIsV0FBVyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsUUFBUSxFQUFFO1lBQzNDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixFQUFFLEVBQUUsSUFBSSxDQUFDLEVBQUU7U0FDWixDQUFDO1FBQ0YsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDO0lBQ3pCLENBQUM7SUFFTyxhQUFhO1FBQ25CLElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsV0FBRSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDN0MsT0FBTywyQkFBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZLENBQUM7YUFDL0M7aUJBQU0sSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUNuRCxPQUFPLDJCQUFTLENBQUMsa0JBQWtCLENBQUMsMkJBQTJCLENBQUM7YUFDakU7U0FDRjtRQUNELElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsV0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMsNEdBQTRHLENBQUMsQ0FBQztTQUMvSDtRQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsc0NBQXNDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNsRyxDQUFDO0lBRU8sWUFBWSxDQUFDLFVBQTBCLEVBQUUsUUFBdUIsRUFBRSxhQUE2QjtRQUNyRyxtSEFBbUg7UUFDbkgsSUFBSSxTQUFTLEdBQUcsRUFBRSxDQUFDO1FBQ25CLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ3BELFNBQVMsSUFBSSxpQkFBaUIsSUFBSSxNQUFNLEtBQUssR0FBRyxDQUFDO1NBQ2xEO1FBQ0QsU0FBUyxJQUFJLGdDQUFnQyxhQUFhLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLHNCQUFhLENBQUMsTUFBTSxFQUFFLENBQUMsT0FBTyxHQUFHLENBQUM7UUFFdkgsT0FBTztZQUNMLE9BQU8sRUFBRSxLQUFLO1lBQ2QsR0FBRyxFQUFFO2dCQUNILFNBQVMsRUFBRTtvQkFDVCxRQUFRLEVBQUUsVUFBVSxDQUFDLGFBQWE7b0JBQ2xDLFFBQVEsRUFBRSxVQUFVLENBQUMsYUFBYTtvQkFDbEMsUUFBUSxFQUFFLGFBQWE7b0JBQ3ZCLFVBQVUsRUFBRSxhQUFhO29CQUN6QixtQkFBbUIsRUFBRSxhQUFhO29CQUNsQyxZQUFZLEVBQUUsYUFBYTtvQkFDM0IsY0FBYyxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsc0JBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxPQUFPO2lCQUN2RjthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLFNBQVMsRUFBRTtvQkFDVCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUM7d0JBQzdCLHdFQUF3RTtxQkFDekUsQ0FBQztpQkFDSDtnQkFDRCxLQUFLLEVBQUU7b0JBQ0wsUUFBUSxFQUFFO3dCQUNSLGlDQUFpQyxTQUFTLEVBQUU7d0JBQzVDLHlCQUF5QjtxQkFDMUI7aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLFFBQVEsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQzt3QkFDOUIsa0JBQWtCO3dCQUNsQixrQkFBa0I7d0JBQ2xCLDhLQUE4Szt3QkFDOUssaUNBQWlDOzRCQUNqQyxLQUFLOzRCQUNMLDZCQUE2Qjs0QkFDN0IsaUNBQWlDOzRCQUNqQyxrREFBa0Q7NEJBQ2xELHdDQUF3Qzs0QkFDeEMsMEJBQTBCOzRCQUMxQiw0QkFBNEIsUUFBUSxDQUFDLFlBQVkscUlBQXFJOzRCQUN0TCxtQ0FBbUMsR0FBRyx1SUFBdUk7NEJBQzdLLEtBQUs7NEJBQ0wsS0FBSzt3QkFDTCx3SkFBd0o7cUJBQ3pKLENBQUM7aUJBQ0g7YUFDRjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sY0FBYyxDQUFDLE9BQTBCO1FBQy9DLE1BQU0sU0FBUyxHQUFHLDZCQUFxQixDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ3JFLFdBQVcsRUFBRSwwR0FBMEc7WUFDdkgsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUNsQyxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDL0MsVUFBVSxFQUFFO2dCQUNWLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixDQUFDO29CQUNqQyxTQUFTLEVBQUUsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO2lCQUNoQyxDQUFDO2dCQUNGLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixFQUFFLGdCQUFnQixDQUFDO29CQUNuRCxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQztpQkFDM0MsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsU0FBUyxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUzQyxNQUFNLEVBQUUsR0FBRyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUM3QyxZQUFZLEVBQUUsU0FBUyxDQUFDLFdBQVc7WUFDbkMsWUFBWSxFQUFFLHNCQUFzQjtZQUNwQyxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDeEMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxXQUFXO2dCQUNoQyxxSEFBcUg7Z0JBQ3JILG9JQUFvSTtnQkFDcEksU0FBUyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsRUFBRTthQUNwQztTQUNGLENBQUMsQ0FBQztRQUVILHNFQUFzRTtRQUN0RSxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMvQixFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5QixFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUVqQyxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFRDs7O09BR0c7SUFDSyxpQkFBaUI7UUFDdkIsa0JBQWtCO1FBQ2xCLElBQUksVUFBVSxHQUFhLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN2RCx1QkFBdUI7UUFDdkIsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxFQUFFLEVBQUU7WUFDMUQsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN0QixVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUNsQztRQUNELHdCQUF3QjtRQUN4QixVQUFVLENBQUMsSUFBSSxDQUFDLElBQUkscUJBQXFCLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1FBQy9ELGdCQUFnQjtRQUNoQixVQUFVLEdBQUcsVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDOUMsVUFBVSxHQUFHLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQy9DLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ3BELFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDdEIsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUN4QjtRQUNELFVBQVU7UUFDVixNQUFNLEdBQUcsR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxPQUEwQixFQUFFLGVBQTBCO1FBQ25GLGVBQWUsR0FBRyxlQUFlLElBQUksc0JBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEQsSUFBSSxlQUFlLENBQUMsY0FBYyxFQUFFLElBQUksQ0FBQyxFQUFFO1lBQ3pDLE1BQU0sWUFBWSxHQUFHLElBQUksd0JBQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO2dCQUMzRCxXQUFXLEVBQUUsNEJBQTRCLElBQUksQ0FBQyxVQUFVLENBQUMsY0FBYyxFQUFFO2dCQUN6RSxRQUFRLEVBQUUsd0JBQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQzthQUNoRCxDQUFDLENBQUM7WUFDSCxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksZ0NBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1NBQ3RFO0lBQ0gsQ0FBQzs7QUFuVkgsc0RBb1ZDOzs7QUFuVmdCLHVDQUFpQixHQUFHLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNyeXB0byBmcm9tICdjcnlwdG8nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7XG4gIEFubm90YXRpb25zLFxuICBhd3NfY29kZWJ1aWxkIGFzIGNvZGVidWlsZCxcbiAgYXdzX2VjMiBhcyBlYzIsXG4gIGF3c19lY3IgYXMgZWNyLFxuICBhd3NfZXZlbnRzIGFzIGV2ZW50cyxcbiAgYXdzX2V2ZW50c190YXJnZXRzIGFzIGV2ZW50c190YXJnZXRzLFxuICBhd3NfaWFtIGFzIGlhbSxcbiAgYXdzX2xvZ3MgYXMgbG9ncyxcbiAgYXdzX3MzX2Fzc2V0cyBhcyBzM19hc3NldHMsXG4gIEN1c3RvbVJlc291cmNlLFxuICBEdXJhdGlvbixcbiAgUmVtb3ZhbFBvbGljeSxcbn0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29tcHV0ZVR5cGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY29kZWJ1aWxkJztcbmltcG9ydCB7IFRhZ011dGFiaWxpdHksIFRhZ1N0YXR1cyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3InO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQnVuZGxlZE5vZGVqc0Z1bmN0aW9uIH0gZnJvbSAnLi4vLi4vdXRpbHMnO1xuaW1wb3J0IHsgQXJjaGl0ZWN0dXJlLCBJSW1hZ2VCdWlsZGVyLCBPcywgUnVubmVySW1hZ2UsIFJ1bm5lclZlcnNpb24gfSBmcm9tICcuLi9jb21tb24nO1xuXG4vKlxuQVdTIEltYWdlIEJ1aWxkZXIgd2FzIG5vdCB1c2VkIGJlY2F1c2U6XG4gIDEuIEl0J3MgdG9vIHNsb3cuIEl0IGhhcyB3ZWlyZCAxNSBtaW51dGVzIG92ZXJoZWFkIHdoZXJlIGl0IHNlZW1zIHRvIGp1c3QgYmUgd2FpdGluZy5cbiAgMi4gTm8gZWFzeSBsb2cgdmlzaWJpbGl0eS5cbiAgMy4gVmVyc2lvbnMgbmVlZCB0byBiZSBidW1wZWQgbWFudWFsbHkuXG4gKi9cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBDb2RlQnVpbGRJbWFnZUJ1aWxkZXIgY29uc3RydWN0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvZGVCdWlsZEltYWdlQnVpbGRlclByb3BzIHtcbiAgLyoqXG4gICAqIEltYWdlIGFyY2hpdGVjdHVyZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQXJjaGl0ZWN0dXJlLlg4Nl82NFxuICAgKi9cbiAgcmVhZG9ubHkgYXJjaGl0ZWN0dXJlPzogQXJjaGl0ZWN0dXJlO1xuXG4gIC8qKlxuICAgKiBJbWFnZSBPUy5cbiAgICpcbiAgICogQGRlZmF1bHQgT1MuTElOVVhcbiAgICovXG4gIHJlYWRvbmx5IG9zPzogT3M7XG5cbiAgLyoqXG4gICAqIFBhdGggdG8gRG9ja2VyZmlsZSB0byBiZSBidWlsdC4gSXQgY2FuIGJlIGEgcGF0aCB0byBhIERvY2tlcmZpbGUsIGEgZm9sZGVyIGNvbnRhaW5pbmcgYSBEb2NrZXJmaWxlLCBvciBhIHppcCBmaWxlIGNvbnRhaW5pbmcgYSBEb2NrZXJmaWxlLlxuICAgKi9cbiAgcmVhZG9ubHkgZG9ja2VyZmlsZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogVmVyc2lvbiBvZiBHaXRIdWIgUnVubmVycyB0byBpbnN0YWxsLlxuICAgKlxuICAgKiBAZGVmYXVsdCBsYXRlc3QgdmVyc2lvbiBhdmFpbGFibGVcbiAgICovXG4gIHJlYWRvbmx5IHJ1bm5lclZlcnNpb24/OiBSdW5uZXJWZXJzaW9uO1xuXG4gIC8qKlxuICAgKiBTY2hlZHVsZSB0aGUgaW1hZ2UgdG8gYmUgcmVidWlsdCBldmVyeSBnaXZlbiBpbnRlcnZhbC4gVXNlZnVsIGZvciBrZWVwaW5nIHRoZSBpbWFnZSB1cC1kby1kYXRlIHdpdGggdGhlIGxhdGVzdCBHaXRIdWIgcnVubmVyIHZlcnNpb24gYW5kIGxhdGVzdCBPUyB1cGRhdGVzLlxuICAgKlxuICAgKiBTZXQgdG8gemVybyB0byBkaXNhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDcpXG4gICAqL1xuICByZWFkb25seSByZWJ1aWxkSW50ZXJ2YWw/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVlBDIHRvIGxhdW5jaCB0aGUgcnVubmVycyBpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gVlBDXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAvKipcbiAgICogU2VjdXJpdHkgR3JvdXAgdG8gYXNzaWduIHRvIHRoaXMgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IHB1YmxpYyBwcm9qZWN0IHdpdGggbm8gc2VjdXJpdHkgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG5cbiAgLyoqXG4gICAqIFdoZXJlIHRvIHBsYWNlIHRoZSBuZXR3b3JrIGludGVyZmFjZXMgd2l0aGluIHRoZSBWUEMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgY29tcHV0ZSB0byB1c2UgZm9yIHRoaXMgYnVpbGQuXG4gICAqIFNlZSB0aGUge0BsaW5rIENvbXB1dGVUeXBlfSBlbnVtIGZvciB0aGUgcG9zc2libGUgdmFsdWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB7QGxpbmsgQ29tcHV0ZVR5cGUjU01BTEx9XG4gICAqL1xuICByZWFkb25seSBjb21wdXRlVHlwZT86IGNvZGVidWlsZC5Db21wdXRlVHlwZTtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBtaW51dGVzIGFmdGVyIHdoaWNoIEFXUyBDb2RlQnVpbGQgc3RvcHMgdGhlIGJ1aWxkIGlmIGl0J3NcbiAgICogbm90IGNvbXBsZXRlLiBGb3IgdmFsaWQgdmFsdWVzLCBzZWUgdGhlIHRpbWVvdXRJbk1pbnV0ZXMgZmllbGQgaW4gdGhlIEFXU1xuICAgKiBDb2RlQnVpbGQgVXNlciBHdWlkZS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uaG91cnMoMSlcbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBkYXlzIGxvZyBldmVudHMgYXJlIGtlcHQgaW4gQ2xvdWRXYXRjaCBMb2dzLiBXaGVuIHVwZGF0aW5nXG4gICAqIHRoaXMgcHJvcGVydHksIHVuc2V0dGluZyBpdCBkb2Vzbid0IHJlbW92ZSB0aGUgbG9nIHJldGVudGlvbiBwb2xpY3kuIFRvXG4gICAqIHJlbW92ZSB0aGUgcmV0ZW50aW9uIHBvbGljeSwgc2V0IHRoZSB2YWx1ZSB0byBgSU5GSU5JVEVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRIXG4gICAqL1xuICByZWFkb25seSBsb2dSZXRlbnRpb24/OiBsb2dzLlJldGVudGlvbkRheXM7XG5cbiAgLyoqXG4gICAqIFJlbW92YWwgcG9saWN5IGZvciBsb2dzIG9mIGltYWdlIGJ1aWxkcy4gSWYgZGVwbG95bWVudCBmYWlscyBvbiB0aGUgY3VzdG9tIHJlc291cmNlLCB0cnkgc2V0dGluZyB0aGlzIHRvIGBSZW1vdmFsUG9saWN5LlJFVEFJTmAuIFRoaXMgd2F5IHRoZSBDb2RlQnVpbGQgbG9ncyBjYW4gc3RpbGwgYmUgdmlld2VkLCBhbmQgeW91IGNhbiBzZWUgd2h5IHRoZSBidWlsZCBmYWlsZWQuXG4gICAqXG4gICAqIFdlIHRyeSB0byBub3QgbGVhdmUgYW55dGhpbmcgYmVoaW5kIHdoZW4gcmVtb3ZlZC4gQnV0IHNvbWV0aW1lcyBhIGxvZyBzdGF5aW5nIGJlaGluZCBpcyB1c2VmdWwuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG59XG5cbi8qKlxuICogQW4gaW1hZ2UgYnVpbGRlciB0aGF0IHVzZXMgQ29kZUJ1aWxkIHRvIGJ1aWxkIERvY2tlciBpbWFnZXMgcHJlLWJha2VkIHdpdGggYWxsIHRoZSBHaXRIdWIgQWN0aW9ucyBydW5uZXIgcmVxdWlyZW1lbnRzLiBCdWlsZGVycyBjYW4gYmUgdXNlZCB3aXRoIHJ1bm5lciBwcm92aWRlcnMuXG4gKlxuICogRWFjaCBidWlsZGVyIHJlLXJ1bnMgYXV0b21hdGljYWxseSBhdCBhIHNldCBpbnRlcnZhbCB0byBtYWtlIHN1cmUgdGhlIGltYWdlcyBjb250YWluIHRoZSBsYXRlc3QgdmVyc2lvbnMgb2YgZXZlcnl0aGluZy5cbiAqXG4gKiBZb3UgY2FuIGNyZWF0ZSBhbiBpbnN0YW5jZSBvZiB0aGlzIGNvbnN0cnVjdCB0byBjdXN0b21pemUgdGhlIGltYWdlIHVzZWQgdG8gc3Bpbi11cCBydW5uZXJzLiBFYWNoIHByb3ZpZGVyIGhhcyBpdHMgb3duIHJlcXVpcmVtZW50cyBmb3Igd2hhdCBhbiBpbWFnZSBzaG91bGQgZG8uIFRoYXQncyB3aHkgdGhleSBlYWNoIHByb3ZpZGUgdGhlaXIgb3duIERvY2tlcmZpbGUuXG4gKlxuICogRm9yIGV4YW1wbGUsIHRvIHNldCBhIHNwZWNpZmljIHJ1bm5lciB2ZXJzaW9uLCByZWJ1aWxkIHRoZSBpbWFnZSBldmVyeSAyIHdlZWtzLCBhbmQgYWRkIGEgZmV3IHBhY2thZ2VzIGZvciB0aGUgRmFyZ2F0ZSBwcm92aWRlciwgdXNlOlxuICpcbiAqIGBgYFxuICogY29uc3QgYnVpbGRlciA9IG5ldyBDb2RlQnVpbGRJbWFnZUJ1aWxkZXIodGhpcywgJ0J1aWxkZXInLCB7XG4gKiAgICAgZG9ja2VyZmlsZVBhdGg6IEZhcmdhdGVQcm92aWRlci5MSU5VWF9YNjRfRE9DS0VSRklMRV9QQVRILFxuICogICAgIHJ1bm5lclZlcnNpb246IFJ1bm5lclZlcnNpb24uc3BlY2lmaWMoJzIuMjkzLjAnKSxcbiAqICAgICByZWJ1aWxkSW50ZXJ2YWw6IER1cmF0aW9uLmRheXMoMTQpLFxuICogfSk7XG4gKiBidWlsZGVyLnNldEJ1aWxkQXJnKCdFWFRSQV9QQUNLQUdFUycsICduZ2lueCB4ei11dGlscycpO1xuICogbmV3IEZhcmdhdGVQcm92aWRlcih0aGlzLCAnRmFyZ2F0ZSBwcm92aWRlcicsIHtcbiAqICAgICBsYWJlbDogJ2N1c3RvbWl6ZWQtZmFyZ2F0ZScsXG4gKiAgICAgaW1hZ2VCdWlsZGVyOiBidWlsZGVyLFxuICogfSk7XG4gKiBgYGBcbiAqL1xuZXhwb3J0IGNsYXNzIENvZGVCdWlsZEltYWdlQnVpbGRlciBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElJbWFnZUJ1aWxkZXIge1xuICBwcml2YXRlIHN0YXRpYyBCVUlMRFNQRUNfVkVSU0lPTiA9IDE7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZTtcbiAgcHJpdmF0ZSByZWFkb25seSBvczogT3M7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVwb3NpdG9yeTogZWNyLlJlcG9zaXRvcnk7XG4gIHByaXZhdGUgcmVhZG9ubHkgZG9ja2VyZmlsZTogczNfYXNzZXRzLkFzc2V0O1xuICBwcml2YXRlIHByZUJ1aWxkOiBzdHJpbmdbXSA9IFtdO1xuICBwcml2YXRlIHBvc3RCdWlsZDogc3RyaW5nW10gPSBbXTtcbiAgcHJpdmF0ZSBidWlsZEFyZ3M6IE1hcDxzdHJpbmcsIHN0cmluZz4gPSBuZXcgTWFwKCk7XG4gIHByaXZhdGUgcG9saWN5U3RhdGVtZW50czogaWFtLlBvbGljeVN0YXRlbWVudFtdID0gW107XG4gIHByaXZhdGUgc2Vjb25kYXJ5QXNzZXRzOiBNYXA8c3RyaW5nLCBzM19hc3NldHMuQXNzZXQ+ID0gbmV3IE1hcCgpO1xuICBwcml2YXRlIHJlYWRvbmx5IGJ1aWxkSW1hZ2U6IGNvZGVidWlsZC5JQnVpbGRJbWFnZTtcbiAgcHJpdmF0ZSBib3VuZEltYWdlPzogUnVubmVySW1hZ2U7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcmVhZG9ubHkgcHJvcHM6IENvZGVCdWlsZEltYWdlQnVpbGRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcy5zdWJuZXRTZWxlY3Rpb24/LnN1Ym5ldFR5cGUgPT0gZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCkge1xuICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZygnUHJpdmF0ZSBpc29sYXRlZCBzdWJuZXRzIGNhbm5vdCBwdWxsIGZyb20gcHVibGljIEVDUiBhbmQgVlBDIGVuZHBvaW50IGlzIG5vdCBzdXBwb3J0ZWQgeWV0LiAnICtcbiAgICAgICAgICAnU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvY29udGFpbmVycy1yb2FkbWFwL2lzc3Vlcy8xMTYwJyk7XG4gICAgfVxuXG4gICAgLy8gc2V0IHBsYXRmb3JtXG4gICAgdGhpcy5hcmNoaXRlY3R1cmUgPSBwcm9wcy5hcmNoaXRlY3R1cmUgPz8gQXJjaGl0ZWN0dXJlLlg4Nl82NDtcbiAgICB0aGlzLm9zID0gcHJvcHMub3MgPz8gT3MuTElOVVg7XG5cbiAgICAvLyBjcmVhdGUgcmVwb3NpdG9yeSB0aGF0IG9ubHkga2VlcHMgb25lIHRhZ1xuICAgIHRoaXMucmVwb3NpdG9yeSA9IG5ldyBlY3IuUmVwb3NpdG9yeSh0aGlzLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgIGltYWdlU2Nhbk9uUHVzaDogdHJ1ZSxcbiAgICAgIGltYWdlVGFnTXV0YWJpbGl0eTogVGFnTXV0YWJpbGl0eS5NVVRBQkxFLFxuICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgbGlmZWN5Y2xlUnVsZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnUmVtb3ZlIHVudGFnZ2VkIGltYWdlcyB0aGF0IGhhdmUgYmVlbiByZXBsYWNlZCBieSBDb2RlQnVpbGQnLFxuICAgICAgICAgIHRhZ1N0YXR1czogVGFnU3RhdHVzLlVOVEFHR0VELFxuICAgICAgICAgIG1heEltYWdlQWdlOiBEdXJhdGlvbi5kYXlzKDEpLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIC8vIHVwbG9hZCBEb2NrZXJmaWxlIHRvIFMzIGFzIGFuIGFzc2V0XG4gICAgdGhpcy5kb2NrZXJmaWxlID0gbmV3IHMzX2Fzc2V0cy5Bc3NldCh0aGlzLCAnRG9ja2VyZmlsZScsIHtcbiAgICAgIHBhdGg6IHByb3BzLmRvY2tlcmZpbGVQYXRoLFxuICAgIH0pO1xuXG4gICAgLy8gY2hvb3NlIGJ1aWxkIGltYWdlXG4gICAgdGhpcy5idWlsZEltYWdlID0gdGhpcy5nZXRCdWlsZEltYWdlKCk7XG4gIH1cblxuICAvKipcbiAgICogVXBsb2FkcyBhIGZvbGRlciB0byB0aGUgYnVpbGQgc2VydmVyIGF0IGEgZ2l2ZW4gZm9sZGVyIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBzb3VyY2VQYXRoIHBhdGggdG8gc291cmNlIGRpcmVjdG9yeVxuICAgKiBAcGFyYW0gZGVzdE5hbWUgbmFtZSBvZiBkZXN0aW5hdGlvbiBmb2xkZXJcbiAgICovXG4gIHB1YmxpYyBhZGRGaWxlcyhzb3VyY2VQYXRoOiBzdHJpbmcsIGRlc3ROYW1lOiBzdHJpbmcpIHtcbiAgICBpZiAodGhpcy5ib3VuZEltYWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ltYWdlIGlzIGFscmVhZHkgYm91bmQuIFVzZSB0aGlzIG1ldGhvZCBiZWZvcmUgcGFzc2luZyB0aGUgYnVpbGRlciB0byBhIHJ1bm5lciBwcm92aWRlci4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBhc3NldCA9IG5ldyBzM19hc3NldHMuQXNzZXQodGhpcywgZGVzdE5hbWUsIHsgcGF0aDogc291cmNlUGF0aCB9KTtcbiAgICB0aGlzLnNlY29uZGFyeUFzc2V0cy5zZXQoZGVzdE5hbWUsIGFzc2V0KTtcbiAgICB0aGlzLnByZUJ1aWxkLnB1c2goYGxuIC1zIFwiJENPREVCVUlMRF9TUkNfRElSXyR7ZGVzdE5hbWV9XCIgXCIke2Rlc3ROYW1lfVwiYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGNvbW1hbmQgdGhhdCBydW5zIGJlZm9yZSBgZG9ja2VyIGJ1aWxkYC5cbiAgICpcbiAgICogQHBhcmFtIGNvbW1hbmQgY29tbWFuZCB0byBhZGRcbiAgICovXG4gIHB1YmxpYyBhZGRQcmVCdWlsZENvbW1hbmQoY29tbWFuZDogc3RyaW5nKSB7XG4gICAgaWYgKHRoaXMuYm91bmRJbWFnZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbWFnZSBpcyBhbHJlYWR5IGJvdW5kLiBVc2UgdGhpcyBtZXRob2QgYmVmb3JlIHBhc3NpbmcgdGhlIGJ1aWxkZXIgdG8gYSBydW5uZXIgcHJvdmlkZXIuJyk7XG4gICAgfVxuICAgIHRoaXMucHJlQnVpbGQucHVzaChjb21tYW5kKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgY29tbWFuZCB0aGF0IHJ1bnMgYWZ0ZXIgYGRvY2tlciBidWlsZGAgYW5kIGBkb2NrZXIgcHVzaGAuXG4gICAqXG4gICAqIEBwYXJhbSBjb21tYW5kIGNvbW1hbmQgdG8gYWRkXG4gICAqL1xuICBwdWJsaWMgYWRkUG9zdEJ1aWxkQ29tbWFuZChjb21tYW5kOiBzdHJpbmcpIHtcbiAgICBpZiAodGhpcy5ib3VuZEltYWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ltYWdlIGlzIGFscmVhZHkgYm91bmQuIFVzZSB0aGlzIG1ldGhvZCBiZWZvcmUgcGFzc2luZyB0aGUgYnVpbGRlciB0byBhIHJ1bm5lciBwcm92aWRlci4nKTtcbiAgICB9XG4gICAgdGhpcy5wb3N0QnVpbGQucHVzaChjb21tYW5kKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgYnVpbGQgYXJndW1lbnQgZm9yIERvY2tlci4gU2VlIHRoZSBkb2N1bWVudGF0aW9uIGZvciB0aGUgRG9ja2VyZmlsZSB5b3UncmUgdXNpbmcgZm9yIGEgbGlzdCBvZiBzdXBwb3J0ZWQgYnVpbGQgYXJndW1lbnRzLlxuICAgKlxuICAgKiBAcGFyYW0gbmFtZSBidWlsZCBhcmd1bWVudCBuYW1lXG4gICAqIEBwYXJhbSB2YWx1ZSBidWlsZCBhcmd1bWVudCB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHNldEJ1aWxkQXJnKG5hbWU6IHN0cmluZywgdmFsdWU6IHN0cmluZykge1xuICAgIGlmICh0aGlzLmJvdW5kSW1hZ2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSW1hZ2UgaXMgYWxyZWFkeSBib3VuZC4gVXNlIHRoaXMgbWV0aG9kIGJlZm9yZSBwYXNzaW5nIHRoZSBidWlsZGVyIHRvIGEgcnVubmVyIHByb3ZpZGVyLicpO1xuICAgIH1cbiAgICB0aGlzLmJ1aWxkQXJncy5zZXQobmFtZSwgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHBvbGljeSBzdGF0ZW1lbnQgdG8gdGhlIGJ1aWxkZXIgdG8gYWNjZXNzIHJlc291cmNlcyByZXF1aXJlZCB0byB0aGUgaW1hZ2UgYnVpbGQuXG4gICAqXG4gICAqIEBwYXJhbSBzdGF0ZW1lbnQgSUFNIHBvbGljeSBzdGF0ZW1lbnRcbiAgICovXG4gIHB1YmxpYyBhZGRQb2xpY3lTdGF0ZW1lbnQoc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KSB7XG4gICAgaWYgKHRoaXMuYm91bmRJbWFnZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbWFnZSBpcyBhbHJlYWR5IGJvdW5kLiBVc2UgdGhpcyBtZXRob2QgYmVmb3JlIHBhc3NpbmcgdGhlIGJ1aWxkZXIgdG8gYSBydW5uZXIgcHJvdmlkZXIuJyk7XG4gICAgfVxuICAgIHRoaXMucG9saWN5U3RhdGVtZW50cy5wdXNoKHN0YXRlbWVudCk7XG4gIH1cblxuICAvKipcbiAgICogQ2FsbGVkIGJ5IElSdW5uZXJQcm92aWRlciB0byBmaW5hbGl6ZSBzZXR0aW5ncyBhbmQgY3JlYXRlIHRoZSBpbWFnZSBidWlsZGVyLlxuICAgKi9cbiAgcHVibGljIGJpbmQoKTogUnVubmVySW1hZ2Uge1xuICAgIGlmICh0aGlzLmJvdW5kSW1hZ2UpIHtcbiAgICAgIHJldHVybiB0aGlzLmJvdW5kSW1hZ2U7XG4gICAgfVxuXG4gICAgLy8gbG9nIGdyb3VwIGZvciB0aGUgaW1hZ2UgYnVpbGRzXG4gICAgY29uc3QgbG9nR3JvdXAgPSBuZXcgbG9ncy5Mb2dHcm91cChcbiAgICAgIHRoaXMsXG4gICAgICAnTG9ncycsXG4gICAgICB7XG4gICAgICAgIHJldGVudGlvbjogdGhpcy5wcm9wcy5sb2dSZXRlbnRpb24gPz8gUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IHRoaXMucHJvcHMubG9nUmVtb3ZhbFBvbGljeSA/PyBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICB9LFxuICAgICk7XG5cbiAgICAvLyBnZW5lcmF0ZSBidWlsZFNwZWNcbiAgICBjb25zdCBidWlsZFNwZWMgPSB0aGlzLmdldEJ1aWxkU3BlYyh0aGlzLnJlcG9zaXRvcnksIGxvZ0dyb3VwLCB0aGlzLnByb3BzLnJ1bm5lclZlcnNpb24pO1xuXG4gICAgLy8gY3JlYXRlIENvZGVCdWlsZCBwcm9qZWN0IHRoYXQgYnVpbGRzIERvY2tlcmZpbGUgYW5kIHB1c2hlcyB0byByZXBvc2l0b3J5XG4gICAgY29uc3QgcHJvamVjdCA9IG5ldyBjb2RlYnVpbGQuUHJvamVjdCh0aGlzLCAnQ29kZUJ1aWxkJywge1xuICAgICAgZGVzY3JpcHRpb246IGBCdWlsZCBkb2NrZXIgaW1hZ2UgZm9yIHNlbGYtaG9zdGVkIEdpdEh1YiBydW5uZXIgJHt0aGlzLm5vZGUucGF0aH0gKCR7dGhpcy5vcy5uYW1lfS8ke3RoaXMuYXJjaGl0ZWN0dXJlLm5hbWV9KWAsXG4gICAgICBidWlsZFNwZWM6IGNvZGVidWlsZC5CdWlsZFNwZWMuZnJvbU9iamVjdChidWlsZFNwZWMpLFxuICAgICAgc291cmNlOiBjb2RlYnVpbGQuU291cmNlLnMzKHtcbiAgICAgICAgYnVja2V0OiB0aGlzLmRvY2tlcmZpbGUuYnVja2V0LFxuICAgICAgICBwYXRoOiB0aGlzLmRvY2tlcmZpbGUuczNPYmplY3RLZXksXG4gICAgICB9KSxcbiAgICAgIHZwYzogdGhpcy5wcm9wcy52cGMsXG4gICAgICBzZWN1cml0eUdyb3VwczogdGhpcy5wcm9wcy5zZWN1cml0eUdyb3VwID8gW3RoaXMucHJvcHMuc2VjdXJpdHlHcm91cF0gOiB1bmRlZmluZWQsXG4gICAgICBzdWJuZXRTZWxlY3Rpb246IHRoaXMucHJvcHMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgdGltZW91dDogdGhpcy5wcm9wcy50aW1lb3V0ID8/IER1cmF0aW9uLmhvdXJzKDEpLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgYnVpbGRJbWFnZTogdGhpcy5idWlsZEltYWdlLFxuICAgICAgICBjb21wdXRlVHlwZTogdGhpcy5wcm9wcy5jb21wdXRlVHlwZSA/PyBDb21wdXRlVHlwZS5TTUFMTCxcbiAgICAgICAgcHJpdmlsZWdlZDogdHJ1ZSxcbiAgICAgIH0sXG4gICAgICBsb2dnaW5nOiB7XG4gICAgICAgIGNsb3VkV2F0Y2g6IHtcbiAgICAgICAgICBsb2dHcm91cCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBwZXJtaXNzaW9uc1xuICAgIHRoaXMucmVwb3NpdG9yeS5ncmFudFB1bGxQdXNoKHByb2plY3QpO1xuICAgIHRoaXMucG9saWN5U3RhdGVtZW50cy5mb3JFYWNoKHByb2plY3QuYWRkVG9Sb2xlUG9saWN5KTtcblxuICAgIC8vIGNhbGwgQ29kZUJ1aWxkIGR1cmluZyBkZXBsb3ltZW50IGFuZCBkZWxldGUgYWxsIGltYWdlcyBmcm9tIHJlcG9zaXRvcnkgZHVyaW5nIGRlc3RydWN0aW9uXG4gICAgY29uc3QgY3IgPSB0aGlzLmN1c3RvbVJlc291cmNlKHByb2plY3QpO1xuXG4gICAgLy8gcmVidWlsZCBpbWFnZSBvbiBhIHNjaGVkdWxlXG4gICAgdGhpcy5yZWJ1aWxkSW1hZ2VPblNjaGVkdWxlKHByb2plY3QsIHRoaXMucHJvcHMucmVidWlsZEludGVydmFsKTtcblxuICAgIGZvciAoY29uc3QgW2Fzc2V0UGF0aCwgYXNzZXRdIG9mIHRoaXMuc2Vjb25kYXJ5QXNzZXRzLmVudHJpZXMoKSkge1xuICAgICAgcHJvamVjdC5hZGRTZWNvbmRhcnlTb3VyY2UoY29kZWJ1aWxkLlNvdXJjZS5zMyh7XG4gICAgICAgIGlkZW50aWZpZXI6IGFzc2V0UGF0aCxcbiAgICAgICAgYnVja2V0OiBhc3NldC5idWNrZXQsXG4gICAgICAgIHBhdGg6IGFzc2V0LnMzT2JqZWN0S2V5LFxuICAgICAgfSkpO1xuICAgIH1cblxuICAgIHRoaXMuYm91bmRJbWFnZSA9IHtcbiAgICAgIGltYWdlUmVwb3NpdG9yeTogZWNyLlJlcG9zaXRvcnkuZnJvbVJlcG9zaXRvcnlBdHRyaWJ1dGVzKHRoaXMsICdEZXBlbmRhYmxlIEltYWdlJywge1xuICAgICAgICByZXBvc2l0b3J5TmFtZTogdGhpcy5yZXBvc2l0b3J5LnJlcG9zaXRvcnlOYW1lLFxuICAgICAgICAvLyBUaGVyZSBhcmUgc2ltcGxlciB3YXlzIHRvIGdldCB0aGUgQVJOLCBidXQgd2Ugd2FudCBhbiBpbWFnZSBvYmplY3QgdGhhdCBkZXBlbmRzIG9uIHRoZSBjdXN0b20gcmVzb3VyY2UuXG4gICAgICAgIC8vIFdlIHdhbnQgd2hvZXZlciBpcyB1c2luZyB0aGlzIGltYWdlIHRvIGF1dG9tYXRpY2FsbHkgd2FpdCBmb3IgQ29kZUJ1aWxkIHRvIHN0YXJ0IGFuZCBmaW5pc2ggdGhyb3VnaCB0aGUgY3VzdG9tIHJlc291cmNlLlxuICAgICAgICByZXBvc2l0b3J5QXJuOiBjci5yZWYsXG4gICAgICB9KSxcbiAgICAgIGltYWdlVGFnOiAnbGF0ZXN0JyxcbiAgICAgIGltYWdlRGlnZXN0OiBjci5nZXRBdHQoJ0RpZ2VzdCcpLnRvU3RyaW5nKCksXG4gICAgICBhcmNoaXRlY3R1cmU6IHRoaXMuYXJjaGl0ZWN0dXJlLFxuICAgICAgb3M6IHRoaXMub3MsXG4gICAgfTtcbiAgICByZXR1cm4gdGhpcy5ib3VuZEltYWdlO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRCdWlsZEltYWdlKCk6IGNvZGVidWlsZC5JQnVpbGRJbWFnZSB7XG4gICAgaWYgKHRoaXMub3MuaXMoT3MuTElOVVgpKSB7XG4gICAgICBpZiAodGhpcy5hcmNoaXRlY3R1cmUuaXMoQXJjaGl0ZWN0dXJlLlg4Nl82NCkpIHtcbiAgICAgICAgcmV0dXJuIGNvZGVidWlsZC5MaW51eEJ1aWxkSW1hZ2UuU1RBTkRBUkRfNF8wO1xuICAgICAgfSBlbHNlIGlmICh0aGlzLmFyY2hpdGVjdHVyZS5pcyhBcmNoaXRlY3R1cmUuQVJNNjQpKSB7XG4gICAgICAgIHJldHVybiBjb2RlYnVpbGQuTGludXhBcm1CdWlsZEltYWdlLkFNQVpPTl9MSU5VWF8yX1NUQU5EQVJEXzJfMDtcbiAgICAgIH1cbiAgICB9XG4gICAgaWYgKHRoaXMub3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29kZUJ1aWxkIGNhbm5vdCBiZSB1c2VkIHRvIGJ1aWxkIFdpbmRvd3MgRG9ja2VyIGltYWdlcyBodHRwczovL2dpdGh1Yi5jb20vZG9ja2VyLWxpYnJhcnkvZG9ja2VyL2lzc3Vlcy80OScpO1xuICAgIH1cblxuICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGZpbmQgQ29kZUJ1aWxkIGltYWdlIGZvciAke3RoaXMub3MubmFtZX0vJHt0aGlzLmFyY2hpdGVjdHVyZS5uYW1lfWApO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRCdWlsZFNwZWMocmVwb3NpdG9yeTogZWNyLlJlcG9zaXRvcnksIGxvZ0dyb3VwOiBsb2dzLkxvZ0dyb3VwLCBydW5uZXJWZXJzaW9uPzogUnVubmVyVmVyc2lvbik6IGFueSB7XG4gICAgLy8gZG9uJ3QgZm9yZ2V0IHRvIGNoYW5nZSBCVUlMRFNQRUNfVkVSU0lPTiB3aGVuIHRoZSBidWlsZFNwZWMgY2hhbmdlcywgYW5kIHlvdSB3YW50IHRvIHRyaWdnZXIgYSByZWJ1aWxkIG9uIGRlcGxveVxuICAgIGxldCBidWlsZEFyZ3MgPSAnJztcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCB2YWx1ZV0gb2YgdGhpcy5idWlsZEFyZ3MuZW50cmllcygpKSB7XG4gICAgICBidWlsZEFyZ3MgKz0gYCAtLWJ1aWxkLWFyZyBcIiR7bmFtZX1cIj1cIiR7dmFsdWV9XCJgO1xuICAgIH1cbiAgICBidWlsZEFyZ3MgKz0gYCAtLWJ1aWxkLWFyZyBSVU5ORVJfVkVSU0lPTj1cIiR7cnVubmVyVmVyc2lvbiA/IHJ1bm5lclZlcnNpb24udmVyc2lvbiA6IFJ1bm5lclZlcnNpb24ubGF0ZXN0KCkudmVyc2lvbn1cImA7XG5cbiAgICByZXR1cm4ge1xuICAgICAgdmVyc2lvbjogJzAuMicsXG4gICAgICBlbnY6IHtcbiAgICAgICAgdmFyaWFibGVzOiB7XG4gICAgICAgICAgUkVQT19BUk46IHJlcG9zaXRvcnkucmVwb3NpdG9yeUFybixcbiAgICAgICAgICBSRVBPX1VSSTogcmVwb3NpdG9yeS5yZXBvc2l0b3J5VXJpLFxuICAgICAgICAgIFNUQUNLX0lEOiAndW5zcGVjaWZpZWQnLFxuICAgICAgICAgIFJFUVVFU1RfSUQ6ICd1bnNwZWNpZmllZCcsXG4gICAgICAgICAgTE9HSUNBTF9SRVNPVVJDRV9JRDogJ3Vuc3BlY2lmaWVkJyxcbiAgICAgICAgICBSRVNQT05TRV9VUkw6ICd1bnNwZWNpZmllZCcsXG4gICAgICAgICAgUlVOTkVSX1ZFUlNJT046IHJ1bm5lclZlcnNpb24gPyBydW5uZXJWZXJzaW9uLnZlcnNpb24gOiBSdW5uZXJWZXJzaW9uLmxhdGVzdCgpLnZlcnNpb24sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcGhhc2VzOiB7XG4gICAgICAgIHByZV9idWlsZDoge1xuICAgICAgICAgIGNvbW1hbmRzOiB0aGlzLnByZUJ1aWxkLmNvbmNhdChbXG4gICAgICAgICAgICAnJChhd3MgZWNyIGdldC1sb2dpbiAtLW5vLWluY2x1ZGUtZW1haWwgLS1yZWdpb24gXCIkQVdTX0RFRkFVTFRfUkVHSU9OXCIpJyxcbiAgICAgICAgICBdKSxcbiAgICAgICAgfSxcbiAgICAgICAgYnVpbGQ6IHtcbiAgICAgICAgICBjb21tYW5kczogW1xuICAgICAgICAgICAgYGRvY2tlciBidWlsZCAuIC10IFwiJFJFUE9fVVJJXCIgJHtidWlsZEFyZ3N9YCxcbiAgICAgICAgICAgICdkb2NrZXIgcHVzaCBcIiRSRVBPX1VSSVwiJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICBwb3N0X2J1aWxkOiB7XG4gICAgICAgICAgY29tbWFuZHM6IHRoaXMucG9zdEJ1aWxkLmNvbmNhdChbXG4gICAgICAgICAgICAnU1RBVFVTPVwiU1VDQ0VTU1wiJyxcbiAgICAgICAgICAgICdESUdFU1Q9XCJVTktOT1dOXCInLFxuICAgICAgICAgICAgJ2lmIFsgJENPREVCVUlMRF9CVUlMRF9TVUNDRUVESU5HIC1uZSAxIF07IHRoZW4gU1RBVFVTPVwiRkFJTEVEXCI7IGVsc2UgRElHRVNUPWBkb2NrZXIgaW5zcGVjdCBcIiRSRVBPX1VSSVwiIHwganEgLXIgXFwnLlswXS5SZXBvRGlnZXN0c1swXSB8IHNwbGl0KFwiQFwiKVsxXSB8IHNwbGl0KFwiOlwiKVsxXVxcJ2A7IGZpJyxcbiAgICAgICAgICAgICdjYXQgPDxFT0YgPiAvdG1wL3BheWxvYWQuanNvblxcbicgK1xuICAgICAgICAgICAgJ3tcXG4nICtcbiAgICAgICAgICAgICcgIFwiU3RhY2tJZFwiOiBcIiRTVEFDS19JRFwiLFxcbicgK1xuICAgICAgICAgICAgJyAgXCJSZXF1ZXN0SWRcIjogXCIkUkVRVUVTVF9JRFwiLFxcbicgK1xuICAgICAgICAgICAgJyAgXCJMb2dpY2FsUmVzb3VyY2VJZFwiOiBcIiRMT0dJQ0FMX1JFU09VUkNFX0lEXCIsXFxuJyArXG4gICAgICAgICAgICAnICBcIlBoeXNpY2FsUmVzb3VyY2VJZFwiOiBcIiRSRVBPX0FSTlwiLFxcbicgK1xuICAgICAgICAgICAgJyAgXCJTdGF0dXNcIjogXCIkU1RBVFVTXCIsXFxuJyArXG4gICAgICAgICAgICBgICBcIlJlYXNvblwiOiBcIlNlZSBsb2dzIGluICR7bG9nR3JvdXAubG9nR3JvdXBOYW1lfS8kQ09ERUJVSUxEX0xPR19QQVRIIChkZXBsb3kgYWdhaW4gd2l0aCBcXCdjZGsgZGVwbG95IC1SXFwnIG9yIGxvZ1JlbW92YWxQb2xpY3k9UmVtb3ZhbFBvbGljeS5SRVRBSU4gaWYgdGhleSBhcmUgYWxyZWFkeSBkZWxldGVkKVwiLFxcbmAgK1xuICAgICAgICAgICAgJyAgXCJEYXRhXCI6IHtcIkRpZ2VzdFwiOiBcIiRESUdFU1RcIn1cXG4nICsgLy8gaW5jbHVkZSB0aGUgZGlnZXN0IHRvIG1hcmsgdGhlIHJlc291cmNlIHVwZGF0ZWQgc28gdGhlIHJ1bm5lciBwcm92aWRlcnMgZ2V0IHVwZGF0ZWQgd2l0aCB0aGUgbGF0ZXN0IGRpZ2VzdCB0b28gKHNwZWNpZmljYWxseSBMYW1iZGEpXG4gICAgICAgICAgICAnfVxcbicgK1xuICAgICAgICAgICAgJ0VPRicsXG4gICAgICAgICAgICAnaWYgWyBcIiRSRVNQT05TRV9VUkxcIiAhPSBcInVuc3BlY2lmaWVkXCIgXTsgdGhlbiBqcSAuIC90bXAvcGF5bG9hZC5qc29uOyBjdXJsIC1mc1NMIC1YIFBVVCAtSCBcIkNvbnRlbnQtVHlwZTpcIiAtZCBcIkAvdG1wL3BheWxvYWQuanNvblwiIFwiJFJFU1BPTlNFX1VSTFwiOyBmaScsXG4gICAgICAgICAgXSksXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGN1c3RvbVJlc291cmNlKHByb2plY3Q6IGNvZGVidWlsZC5Qcm9qZWN0KSB7XG4gICAgY29uc3QgY3JIYW5kbGVyID0gQnVuZGxlZE5vZGVqc0Z1bmN0aW9uLnNpbmdsZXRvbih0aGlzLCAnYnVpbGQtaW1hZ2UnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ0N1c3RvbSByZXNvdXJjZSBoYW5kbGVyIHRoYXQgdHJpZ2dlcnMgQ29kZUJ1aWxkIHRvIGJ1aWxkIHJ1bm5lciBpbWFnZXMsIGFuZCBjbGVhbnMtdXAgaW1hZ2VzIG9uIGRlbGV0aW9uJyxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHBvbGljeSA9IG5ldyBpYW0uUG9saWN5KHRoaXMsICdDUiBQb2xpY3knLCB7XG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2NvZGVidWlsZDpTdGFydEJ1aWxkJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbcHJvamVjdC5wcm9qZWN0QXJuXSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2VjcjpCYXRjaERlbGV0ZUltYWdlJywgJ2VjcjpMaXN0SW1hZ2VzJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5yZXBvc2l0b3J5LnJlcG9zaXRvcnlBcm5dLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG4gICAgY3JIYW5kbGVyLnJvbGU/LmF0dGFjaElubGluZVBvbGljeShwb2xpY3kpO1xuXG4gICAgY29uc3QgY3IgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0J1aWxkZXInLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNySGFuZGxlci5mdW5jdGlvbkFybixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6SW1hZ2VCdWlsZGVyJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVwb05hbWU6IHRoaXMucmVwb3NpdG9yeS5yZXBvc2l0b3J5TmFtZSxcbiAgICAgICAgUHJvamVjdE5hbWU6IHByb2plY3QucHJvamVjdE5hbWUsXG4gICAgICAgIC8vIFdlIGluY2x1ZGUgYSBoYXNoIHNvIHRoZSBpbWFnZSBpcyBidWlsdCBpbW1lZGlhdGVseSBvbiBjaGFuZ2VzLCBhbmQgd2UgZG9uJ3QgaGF2ZSB0byB3YWl0IGZvciBpdHMgc2NoZWR1bGVkIGJ1aWxkLlxuICAgICAgICAvLyBUaGlzIGFsc28gaGVscHMgbWFrZSBzdXJlIHRoZSBjaGFuZ2VzIGFyZSBnb29kLiBJZiB0aGV5IGhhdmUgYSBidWcsIHRoZSBkZXBsb3ltZW50IHdpbGwgZmFpbCBpbnN0ZWFkIG9mIGp1c3QgdGhlIHNjaGVkdWxlZCBidWlsZC5cbiAgICAgICAgQnVpbGRIYXNoOiB0aGlzLmhhc2hCdWlsZFNldHRpbmdzKCksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gYWRkIGRlcGVuZGVuY2llcyB0byBtYWtlIHN1cmUgcmVzb3VyY2VzIGFyZSB0aGVyZSB3aGVuIHdlIG5lZWQgdGhlbVxuICAgIGNyLm5vZGUuYWRkRGVwZW5kZW5jeShwcm9qZWN0KTtcbiAgICBjci5ub2RlLmFkZERlcGVuZGVuY3kocG9saWN5KTtcbiAgICBjci5ub2RlLmFkZERlcGVuZGVuY3koY3JIYW5kbGVyKTtcblxuICAgIHJldHVybiBjcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gaGFzaCBvZiBhbGwgc2V0dGluZ3MgdGhhdCBjYW4gYWZmZWN0IHRoZSByZXN1bHQgaW1hZ2Ugc28gd2UgY2FuIHRyaWdnZXIgdGhlIGJ1aWxkIHdoZW4gaXQgY2hhbmdlcy5cbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgaGFzaEJ1aWxkU2V0dGluZ3MoKTogc3RyaW5nIHtcbiAgICAvLyBtYWluIERvY2tlcmZpbGVcbiAgICBsZXQgY29tcG9uZW50czogc3RyaW5nW10gPSBbdGhpcy5kb2NrZXJmaWxlLmFzc2V0SGFzaF07XG4gICAgLy8gYWxsIGFkZGl0aW9uYWwgZmlsZXNcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBhc3NldF0gb2YgdGhpcy5zZWNvbmRhcnlBc3NldHMuZW50cmllcygpKSB7XG4gICAgICBjb21wb25lbnRzLnB1c2gobmFtZSk7XG4gICAgICBjb21wb25lbnRzLnB1c2goYXNzZXQuYXNzZXRIYXNoKTtcbiAgICB9XG4gICAgLy8gYnVpbGRzcGVjLnltbCB2ZXJzaW9uXG4gICAgY29tcG9uZW50cy5wdXNoKGB2JHtDb2RlQnVpbGRJbWFnZUJ1aWxkZXIuQlVJTERTUEVDX1ZFUlNJT059YCk7XG4gICAgLy8gdXNlciBjb21tYW5kc1xuICAgIGNvbXBvbmVudHMgPSBjb21wb25lbnRzLmNvbmNhdCh0aGlzLnByZUJ1aWxkKTtcbiAgICBjb21wb25lbnRzID0gY29tcG9uZW50cy5jb25jYXQodGhpcy5wb3N0QnVpbGQpO1xuICAgIGZvciAoY29uc3QgW25hbWUsIHZhbHVlXSBvZiB0aGlzLmJ1aWxkQXJncy5lbnRyaWVzKCkpIHtcbiAgICAgIGNvbXBvbmVudHMucHVzaChuYW1lKTtcbiAgICAgIGNvbXBvbmVudHMucHVzaCh2YWx1ZSk7XG4gICAgfVxuICAgIC8vIGhhc2ggaXRcbiAgICBjb25zdCBhbGwgPSBjb21wb25lbnRzLmpvaW4oJy0nKTtcbiAgICByZXR1cm4gY3J5cHRvLmNyZWF0ZUhhc2goJ21kNScpLnVwZGF0ZShhbGwpLmRpZ2VzdCgnaGV4Jyk7XG4gIH1cblxuICBwcml2YXRlIHJlYnVpbGRJbWFnZU9uU2NoZWR1bGUocHJvamVjdDogY29kZWJ1aWxkLlByb2plY3QsIHJlYnVpbGRJbnRlcnZhbD86IER1cmF0aW9uKSB7XG4gICAgcmVidWlsZEludGVydmFsID0gcmVidWlsZEludGVydmFsID8/IER1cmF0aW9uLmRheXMoNyk7XG4gICAgaWYgKHJlYnVpbGRJbnRlcnZhbC50b01pbGxpc2Vjb25kcygpICE9IDApIHtcbiAgICAgIGNvbnN0IHNjaGVkdWxlUnVsZSA9IG5ldyBldmVudHMuUnVsZSh0aGlzLCAnQnVpbGQgU2NoZWR1bGUnLCB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiBgUmVidWlsZCBydW5uZXIgaW1hZ2UgZm9yICR7dGhpcy5yZXBvc2l0b3J5LnJlcG9zaXRvcnlOYW1lfWAsXG4gICAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShyZWJ1aWxkSW50ZXJ2YWwpLFxuICAgICAgfSk7XG4gICAgICBzY2hlZHVsZVJ1bGUuYWRkVGFyZ2V0KG5ldyBldmVudHNfdGFyZ2V0cy5Db2RlQnVpbGRQcm9qZWN0KHByb2plY3QpKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==