"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        const defaultProvider = this.getDefaultProvider();
        if (!defaultProvider) {
            throw new Error(`No provider was found for the default label "${this.props.defaultProviderLabel}"`);
        }
        else {
            this.defaultProvider = defaultProvider;
        }
        this.orchestrator = this.stateMachine();
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.statusFunctions();
    }
    getDefaultProvider() {
        for (const provider of this.providers) {
            if ((this.props.defaultProviderLabel || 'codebuild') == provider.label) {
                return provider;
            }
        }
        return null;
    }
    stateMachine() {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: this.deleteRunner(),
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${provider.label}`), providerTask);
            if (this.defaultProvider == provider) {
                providerChooser.otherwise(providerTask);
            }
        }
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        return new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunctions() {
        const func = this.providers.map(provider => {
            return {
                type: provider.constructor.name,
                label: provider.label,
                vpcArn: provider.vpc && provider.vpc.vpcArn,
                securityGroup: provider.securityGroup && provider.securityGroup.securityGroupId,
                roleArn: provider.grantPrincipal.grantPrincipal.roleArn,
            };
        });
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                PROVIDERS: JSON.stringify(func),
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
            },
            timeout: cdk.Duration.minutes(3),
        });
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${cdk.Stack.of(this).region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.0.8" };
//# sourceMappingURL=data:application/json;base64,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