"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'Can create a scheduled Fargate Task - with only required props'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        new lib_1.ScheduledFargateTask(stack, 'ScheduledFargateTask', {
            cluster,
            scheduledFargateTaskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('henk'),
                memoryLimitMiB: 512,
            },
            schedule: events.Schedule.expression('rate(1 minute)')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                    EcsParameters: {
                        LaunchType: 'FARGATE',
                        NetworkConfiguration: {
                            AwsVpcConfiguration: {
                                AssignPublicIp: 'DISABLED',
                                SecurityGroups: [
                                    {
                                        'Fn::GetAtt': [
                                            'ScheduledFargateTaskScheduledTaskDefSecurityGroupE075BC19',
                                            'GroupId'
                                        ]
                                    }
                                ],
                                Subnets: [
                                    {
                                        Ref: 'VpcPrivateSubnet1Subnet536B997A'
                                    }
                                ]
                            }
                        },
                        TaskCount: 1,
                        TaskDefinitionArn: { Ref: 'ScheduledFargateTaskScheduledTaskDef521FA675' }
                    },
                    Id: 'Target0',
                    Input: '{}',
                    RoleArn: { 'Fn::GetAtt': ['ScheduledFargateTaskScheduledTaskDefEventsRole6CE19522', 'Arn'] }
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Essential: true,
                    Image: 'henk',
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                Ref: 'ScheduledFargateTaskScheduledTaskDefScheduledContainerLogGroup4134B16C'
                            },
                            'awslogs-stream-prefix': 'ScheduledFargateTask',
                            'awslogs-region': {
                                Ref: 'AWS::Region'
                            }
                        }
                    },
                    Name: 'ScheduledContainer'
                }
            ]
        }));
        test.done();
    },
    'Can create a scheduled Fargate Task - with optional props'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        new lib_1.ScheduledFargateTask(stack, 'ScheduledFargateTask', {
            cluster,
            scheduledFargateTaskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('henk'),
                memoryLimitMiB: 512,
                cpu: 2,
                environment: { TRIGGER: 'CloudWatch Events' },
            },
            desiredTaskCount: 2,
            schedule: events.Schedule.expression('rate(1 minute)')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                    EcsParameters: {
                        LaunchType: 'FARGATE',
                        NetworkConfiguration: {
                            AwsVpcConfiguration: {
                                AssignPublicIp: 'DISABLED',
                                SecurityGroups: [
                                    {
                                        'Fn::GetAtt': [
                                            'ScheduledFargateTaskScheduledTaskDefSecurityGroupE075BC19',
                                            'GroupId'
                                        ]
                                    }
                                ],
                                Subnets: [
                                    {
                                        Ref: 'VpcPrivateSubnet1Subnet536B997A'
                                    }
                                ]
                            }
                        },
                        TaskCount: 2,
                        TaskDefinitionArn: { Ref: 'ScheduledFargateTaskScheduledTaskDef521FA675' }
                    },
                    Id: 'Target0',
                    Input: '{}',
                    RoleArn: { 'Fn::GetAtt': ['ScheduledFargateTaskScheduledTaskDefEventsRole6CE19522', 'Arn'] }
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TRIGGER',
                            Value: 'CloudWatch Events'
                        }
                    ],
                    Essential: true,
                    Image: 'henk',
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                Ref: 'ScheduledFargateTaskScheduledTaskDefScheduledContainerLogGroup4134B16C'
                            },
                            'awslogs-stream-prefix': 'ScheduledFargateTask',
                            'awslogs-region': {
                                Ref: 'AWS::Region'
                            }
                        }
                    },
                    Name: 'ScheduledContainer'
                }
            ]
        }));
        test.done();
    },
    'Scheduled Fargate Task - with MemoryReservation defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        new lib_1.ScheduledFargateTask(stack, 'ScheduledFargateTask', {
            cluster,
            scheduledFargateTaskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('henk'),
            },
            schedule: events.Schedule.expression('rate(1 minute)')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Essential: true,
                    Image: 'henk',
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                Ref: 'ScheduledFargateTaskScheduledTaskDefScheduledContainerLogGroup4134B16C'
                            },
                            'awslogs-stream-prefix': 'ScheduledFargateTask',
                            'awslogs-region': {
                                Ref: 'AWS::Region'
                            }
                        }
                    },
                    Name: 'ScheduledContainer'
                }
            ]
        }));
        test.done();
    },
    'Scheduled Fargate Task - with Command defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        new lib_1.ScheduledFargateTask(stack, 'ScheduledFargateTask', {
            cluster,
            scheduledFargateTaskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('henk'),
                command: ['-c', '4', 'amazon.com'],
            },
            schedule: events.Schedule.expression('rate(1 minute)')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Command: [
                        '-c',
                        '4',
                        'amazon.com'
                    ],
                    Essential: true,
                    Image: 'henk',
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                Ref: 'ScheduledFargateTaskScheduledTaskDefScheduledContainerLogGroup4134B16C'
                            },
                            'awslogs-stream-prefix': 'ScheduledFargateTask',
                            'awslogs-region': {
                                Ref: 'AWS::Region'
                            }
                        }
                    },
                    Name: 'ScheduledContainer'
                }
            ]
        }));
        test.done();
    },
    'Scheduled Fargate Task - with subnetSelection defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc', {
            maxAzs: 1,
            subnetConfiguration: [
                { name: 'Public', cidrMask: 28, subnetType: ec2.SubnetType.PUBLIC }
            ],
        });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        new lib_1.ScheduledFargateTask(stack, 'ScheduledFargateTask', {
            cluster,
            scheduledFargateTaskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('henk'),
            },
            subnetSelection: { subnetType: ec2.SubnetType.PUBLIC },
            schedule: events.Schedule.expression('rate(1 minute)')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            Targets: [
                {
                    EcsParameters: {
                        NetworkConfiguration: {
                            AwsVpcConfiguration: {
                                AssignPublicIp: 'ENABLED',
                                Subnets: [
                                    {
                                        Ref: 'VpcPublicSubnet1Subnet5C2D37C4'
                                    }
                                ]
                            }
                        },
                    }
                }
            ]
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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