"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const ecsPatterns = require("../../lib");
module.exports = {
    'setting loadBalancerType to Network creates an NLB Public'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'network',
            Scheme: 'internet-facing'
        }));
        test.done();
    },
    'setting loadBalancerType to Network and publicLoadBalancer to false creates an NLB Private'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app')
            },
            publicLoadBalancer: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'network',
            Scheme: 'internal'
        }));
        test.done();
    },
    'setting vpc and cluster throws error'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        test.throws(() => new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            vpc,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app')
            },
        }));
        test.done();
    },
    'setting executionRole updated taskDefinition with given execution role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const executionRole = new iam.Role(stack, 'ExecutionRole', {
            path: '/',
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com'))
        });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                executionRole
            },
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.ServiceTaskDef1922A00F;
        test.deepEqual(serviceTaskDefinition.Properties.ExecutionRoleArn, { 'Fn::GetAtt': ['ExecutionRole605A040B', 'Arn'] });
        test.done();
    },
    'setting taskRole updated taskDefinition with given task role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const taskRole = new iam.Role(stack, 'taskRoleTest', {
            path: '/',
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com'))
        });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                taskRole
            },
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.ServiceTaskDef1922A00F;
        test.deepEqual(serviceTaskDefinition.Properties.TaskRoleArn, { 'Fn::GetAtt': ['taskRoleTest9DA66B6E', 'Arn'] });
        test.done();
    },
    'setting containerName updates container name with given name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                containerName: 'bob'
            },
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.ServiceTaskDef1922A00F;
        test.deepEqual(serviceTaskDefinition.Properties.ContainerDefinitions[0].Name, 'bob');
        test.done();
    },
    'not setting containerName updates container name with default'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
            },
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.ServiceTaskDef1922A00F;
        test.deepEqual(serviceTaskDefinition.Properties.ContainerDefinitions[0].Name, 'web');
        test.done();
    },
    'setting servicename updates service name with given name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
            },
            serviceName: 'bob',
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.Service9571FDD8;
        test.deepEqual(serviceTaskDefinition.Properties.ServiceName, 'bob');
        test.done();
    },
    'not setting servicename updates service name with default'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
            },
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.Service9571FDD8;
        test.equal(serviceTaskDefinition.Properties.ServiceName, undefined);
        test.done();
    },
    'setting healthCheckGracePeriod works'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
            },
            healthCheckGracePeriod: cdk.Duration.seconds(600),
        });
        // THEN
        const serviceTaskDefinition = assert_1.SynthUtils.synthesize(stack).template.Resources.Service9571FDD8;
        test.deepEqual(serviceTaskDefinition.Properties.HealthCheckGracePeriodSeconds, 600);
        test.done();
    },
    'test load balanced service with family defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                enableLogging: false,
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value'
                },
                family: 'fargate-task-family',
            },
            desiredCount: 2,
            memoryLimitMiB: 512,
            serviceName: 'fargate-test-service',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            DesiredCount: 2,
            LaunchType: 'FARGATE',
            ServiceName: 'fargate-test-service'
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value'
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value'
                        }
                    ],
                    Image: '/aws/aws-example-app',
                }
            ],
            Family: 'fargate-task-family'
        }));
        test.done();
    },
    'setting NLB special listener port to create the listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'FargateNlbService', {
            cluster,
            listenerPort: 2015,
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    Type: 'forward'
                }
            ],
            Port: 2015,
            Protocol: 'TCP'
        }));
        test.done();
    },
    'setting ALB special listener port to create the listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'FargateAlbService', {
            cluster,
            listenerPort: 2015,
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    Type: 'forward'
                }
            ],
            Port: 2015,
            Protocol: 'HTTP'
        }));
        test.done();
    },
    'setting ALB HTTPS protocol to create the listener on 443'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'FargateAlbService', {
            cluster,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
            domainName: 'domain.com',
            domainZone: {
                hostedZoneId: 'fakeId',
                zoneName: 'domain.com',
                hostedZoneArn: 'arn:aws:route53:::hostedzone/fakeId',
                stack,
                node: stack.node,
            },
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    Type: 'forward'
                }
            ],
            Port: 443,
            Protocol: 'HTTPS'
        }));
        test.done();
    },
    'setting ALB HTTPS correctly sets the recordset name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'FargateAlbService', {
            cluster,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
            domainName: 'test.domain.com',
            domainZone: {
                hostedZoneId: 'fakeId',
                zoneName: 'domain.com.',
                hostedZoneArn: 'arn:aws:route53:::hostedzone/fakeId',
                stack,
                node: stack.node,
            },
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: 'test.domain.com.',
        }));
        test.done();
    },
    'setting ALB HTTP protocol to create the listener on 80'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'FargateAlbService', {
            cluster,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    Type: 'forward'
                }
            ],
            Port: 80,
            Protocol: 'HTTP'
        }));
        test.done();
    },
    'setting ALB without any protocol or listenerPort to create the listener on 80'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'FargateAlbService', {
            cluster,
            taskImageOptions: {
                containerPort: 2015,
                image: ecs.ContainerImage.fromRegistry('abiosoft/caddy')
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    Type: 'forward'
                }
            ],
            Port: 80,
            Protocol: 'HTTP'
        }));
        test.done();
    },
    'passing in existing network load balancer to NLB Fargate Service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const nlb = new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(stack, 'NLB', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            vpc,
            loadBalancer: nlb,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'FARGATE',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'network'
        }));
        test.done();
    },
    'passing in imported network load balancer and resources to NLB Fargate service'(test) {
        // GIVEN
        const stack1 = new cdk.Stack();
        const vpc1 = new ec2.Vpc(stack1, 'VPC');
        const cluster1 = new ecs.Cluster(stack1, 'Cluster', { vpc: vpc1 });
        const nlbArn = 'arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer';
        const stack2 = new cdk.Stack(stack1, 'Stack2');
        const cluster2 = ecs.Cluster.fromClusterAttributes(stack2, 'ImportedCluster', {
            vpc: vpc1,
            securityGroups: cluster1.connections.securityGroups,
            clusterName: 'cluster-name'
        });
        // WHEN
        const nlb2 = aws_elasticloadbalancingv2_1.NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(stack2, 'ImportedNLB', {
            loadBalancerArn: nlbArn,
            vpc: vpc1,
        });
        const taskDef = new ecs.FargateTaskDefinition(stack2, 'TaskDef', {
            cpu: 1024,
            memoryLimitMiB: 1024,
        });
        const container = taskDef.addContainer('myContainer', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024
        });
        container.addPortMappings({
            containerPort: 80,
        });
        new ecsPatterns.NetworkLoadBalancedFargateService(stack2, 'FargateNLBService', {
            cluster: cluster2,
            loadBalancer: nlb2,
            desiredCount: 1,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'FARGATE',
            LoadBalancers: [{ ContainerName: 'myContainer', ContainerPort: 80 }]
        }));
        assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup'));
        assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            LoadBalancerArn: nlb2.loadBalancerArn,
            Port: 80,
        }));
        test.done();
    },
    'passing in previously created application load balancer to ALB Fargate Service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc, clusterName: 'MyCluster' });
        const sg = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc });
        cluster.connections.addSecurityGroup(sg);
        const alb = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(stack, 'ALB', { vpc, securityGroup: sg });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            loadBalancer: alb,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'FARGATE',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'application'
        }));
        test.done();
    },
    'passing in imported application load balancer and resources to ALB Fargate Service'(test) {
        // GIVEN
        const stack1 = new cdk.Stack();
        const albArn = 'arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer';
        const vpc = new ec2.Vpc(stack1, 'Vpc');
        const cluster = new ecs.Cluster(stack1, 'Cluster', { vpc, clusterName: 'MyClusterName', });
        const sg = new ec2.SecurityGroup(stack1, 'SecurityGroup', { vpc });
        cluster.connections.addSecurityGroup(sg);
        const alb = aws_elasticloadbalancingv2_1.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(stack1, 'ALB', {
            loadBalancerArn: albArn,
            vpc,
            securityGroupId: sg.securityGroupId,
            loadBalancerDnsName: 'MyDnsName'
        });
        // WHEN
        const taskDef = new ecs.FargateTaskDefinition(stack1, 'TaskDef', {
            cpu: 1024,
            memoryLimitMiB: 1024,
        });
        const container = taskDef.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024,
        });
        container.addPortMappings({
            containerPort: 80,
        });
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack1, 'FargateALBService', {
            cluster,
            loadBalancer: alb,
            desiredCount: 1,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack1).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'FARGATE',
            LoadBalancers: [{ ContainerName: 'Container', ContainerPort: 80 }]
        }));
        assert_1.expect(stack1).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup'));
        assert_1.expect(stack1).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            LoadBalancerArn: alb.loadBalancerArn,
            Port: 80,
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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