"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'When Application Load Balancer': {
        'test Fargate loadbalanced construct with default settings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // WHEN
            new lib_1.ApplicationMultipleTargetGroupsFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: aws_ecs_1.ContainerImage.fromRegistry('test')
                }
            });
            // THEN - stack contains a load balancer and a service
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                DesiredCount: 1,
                LaunchType: 'FARGATE',
                LoadBalancers: [
                    {
                        ContainerName: 'web',
                        ContainerPort: 80,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSGroup0CC8688C'
                        }
                    }
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'ServiceTaskDefwebLogGroup2A898F61'
                                },
                                'awslogs-stream-prefix': 'Service',
                                'awslogs-region': {
                                    Ref: 'AWS::Region'
                                }
                            }
                        },
                        Name: 'web',
                        PortMappings: [
                            {
                                ContainerPort: 80,
                                Protocol: 'tcp'
                            }
                        ]
                    }
                ],
                Cpu: '256',
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ServiceTaskDefExecutionRole919F7BE3',
                        'Arn'
                    ]
                },
                Family: 'ServiceTaskDef79D79521',
                Memory: '512',
                NetworkMode: 'awsvpc',
                RequiresCompatibilities: [
                    'FARGATE'
                ]
            }));
            test.done();
        },
        'test Fargate loadbalanced construct with all settings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // WHEN
            new lib_1.ApplicationMultipleTargetGroupsFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: aws_ecs_1.ContainerImage.fromRegistry('test'),
                    containerName: 'hello',
                    containerPorts: [80, 90],
                    enableLogging: false,
                    environment: {
                        TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                        TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value'
                    },
                    logDriver: new aws_ecs_1.AwsLogDriver({
                        streamPrefix: 'TestStream'
                    }),
                    family: 'Ec2TaskDef',
                    executionRole: new aws_iam_1.Role(stack, 'ExecutionRole', {
                        path: '/',
                        assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('ecs.amazonaws.com'), new aws_iam_1.ServicePrincipal('ecs-tasks.amazonaws.com'))
                    }),
                    taskRole: new aws_iam_1.Role(stack, 'TaskRole', {
                        assumedBy: new aws_iam_1.ServicePrincipal('ecs-tasks.amazonaws.com'),
                    })
                },
                cpu: 256,
                assignPublicIp: true,
                memoryLimitMiB: 512,
                desiredCount: 3,
                enableECSManagedTags: true,
                healthCheckGracePeriod: core_1.Duration.millis(2000),
                propagateTags: aws_ecs_1.PropagatedTagSource.SERVICE,
                serviceName: 'myService',
                targetGroups: [
                    {
                        containerPort: 80,
                    },
                    {
                        containerPort: 90,
                        pathPattern: 'a/b/c',
                        priority: 10,
                        protocol: aws_ecs_1.Protocol.TCP
                    }
                ]
            });
            // THEN - stack contains a load balancer and a service
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                DesiredCount: 3,
                EnableECSManagedTags: true,
                HealthCheckGracePeriodSeconds: 2,
                LaunchType: 'FARGATE',
                LoadBalancers: [
                    {
                        ContainerName: 'hello',
                        ContainerPort: 80,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSTargetGrouphello80Group233A4D54'
                        }
                    },
                    {
                        ContainerName: 'hello',
                        ContainerPort: 90,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSTargetGrouphello90GroupE58E4EAB'
                        }
                    }
                ],
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'ServiceSecurityGroupEEA09B68',
                                    'GroupId'
                                ]
                            }
                        ],
                        Subnets: [
                            {
                                Ref: 'VPCPublicSubnet1SubnetB4246D30'
                            },
                            {
                                Ref: 'VPCPublicSubnet2Subnet74179F39'
                            }
                        ]
                    }
                },
                PropagateTags: 'SERVICE',
                ServiceName: 'myService'
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: [
                            {
                                Name: 'TEST_ENVIRONMENT_VARIABLE1',
                                Value: 'test environment variable 1 value'
                            },
                            {
                                Name: 'TEST_ENVIRONMENT_VARIABLE2',
                                Value: 'test environment variable 2 value'
                            }
                        ],
                        Essential: true,
                        Image: 'test',
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'ServiceTaskDefhelloLogGroup44519781'
                                },
                                'awslogs-stream-prefix': 'TestStream',
                                'awslogs-region': {
                                    Ref: 'AWS::Region'
                                }
                            }
                        },
                        Name: 'hello',
                        PortMappings: [
                            {
                                ContainerPort: 80,
                                Protocol: 'tcp'
                            },
                            {
                                ContainerPort: 90,
                                Protocol: 'tcp'
                            }
                        ]
                    }
                ],
                Cpu: '256',
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ExecutionRole605A040B',
                        'Arn'
                    ]
                },
                Family: 'Ec2TaskDef',
                Memory: '512',
                NetworkMode: 'awsvpc',
                RequiresCompatibilities: [
                    'FARGATE'
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        'TaskRole30FC0FBB',
                        'Arn'
                    ]
                }
            }));
            test.done();
        },
        'errors if no essential container in pre-defined task definition'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            const taskDefinition = new aws_ecs_1.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                new lib_1.ApplicationMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
            }, /At least one essential container must be specified/);
            test.done();
        },
        'errors when setting both taskDefinition and taskImageOptions'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            const taskDefinition = new aws_ecs_1.FargateTaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            test.throws(() => {
                new lib_1.ApplicationMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster,
                    taskImageOptions: {
                        image: aws_ecs_1.ContainerImage.fromRegistry('test'),
                    },
                    taskDefinition
                });
            }, /You must specify only one of TaskDefinition or TaskImageOptions./);
            test.done();
        },
        'errors when setting neither taskDefinition nor taskImageOptions'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // THEN
            test.throws(() => {
                new lib_1.ApplicationMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster
                });
            }, /You must specify one of: taskDefinition or image/);
            test.done();
        }
    },
    'When Network Load Balancer': {
        'test Fargate loadbalanced construct with default settings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // WHEN
            new lib_1.NetworkMultipleTargetGroupsFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: aws_ecs_1.ContainerImage.fromRegistry('test')
                }
            });
            // THEN - stack contains a load balancer and a service
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                DesiredCount: 1,
                LaunchType: 'FARGATE',
                LoadBalancers: [
                    {
                        ContainerName: 'web',
                        ContainerPort: 80,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSGroup0CC8688C'
                        }
                    }
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'ServiceTaskDefwebLogGroup2A898F61'
                                },
                                'awslogs-stream-prefix': 'Service',
                                'awslogs-region': {
                                    Ref: 'AWS::Region'
                                }
                            }
                        },
                        Name: 'web',
                        PortMappings: [
                            {
                                ContainerPort: 80,
                                Protocol: 'tcp'
                            }
                        ]
                    }
                ],
                Cpu: '256',
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ServiceTaskDefExecutionRole919F7BE3',
                        'Arn'
                    ]
                },
                Family: 'ServiceTaskDef79D79521',
                Memory: '512',
                NetworkMode: 'awsvpc',
                RequiresCompatibilities: [
                    'FARGATE'
                ]
            }));
            test.done();
        },
        'test Fargate loadbalanced construct with all settings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // WHEN
            new lib_1.NetworkMultipleTargetGroupsFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: aws_ecs_1.ContainerImage.fromRegistry('test'),
                    containerName: 'hello',
                    containerPorts: [80, 90],
                    enableLogging: false,
                    environment: {
                        TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                        TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value'
                    },
                    logDriver: new aws_ecs_1.AwsLogDriver({
                        streamPrefix: 'TestStream'
                    }),
                    family: 'Ec2TaskDef',
                    executionRole: new aws_iam_1.Role(stack, 'ExecutionRole', {
                        path: '/',
                        assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('ecs.amazonaws.com'), new aws_iam_1.ServicePrincipal('ecs-tasks.amazonaws.com'))
                    }),
                    taskRole: new aws_iam_1.Role(stack, 'TaskRole', {
                        assumedBy: new aws_iam_1.ServicePrincipal('ecs-tasks.amazonaws.com'),
                    })
                },
                cpu: 256,
                assignPublicIp: true,
                memoryLimitMiB: 512,
                desiredCount: 3,
                enableECSManagedTags: true,
                healthCheckGracePeriod: core_1.Duration.millis(2000),
                propagateTags: aws_ecs_1.PropagatedTagSource.SERVICE,
                serviceName: 'myService',
                targetGroups: [
                    {
                        containerPort: 80,
                    },
                    {
                        containerPort: 90
                    }
                ]
            });
            // THEN - stack contains a load balancer and a service
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                DesiredCount: 3,
                EnableECSManagedTags: true,
                HealthCheckGracePeriodSeconds: 2,
                LaunchType: 'FARGATE',
                LoadBalancers: [
                    {
                        ContainerName: 'hello',
                        ContainerPort: 80,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSTargetGrouphello80Group233A4D54'
                        }
                    },
                    {
                        ContainerName: 'hello',
                        ContainerPort: 90,
                        TargetGroupArn: {
                            Ref: 'ServiceLBPublicListenerECSTargetGrouphello90GroupE58E4EAB'
                        }
                    }
                ],
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'ServiceSecurityGroupEEA09B68',
                                    'GroupId'
                                ]
                            }
                        ],
                        Subnets: [
                            {
                                Ref: 'VPCPublicSubnet1SubnetB4246D30'
                            },
                            {
                                Ref: 'VPCPublicSubnet2Subnet74179F39'
                            }
                        ]
                    }
                },
                PropagateTags: 'SERVICE',
                ServiceName: 'myService'
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: [
                            {
                                Name: 'TEST_ENVIRONMENT_VARIABLE1',
                                Value: 'test environment variable 1 value'
                            },
                            {
                                Name: 'TEST_ENVIRONMENT_VARIABLE2',
                                Value: 'test environment variable 2 value'
                            }
                        ],
                        Essential: true,
                        Image: 'test',
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'ServiceTaskDefhelloLogGroup44519781'
                                },
                                'awslogs-stream-prefix': 'TestStream',
                                'awslogs-region': {
                                    Ref: 'AWS::Region'
                                }
                            }
                        },
                        Name: 'hello',
                        PortMappings: [
                            {
                                ContainerPort: 80,
                                Protocol: 'tcp'
                            },
                            {
                                ContainerPort: 90,
                                Protocol: 'tcp'
                            }
                        ]
                    }
                ],
                Cpu: '256',
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ExecutionRole605A040B',
                        'Arn'
                    ]
                },
                Family: 'Ec2TaskDef',
                Memory: '512',
                NetworkMode: 'awsvpc',
                RequiresCompatibilities: [
                    'FARGATE'
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        'TaskRole30FC0FBB',
                        'Arn'
                    ]
                }
            }));
            test.done();
        },
        'errors if no essential container in pre-defined task definition'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            const taskDefinition = new aws_ecs_1.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                new lib_1.NetworkMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
            }, /At least one essential container must be specified/);
            test.done();
        },
        'errors when setting both taskDefinition and taskImageOptions'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            const taskDefinition = new aws_ecs_1.FargateTaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            test.throws(() => {
                new lib_1.NetworkMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster,
                    taskImageOptions: {
                        image: aws_ecs_1.ContainerImage.fromRegistry('test'),
                    },
                    taskDefinition
                });
            }, /You must specify only one of TaskDefinition or TaskImageOptions./);
            test.done();
        },
        'errors when setting neither taskDefinition nor taskImageOptions'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
            const cluster = new aws_ecs_1.Cluster(stack, 'Cluster', { vpc });
            // THEN
            test.throws(() => {
                new lib_1.NetworkMultipleTargetGroupsFargateService(stack, 'Service', {
                    cluster
                });
            }, /You must specify one of: taskDefinition or image/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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