"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const network_multiple_target_groups_service_base_1 = require("../base/network-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkMultipleTargetGroupsFargateService extends network_multiple_target_groups_service_base_1.NetworkMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
        });
    }
}
exports.NetworkMultipleTargetGroupsFargateService = NetworkMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay1tdWx0aXBsZS10YXJnZXQtZ3JvdXBzLWZhcmdhdGUtc2VydmljZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm5ldHdvcmstbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1mYXJnYXRlLXNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw4Q0FBeUU7QUFHekUscUhBQzJHO0FBbUUzRzs7R0FFRztBQUNILE1BQWEseUNBQTBDLFNBQVEsb0ZBQXNDO0lBc0JuRzs7T0FFRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBd0QsRUFBRTtRQUNsRyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7UUFFeEYsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7U0FDckY7YUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDL0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1NBQzVDO2FBQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDakMsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7WUFDaEQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLCtCQUFxQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7Z0JBQy9ELGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDcEMsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLGFBQWEsRUFBRSxnQkFBZ0IsQ0FBQyxhQUFhO2dCQUM3QyxRQUFRLEVBQUUsZ0JBQWdCLENBQUMsUUFBUTtnQkFDbkMsTUFBTSxFQUFFLGdCQUFnQixDQUFDLE1BQU07YUFDaEMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxhQUFhLEdBQUcsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7WUFDNUcsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFO2dCQUNoRSxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsS0FBSztnQkFDN0IsT0FBTyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN2QixXQUFXLEVBQUUsZ0JBQWdCLENBQUMsV0FBVztnQkFDekMsT0FBTyxFQUFFLGdCQUFnQixDQUFDLE9BQU87YUFDbEMsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUU7Z0JBQ25DLEtBQUssTUFBTSxhQUFhLElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFO29CQUMzRCxTQUFTLENBQUMsZUFBZSxDQUFDO3dCQUN4QixhQUFhO3FCQUNkLENBQUMsQ0FBQztpQkFDSjthQUNGO1NBQ0Y7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztTQUN2RTtRQUNELElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNsRSxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixDQUFDLGVBQWUsQ0FBQztnQkFDbkQsYUFBYSxFQUFFLEVBQUU7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoRCxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdEIsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ3hGLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDcEg7YUFBTTtZQUNMLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYTthQUN6RSxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxLQUFxRDtRQUNoRixPQUFPLElBQUksd0JBQWMsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3pDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO1lBQ25DLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztZQUNuQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDbEMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUNoRCxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWU7U0FDdkMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBL0ZELDhGQStGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEZhcmdhdGVTZXJ2aWNlLCBGYXJnYXRlVGFza0RlZmluaXRpb24gfSBmcm9tICdAYXdzLWNkay9hd3MtZWNzJztcbmltcG9ydCB7IE5ldHdvcmtUYXJnZXRHcm91cCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2UsXG4gIE5ldHdvcmtNdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlUHJvcHMgfSBmcm9tICcuLi9iYXNlL25ldHdvcmstbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1zZXJ2aWNlLWJhc2UnO1xuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciB0aGUgTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2Ugc2VydmljZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrTXVsdGlwbGVUYXJnZXRHcm91cHNGYXJnYXRlU2VydmljZVByb3BzIGV4dGVuZHMgTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2VQcm9wcyB7XG5cbiAgLyoqXG4gICAqIFRoZSB0YXNrIGRlZmluaXRpb24gdG8gdXNlIGZvciB0YXNrcyBpbiB0aGUgc2VydmljZS4gT25seSBvbmUgb2YgVGFza0RlZmluaXRpb24gb3IgVGFza0ltYWdlT3B0aW9ucyBtdXN0IGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSB0YXNrRGVmaW5pdGlvbj86IEZhcmdhdGVUYXNrRGVmaW5pdGlvbjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBjcHUgdW5pdHMgdXNlZCBieSB0aGUgdGFzay5cbiAgICpcbiAgICogVmFsaWQgdmFsdWVzLCB3aGljaCBkZXRlcm1pbmVzIHlvdXIgcmFuZ2Ugb2YgdmFsaWQgdmFsdWVzIGZvciB0aGUgbWVtb3J5IHBhcmFtZXRlcjpcbiAgICpcbiAgICogMjU2ICguMjUgdkNQVSkgLSBBdmFpbGFibGUgbWVtb3J5IHZhbHVlczogMC41R0IsIDFHQiwgMkdCXG4gICAqXG4gICAqIDUxMiAoLjUgdkNQVSkgLSBBdmFpbGFibGUgbWVtb3J5IHZhbHVlczogMUdCLCAyR0IsIDNHQiwgNEdCXG4gICAqXG4gICAqIDEwMjQgKDEgdkNQVSkgLSBBdmFpbGFibGUgbWVtb3J5IHZhbHVlczogMkdCLCAzR0IsIDRHQiwgNUdCLCA2R0IsIDdHQiwgOEdCXG4gICAqXG4gICAqIDIwNDggKDIgdkNQVSkgLSBBdmFpbGFibGUgbWVtb3J5IHZhbHVlczogQmV0d2VlbiA0R0IgYW5kIDE2R0IgaW4gMUdCIGluY3JlbWVudHNcbiAgICpcbiAgICogNDA5NiAoNCB2Q1BVKSAtIEF2YWlsYWJsZSBtZW1vcnkgdmFsdWVzOiBCZXR3ZWVuIDhHQiBhbmQgMzBHQiBpbiAxR0IgaW5jcmVtZW50c1xuICAgKlxuICAgKiBUaGlzIGRlZmF1bHQgaXMgc2V0IGluIHRoZSB1bmRlcmx5aW5nIEZhcmdhdGVUYXNrRGVmaW5pdGlvbiBjb25zdHJ1Y3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IDI1NlxuICAgKi9cbiAgcmVhZG9ubHkgY3B1PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgYW1vdW50IChpbiBNaUIpIG9mIG1lbW9yeSB1c2VkIGJ5IHRoZSB0YXNrLlxuICAgKlxuICAgKiBUaGlzIGZpZWxkIGlzIHJlcXVpcmVkIGFuZCB5b3UgbXVzdCB1c2Ugb25lIG9mIHRoZSBmb2xsb3dpbmcgdmFsdWVzLCB3aGljaCBkZXRlcm1pbmVzIHlvdXIgcmFuZ2Ugb2YgdmFsaWQgdmFsdWVzXG4gICAqIGZvciB0aGUgY3B1IHBhcmFtZXRlcjpcbiAgICpcbiAgICogNTEyICgwLjUgR0IpLCAxMDI0ICgxIEdCKSwgMjA0OCAoMiBHQikgLSBBdmFpbGFibGUgY3B1IHZhbHVlczogMjU2ICguMjUgdkNQVSlcbiAgICpcbiAgICogMTAyNCAoMSBHQiksIDIwNDggKDIgR0IpLCAzMDcyICgzIEdCKSwgNDA5NiAoNCBHQikgLSBBdmFpbGFibGUgY3B1IHZhbHVlczogNTEyICguNSB2Q1BVKVxuICAgKlxuICAgKiAyMDQ4ICgyIEdCKSwgMzA3MiAoMyBHQiksIDQwOTYgKDQgR0IpLCA1MTIwICg1IEdCKSwgNjE0NCAoNiBHQiksIDcxNjggKDcgR0IpLCA4MTkyICg4IEdCKSAtIEF2YWlsYWJsZSBjcHUgdmFsdWVzOiAxMDI0ICgxIHZDUFUpXG4gICAqXG4gICAqIEJldHdlZW4gNDA5NiAoNCBHQikgYW5kIDE2Mzg0ICgxNiBHQikgaW4gaW5jcmVtZW50cyBvZiAxMDI0ICgxIEdCKSAtIEF2YWlsYWJsZSBjcHUgdmFsdWVzOiAyMDQ4ICgyIHZDUFUpXG4gICAqXG4gICAqIEJldHdlZW4gODE5MiAoOCBHQikgYW5kIDMwNzIwICgzMCBHQikgaW4gaW5jcmVtZW50cyBvZiAxMDI0ICgxIEdCKSAtIEF2YWlsYWJsZSBjcHUgdmFsdWVzOiA0MDk2ICg0IHZDUFUpXG4gICAqXG4gICAqIFRoaXMgZGVmYXVsdCBpcyBzZXQgaW4gdGhlIHVuZGVybHlpbmcgRmFyZ2F0ZVRhc2tEZWZpbml0aW9uIGNvbnN0cnVjdC5cbiAgICpcbiAgICogQGRlZmF1bHQgNTEyXG4gICAqL1xuICByZWFkb25seSBtZW1vcnlMaW1pdE1pQj86IG51bWJlcjtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB3aGV0aGVyIHRoZSBzZXJ2aWNlIHdpbGwgYmUgYXNzaWduZWQgYSBwdWJsaWMgSVAgYWRkcmVzcy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGFzc2lnblB1YmxpY0lwPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIEZhcmdhdGUgc2VydmljZSBydW5uaW5nIG9uIGFuIEVDUyBjbHVzdGVyIGZyb250ZWQgYnkgYSBuZXR3b3JrIGxvYWQgYmFsYW5jZXIuXG4gKi9cbmV4cG9ydCBjbGFzcyBOZXR3b3JrTXVsdGlwbGVUYXJnZXRHcm91cHNGYXJnYXRlU2VydmljZSBleHRlbmRzIE5ldHdvcmtNdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlIHtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB3aGV0aGVyIHRoZSBzZXJ2aWNlIHdpbGwgYmUgYXNzaWduZWQgYSBwdWJsaWMgSVAgYWRkcmVzcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhc3NpZ25QdWJsaWNJcDogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIEZhcmdhdGUgc2VydmljZSBpbiB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzZXJ2aWNlOiBGYXJnYXRlU2VydmljZTtcblxuICAvKipcbiAgICogVGhlIEZhcmdhdGUgdGFzayBkZWZpbml0aW9uIGluIHRoaXMgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uOiBGYXJnYXRlVGFza0RlZmluaXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHRhcmdldCBncm91cCBmb3IgdGhlIHNlcnZpY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGFyZ2V0R3JvdXA6IE5ldHdvcmtUYXJnZXRHcm91cDtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2UgY2xhc3MuXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2VQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmFzc2lnblB1YmxpY0lwID0gcHJvcHMuYXNzaWduUHVibGljSXAgIT09IHVuZGVmaW5lZCA/IHByb3BzLmFzc2lnblB1YmxpY0lwIDogZmFsc2U7XG5cbiAgICBpZiAocHJvcHMudGFza0RlZmluaXRpb24gJiYgcHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBzcGVjaWZ5IG9ubHkgb25lIG9mIFRhc2tEZWZpbml0aW9uIG9yIFRhc2tJbWFnZU9wdGlvbnMuJyk7XG4gICAgfSBlbHNlIGlmIChwcm9wcy50YXNrRGVmaW5pdGlvbikge1xuICAgICAgdGhpcy50YXNrRGVmaW5pdGlvbiA9IHByb3BzLnRhc2tEZWZpbml0aW9uO1xuICAgIH0gZWxzZSBpZiAocHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgY29uc3QgdGFza0ltYWdlT3B0aW9ucyA9IHByb3BzLnRhc2tJbWFnZU9wdGlvbnM7XG4gICAgICB0aGlzLnRhc2tEZWZpbml0aW9uID0gbmV3IEZhcmdhdGVUYXNrRGVmaW5pdGlvbih0aGlzLCAnVGFza0RlZicsIHtcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IHByb3BzLm1lbW9yeUxpbWl0TWlCLFxuICAgICAgICBjcHU6IHByb3BzLmNwdSxcbiAgICAgICAgZXhlY3V0aW9uUm9sZTogdGFza0ltYWdlT3B0aW9ucy5leGVjdXRpb25Sb2xlLFxuICAgICAgICB0YXNrUm9sZTogdGFza0ltYWdlT3B0aW9ucy50YXNrUm9sZSxcbiAgICAgICAgZmFtaWx5OiB0YXNrSW1hZ2VPcHRpb25zLmZhbWlseSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBjb250YWluZXJOYW1lID0gdGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJOYW1lICE9PSB1bmRlZmluZWQgPyB0YXNrSW1hZ2VPcHRpb25zLmNvbnRhaW5lck5hbWUgOiAnd2ViJztcbiAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRoaXMudGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKGNvbnRhaW5lck5hbWUsIHtcbiAgICAgICAgaW1hZ2U6IHRhc2tJbWFnZU9wdGlvbnMuaW1hZ2UsXG4gICAgICAgIGxvZ2dpbmc6IHRoaXMubG9nRHJpdmVyLFxuICAgICAgICBlbnZpcm9ubWVudDogdGFza0ltYWdlT3B0aW9ucy5lbnZpcm9ubWVudCxcbiAgICAgICAgc2VjcmV0czogdGFza0ltYWdlT3B0aW9ucy5zZWNyZXRzLFxuICAgICAgfSk7XG4gICAgICBpZiAodGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJQb3J0cykge1xuICAgICAgICBmb3IgKGNvbnN0IGNvbnRhaW5lclBvcnQgb2YgdGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJQb3J0cykge1xuICAgICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgICAgY29udGFpbmVyUG9ydFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3Qgc3BlY2lmeSBvbmUgb2Y6IHRhc2tEZWZpbml0aW9uIG9yIGltYWdlJyk7XG4gICAgfVxuICAgIGlmICghdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBlc3NlbnRpYWwgY29udGFpbmVyIG11c3QgYmUgc3BlY2lmaWVkJyk7XG4gICAgfVxuICAgIGlmICh0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIucG9ydE1hcHBpbmdzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLnNlcnZpY2UgPSB0aGlzLmNyZWF0ZUZhcmdhdGVTZXJ2aWNlKHByb3BzKTtcbiAgICBpZiAocHJvcHMudGFyZ2V0R3JvdXBzKSB7XG4gICAgICB0aGlzLmFkZFBvcnRNYXBwaW5nRm9yVGFyZ2V0cyh0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIsIHByb3BzLnRhcmdldEdyb3Vwcyk7XG4gICAgICB0aGlzLnRhcmdldEdyb3VwID0gdGhpcy5yZWdpc3RlckVDU1RhcmdldHModGhpcy5zZXJ2aWNlLCB0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIsIHByb3BzLnRhcmdldEdyb3Vwcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMudGFyZ2V0R3JvdXAgPSB0aGlzLmxpc3RlbmVyLmFkZFRhcmdldHMoJ0VDUycsIHtcbiAgICAgICAgdGFyZ2V0czogW3RoaXMuc2VydmljZV0sXG4gICAgICAgIHBvcnQ6IHRoaXMudGFza0RlZmluaXRpb24uZGVmYXVsdENvbnRhaW5lci5wb3J0TWFwcGluZ3NbMF0uY29udGFpbmVyUG9ydFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVGYXJnYXRlU2VydmljZShwcm9wczogTmV0d29ya011bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2VQcm9wcyk6IEZhcmdhdGVTZXJ2aWNlIHtcbiAgICByZXR1cm4gbmV3IEZhcmdhdGVTZXJ2aWNlKHRoaXMsICdTZXJ2aWNlJywge1xuICAgICAgY2x1c3RlcjogdGhpcy5jbHVzdGVyLFxuICAgICAgZGVzaXJlZENvdW50OiB0aGlzLmRlc2lyZWRDb3VudCxcbiAgICAgIHRhc2tEZWZpbml0aW9uOiB0aGlzLnRhc2tEZWZpbml0aW9uLFxuICAgICAgYXNzaWduUHVibGljSXA6IHRoaXMuYXNzaWduUHVibGljSXAsXG4gICAgICBzZXJ2aWNlTmFtZTogcHJvcHMuc2VydmljZU5hbWUsXG4gICAgICBoZWFsdGhDaGVja0dyYWNlUGVyaW9kOiBwcm9wcy5oZWFsdGhDaGVja0dyYWNlUGVyaW9kLFxuICAgICAgcHJvcGFnYXRlVGFnczogcHJvcHMucHJvcGFnYXRlVGFncyxcbiAgICAgIGVuYWJsZUVDU01hbmFnZWRUYWdzOiBwcm9wcy5lbmFibGVFQ1NNYW5hZ2VkVGFncyxcbiAgICAgIGNsb3VkTWFwT3B0aW9uczogcHJvcHMuY2xvdWRNYXBPcHRpb25zLFxuICAgIH0pO1xuICB9XG59Il19