from http import HTTPStatus
from typing import Any, Dict, List, Optional, Union, cast

import httpx

from ...client import Client
from ...models.problem_details import ProblemDetails
from ...models.rule import Rule
from ...types import Response


def _get_kwargs(
    workspace_id: str,
    data_point_id: str,
    *,
    client: Client,
    json_body: List["Rule"],
) -> Dict[str, Any]:
    url = "{}/DataPoint/workspace/{workspaceId}/dataPoint/{dataPointId}/rule".format(
        client.base_url, workspaceId=workspace_id, dataPointId=data_point_id
    )

    headers: Dict[str, str] = client.get_headers()
    cookies: Dict[str, Any] = client.get_cookies()

    json_json_body = []
    for json_body_item_data in json_body:
        json_body_item = json_body_item_data.to_dict()

        json_json_body.append(json_body_item)

    return {
        "method": "patch",
        "url": url,
        "headers": headers,
        "cookies": cookies,
        "timeout": client.get_timeout(),
        "json": json_json_body,
    }


def _parse_response(*, response: httpx.Response) -> Optional[Union[Any, ProblemDetails]]:
    if response.status_code == HTTPStatus.ACCEPTED:
        response_202 = cast(Any, None)
        return response_202
    if response.status_code == HTTPStatus.BAD_REQUEST:
        response_400 = ProblemDetails.from_dict(response.json())

        return response_400
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        response_401 = ProblemDetails.from_dict(response.json())

        return response_401
    if response.status_code == HTTPStatus.FORBIDDEN:
        response_403 = ProblemDetails.from_dict(response.json())

        return response_403
    return None


def _build_response(*, response: httpx.Response) -> Response[Union[Any, ProblemDetails]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(response=response),
    )


def sync_detailed(
    workspace_id: str,
    data_point_id: str,
    *,
    client: Client,
    json_body: List["Rule"],
) -> Response[Union[Any, ProblemDetails]]:
    """
    Args:
        workspace_id (str):
        data_point_id (str):
        json_body (List['Rule']):

    Returns:
        Response[Union[Any, ProblemDetails]]
    """

    kwargs = _get_kwargs(
        workspace_id=workspace_id,
        data_point_id=data_point_id,
        client=client,
        json_body=json_body,
    )

    response = httpx.request(
        verify=client.verify_ssl,
        **kwargs,
    )

    return _build_response(response=response)


def sync(
    workspace_id: str,
    data_point_id: str,
    *,
    client: Client,
    json_body: List["Rule"],
) -> Optional[Union[Any, ProblemDetails]]:
    """
    Args:
        workspace_id (str):
        data_point_id (str):
        json_body (List['Rule']):

    Returns:
        Response[Union[Any, ProblemDetails]]
    """

    return sync_detailed(
        workspace_id=workspace_id,
        data_point_id=data_point_id,
        client=client,
        json_body=json_body,
    ).parsed


async def asyncio_detailed(
    workspace_id: str,
    data_point_id: str,
    *,
    client: Client,
    json_body: List["Rule"],
) -> Response[Union[Any, ProblemDetails]]:
    """
    Args:
        workspace_id (str):
        data_point_id (str):
        json_body (List['Rule']):

    Returns:
        Response[Union[Any, ProblemDetails]]
    """

    kwargs = _get_kwargs(
        workspace_id=workspace_id,
        data_point_id=data_point_id,
        client=client,
        json_body=json_body,
    )

    async with httpx.AsyncClient(verify=client.verify_ssl) as _client:
        response = await _client.request(**kwargs)

    return _build_response(response=response)


async def asyncio(
    workspace_id: str,
    data_point_id: str,
    *,
    client: Client,
    json_body: List["Rule"],
) -> Optional[Union[Any, ProblemDetails]]:
    """
    Args:
        workspace_id (str):
        data_point_id (str):
        json_body (List['Rule']):

    Returns:
        Response[Union[Any, ProblemDetails]]
    """

    return (
        await asyncio_detailed(
            workspace_id=workspace_id,
            data_point_id=data_point_id,
            client=client,
            json_body=json_body,
        )
    ).parsed
