"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Document = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document
 */
class Document extends constructs_1.Construct {
    /**
     * Defines a new SSM document
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Name of the document
         */
        this.name = '';
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag(createdByTag, ID);
        const stack = aws_cdk_lib_1.Stack.of(this).stackName;
        this.lambda = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            aws_cdk_lib_1.Annotations.of(this).addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new aws_cdk_lib_1.CustomResource(this, `SSM-Document-${name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                attachments: props.attachments,
                versionName: props.versionName,
                StackName: stack,
                tags: aws_cdk_lib_1.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
            pascalCaseProperties: true,
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ssm:ListDocuments', 'ssm:ListTagsForResource'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ssm:AddTagsToResource', 'ssm:CreateDocument'],
                    resources: ['*'],
                    conditions: {
                        StringLike: {
                            'aws:RequestTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'ssm:AddTagsToResource',
                        'ssm:DeleteDocument',
                        'ssm:DescribeDocument',
                        'ssm:GetDocument',
                        'ssm:ListDocumentVersions',
                        'ssm:ModifyDocumentPermission',
                        'ssm:RemoveTagsFromResource',
                        'ssm:UpdateDocument',
                        'ssm:UpdateDocumentDefaultVersion',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringLike: {
                            'ssm:ResourceTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                }),
            ],
        });
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new aws_cdk_lib_1.aws_lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: aws_cdk_lib_1.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
_a = JSII_RTTI_SYMBOL_1;
Document[_a] = { fqn: "cdk-ssm-document.Document", version: "3.1.1" };
//# sourceMappingURL=data:application/json;base64,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