# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pydpkg']

package_data = \
{'': ['*']}

install_requires = \
['PGPy==0.5.4',
 'arpy>=2.2.0,<3.0.0',
 'six>=1.16.0,<2.0.0',
 'zstandard>=0.18.0,<0.19.0']

entry_points = \
{'console_scripts': ['dpkg-inspect = pydpkg.dpkg_inspect:main']}

setup_kwargs = {
    'name': 'pydpkg',
    'version': '1.7.0',
    'description': 'A python library for parsing debian package control headers and version strings',
    'long_description': "[![Build Status](https://travis-ci.org/TheClimateCorporation/python-dpkg.svg?branch=master)](https://travis-ci.org/TheClimateCorporation/python-dpkg)\n\n# python-dpkg\n\nThis library can be used to:\n\n1. read and extract control data from Debian-format package files, even\n   on platforms that generally lack a native implementation of dpkg\n\n2. compare dpkg version strings, using a pure Python implementation of\n   the algorithm described in section 5.6.12 of the debian-policy manual:\n   https://www.debian.org/doc/debian-policy/ch-controlfields.html#version\n\n3. Parse debian source description (dsc) files, inspect their contents\n   and verify that their source files are present and checksums are\n   correct.\n\nThis is primarily intended for use on platforms that do not normally\nship [python-apt](http://apt.alioth.debian.org/python-apt-doc/) due to\nlicensing restrictions or the lack of a native libapt.so (e.g. macOS)\n\nCurrently only tested on CPython 3.x, but at least in theory should run\non any python distribution that can install the [arpy](https://pypi.python.org/pypi/arpy/)\nlibrary.\n\n(Note: python 2.7 compatibility was removed in version 1.4.0.  We are\nkeeping a 1.3 release branch alive and are trying to make a good faith\neffort to backport fixes there, but it's strictly on a best-effort /\nspare-time basis, and we urge users to update to py3 as soon as possible.)\n\n## Installing\n\nInstall the 'pydpkg' package from [PyPi](https://pypi.python.org) using\nthe [pip](https://packaging.python.org/installing/) tool:\n\n    $ pip install pydpkg\n    Collecting pydpkg\n      Downloading pydpkg-1.1-py2-none-any.whl\n      Installing collected packages: pydpkg\n      Successfully installed pydpkg-1.1\n\n## Developing\n\npython-dpkg uses [Poetry](https://python-poetry.org/) to manage its dependencies.\n\nThe [Makefile](Makefile) will attempt to set up a reasonable build/test\nenvironment on both macOS/Darwin and more traditional unixes (linux, freebsd,\netc), but relies on the existence of [pyenv](https://github.com/pyenv/pyenv),\n[pyenv-virtualenv](https://github.com/pyenv/pyenv-virtualenv) everywhere and\n[Homebrew](https://brew.sh) on macOS.  You don't _need_ to use the workflow\nthat the makefile enforces (modern versions of [pip](https://pypi.org/project/pip/)\nwill happily digest `pyproject.toml` files and you can run the test commands\nmanually) but please ensure all tests pass before submitting PRs.\n\n## Usage\n\n### Binary Packages\n\n#### Read and extract headers\n\n    >>> from pydpkg import Dpkg\n    >>> dp = Dpkg('/tmp/testdeb_1:0.0.0-test_all.deb')\n\n    >>> dp.headers\n    {'maintainer': u'Climate Corp Engineering <no-reply@climate.com>', 'description': u'testdeb\\n a bogus debian package for testing dpkg builds', 'package': u'testdeb', 'section': u'base', 'priority': u'extra', 'installed-size': u'0', 'version': u'1:0.0.0-test', 'architecture': u'all'}\n\n    >>> print(dp)\n    Package: testdeb\n    Version: 1:0.0.0-test\n    Section: base\n    Priority: extra\n    Architecture: all\n    Installed-Size: 0\n    Maintainer: Climate Corp Engineering <no-reply@climate.com>\n    Description: testdeb\n     a bogus debian package for testing dpkg builds\n\n#### Interact directly with the package control message\n\n    >>> dp.message\n    <email.message.Message instance at 0x10895c6c8>\n    >>> dp.message.get_content_type()\n    'text/plain'\n\n#### Get package file fingerprints\n\n    >>> dp.fileinfo\n    {'sha256': '547500652257bac6f6bc83f0667d0d66c8abd1382c776c4de84b89d0f550ab7f', 'sha1': 'a5d28ae2f23e726a797349d7dd5f21baf8aa02b4', 'filesize': 910, 'md5': '149e61536a9fe36374732ec95cf7945d'}\n    >>> dp.md5\n    '149e61536a9fe36374732ec95cf7945d'\n    >>> dp.sha1\n    'a5d28ae2f23e726a797349d7dd5f21baf8aa02b4'\n    >>> dp.sha256\n    '547500652257bac6f6bc83f0667d0d66c8abd1382c776c4de84b89d0f550ab7f'\n    >>> dp.filesize\n    910\n\n#### Get the components of the package version\n\n    >>> d.epoch\n    1\n    >>> d.upstream_version\n    u'0.0.0'\n    >>> d.debian_revision\n    u'test'\n\n#### Get an arbitrary control header, case-independent\n\n    >>> d.version\n    u'1:0.0.0-test'\n    \n    >>> d.VERSION\n    u'1:0.0.0-test'\n    \n    >>> d.description\n    u'testdeb\\n a bogus debian package for testing dpkg builds'\n    \n    >>> d.get('nosuchheader', 'default')\n    'default'\n\n#### Compare current version to a candidate version\n\n    >>> dp.compare_version_with('1.0')\n    1\n\n    >>> dp.compare_version_with('1:1.0')\n    -1\n\n#### Compare two arbitrary version strings\n\n    >>> from pydpkg import Dpkg\n    >>> ver_1 = '0:1.0-test1'\n    >>> ver_2 = '0:1.0-test2'\n    >>> Dpkg.compare_versions(ver_1, ver_2)\n    -1\n\n#### Use as a key function to sort a list of version strings\n\n    >>> from pydpkg import Dpkg\n    >>> sorted(['0:1.0-test1', '1:0.0-test0', '0:1.0-test2'] , key=Dpkg.compare_versions_key)\n    ['0:1.0-test1', '0:1.0-test2', '1:0.0-test0']\n\n#### Use the `dpkg-inspect` script to inspect packages\n\n    $ dpkg-inspect ~/testdeb*deb\n    Filename: /Home/n/testdeb_1:0.0.0-test_all.deb\n    Size:     910\n    MD5:      149e61536a9fe36374732ec95cf7945d\n    SHA1:     a5d28ae2f23e726a797349d7dd5f21baf8aa02b4\n    SHA256:   547500652257bac6f6bc83f0667d0d66c8abd1382c776c4de84b89d0f550ab7f\n    Headers:\n      Package: testdeb\n      Version: 1:0.0.0-test\n      Section: base\n      Priority: extra\n      Architecture: all\n      Installed-Size: 0\n      Maintainer: Nathan Mehl <n@climate.com>\n      Description: testdeb\n       a bogus debian package for testing dpkg builds\n\n### Source Packages\n\n#### Read and extract headers\n\n    >>> from pydpkg import Dsc\n    >>> dsc = Dsc('testdeb_0.0.0.dsc')\n    >>> dsc.standards_version\n    '3.9.6'\n    >>> dsc.format\n    '3.0 (quilt)'\n    >>> x.build_depends\n    'python (>= 2.6.6-3), debhelper (>= 9)'\n\n#### Get the full set of dsc headers as a dictionary\n\n    >>> dsc.headers\n    {'Architecture': 'all',\n     'Binary': 'testdeb',\n     'Build-Depends': 'python (>= 2.6.6-3), debhelper (>= 9)',\n     'Checksums-Sha1': ' f250ac0a426b31df24fc2c98050f4fab90e456cd 280 testdeb_0.0.0.orig.tar.gz\\n cb3474ff94053018957ebcf1d8a2b45f75dda449 232 testdeb_0.0.0-1.debian.tar.xz\\n 80cd7b01014a269d445c63b037b885d6002cf533 841 testdeb_0.0.0.dsc',\n     'Checksums-Sha256': ' aa57ba8f29840383f5a96c5c8f166a9e6da7a484151938643ce2618e82bfeea7 280 testdeb_0.0.0.orig.tar.gz\\n 1ddb2a7336a99bc1d203f3ddb59f6fa2d298e90cb3e59cccbe0c84e359979858 232 testdeb_0.0.0-1.debian.tar.xz\\n b5ad1591349eb48db65e6865be506ad7dbd21931902a71addee5b1db9ae1ac2a 841 testdeb_0.0.0.dsc',\n     'Files': ' 142ca7334ed1f70302b4504566e0c233 280 testdeb_0.0.0.orig.tar.gz\\n fc80e6e7f1c1a08b78a674aaee6c1548 232 testdeb_0.0.0-1.debian.tar.xz\\n 893d13a2ef13f7409c9521e8ab1dbccb 841 testdeb_0.0.0.dsc',\n     'Format': '3.0 (quilt)',\n     'Homepage': 'https://github.com/TheClimateCorporation',\n     'Maintainer': 'Nathan J. Mehl <n@climate.com>',\n     'Package-List': 'testdeb',\n     'Source': 'testdeb',\n     'Standards-Version': '3.9.6',\n     'Uploaders': 'Nathan J. Mehl <n@climate.com>',\n     'Version': '0.0.0-1'}\n\n#### Interact directly with the dsc message\n\n    >>> dsc.message\n    <email.message.Message instance at 0x106fedea8>\n    >>> dsc.message.get_content_type()\n    'text/plain'\n    >>> dsc.message.get('uploaders')\n    'Nathan J. Mehl <n@climate.com>'\n\n#### Render the dsc message as a string\n\n    >>> print(dsc)\n    Format: 3.0 (quilt)\n    Source: testdeb\n    Binary: testdeb\n    Architecture: all\n    Version: 0.0.0-1\n    Maintainer: Nathan J. Mehl <n@climate.com>\n    Uploaders: Nathan J. Mehl <n@climate.com>\n    Homepage: https://github.com/TheClimateCorporation\n    Standards-Version: 3.9.6\n    Build-Depends: python (>= 2.6.6-3), debhelper (>= 9)\n    Package-List: testdeb\n    Checksums-Sha1:\n     f250ac0a426b31df24fc2c98050f4fab90e456cd 280 testdeb_0.0.0.orig.tar.gz\n     cb3474ff94053018957ebcf1d8a2b45f75dda449 232 testdeb_0.0.0-1.debian.tar.xz\n     80cd7b01014a269d445c63b037b885d6002cf533 841 testdeb_0.0.0.dsc\n    Checksums-Sha256:\n     aa57ba8f29840383f5a96c5c8f166a9e6da7a484151938643ce2618e82bfeea7 280 testdeb_0.0.0.orig.tar.gz\n     1ddb2a7336a99bc1d203f3ddb59f6fa2d298e90cb3e59cccbe0c84e359979858 232 testdeb_0.0.0-1.debian.tar.xz\n     b5ad1591349eb48db65e6865be506ad7dbd21931902a71addee5b1db9ae1ac2a 841 testdeb_0.0.0.dsc\n    Files:\n     142ca7334ed1f70302b4504566e0c233 280 testdeb_0.0.0.orig.tar.gz\n     fc80e6e7f1c1a08b78a674aaee6c1548 232 testdeb_0.0.0-1.debian.tar.xz\n     893d13a2ef13f7409c9521e8ab1dbccb 841 testdeb_0.0.0.dsc\n\n#### List the package source files from the dsc\n\n    >>> dsc.source_files\n    ['/tmp/testdeb_0.0.0.orig.tar.gz',\n     '/tmp/testdeb_0.0.0-1.debian.tar.xz',\n     '/tmp/testdeb_0.0.0.dsc' ]\n\n#### Validate that the package source files are present\n\n    >>> dsc.missing_files\n    []\n    >>> dsc.all_files_present\n    True\n    >>> dsc.validate()\n    >>>\n\n    >>> bad = Dsc('testdeb_1.1.1-bad.dsc')\n    >>> bad.missing_files\n    ['/tmp/testdeb_1.1.1.orig.tar.gz', '/tmp/testdeb_1.1.1-1.debian.tar.xz']\n    >>> bad.all_files_present\n    False\n    >>> bad.validate()\n    pydpkg.DscMissingFileError: ['/tmp/testdeb_1.1.1.orig.tar.gz', '/tmp/testdeb_1.1.1-1.debian.tar.xz']\n\n#### Inspect the source file checksums from the dsc\n\n    >>> pp(dsc.checksums)\n    {'md5': {'/tmp/testdeb_0.0.0-1.debian.tar.xz': 'fc80e6e7f1c1a08b78a674aaee6c1548',\n             '/tmp/testdeb_0.0.0.dsc': '893d13a2ef13f7409c9521e8ab1dbccb',\n             '/tmp/testdeb_0.0.0.orig.tar.gz': '142ca7334ed1f70302b4504566e0c233'},\n     'sha1': {'/tmp/testdeb_0.0.0-1.debian.tar.xz': 'cb3474ff94053018957ebcf1d8a2b45f75dda449',\n              '/tmp/testdeb_0.0.0.dsc': '80cd7b01014a269d445c63b037b885d6002cf533',\n              '/tmp/testdeb_0.0.0.orig.tar.gz': 'f250ac0a426b31df24fc2c98050f4fab90e456cd'},\n     'sha256': {'/tmp/testdeb_0.0.0-1.debian.tar.xz': '1ddb2a7336a99bc1d203f3ddb59f6fa2d298e90cb3e59cccbe0c84e359979858',\n                '/tmp/testdeb_0.0.0.dsc': 'b5ad1591349eb48db65e6865be506ad7dbd21931902a71addee5b1db9ae1ac2a',\n                '/tmp/testdeb_0.0.0.orig.tar.gz': 'aa57ba8f29840383f5a96c5c8f166a9e6da7a484151938643ce2618e82bfeea7'}}\n\n#### Validate that all source file checksums are correct\n\n    >>> dsc.corrected_checksums\n    {}\n    >>> dsc.all_checksums_correct\n    True\n    >>> dsc.validate()\n    >>>\n\n    >>> bad = Dsc('testdeb_0.0.0-badchecksums.dsc')\n    >>> bad.corrected_checksums\n    {'sha256': defaultdict(None, {'/tmp/testdeb_0.0.0-1.debian.tar.xz': '1ddb2a7336a99bc1d203f3ddb59f6fa2d298e90cb3e59cccbe0c84e359979858', '/tmp/testdeb_0.0.0.orig.tar.gz': 'aa57ba8f29840383f5a96c5c8f166a9e6da7a484151938643ce2618e82bfeea7'}), 'sha1': defaultdict(None, {'/tmp/testdeb_0.0.0-1.debian.tar.xz': 'cb3474ff94053018957ebcf1d8a2b45f75dda449', '/tmp/testdeb_0.0.0.orig.tar.gz': 'f250ac0a426b31df24fc2c98050f4fab90e456cd'})}\n    >>> bad.all_checksums_correct\n    False\n    >>> bad.validate()\n    pydpkg.DscBadChecksumsError: {'sha256': defaultdict(None, {'/tmp/testdeb_0.0.0-1.debian.tar.xz': '1ddb2a7336a99bc1d203f3ddb59f6fa2d298e90cb3e59cccbe0c84e359979858', '/tmp/testdeb_0.0.0.orig.tar.gz': 'aa57ba8f29840383f5a96c5c8f166a9e6da7a484151938643ce2618e82bfeea7'}), 'sha1': defaultdict(None, {'/tmp/testdeb_0.0.0-1.debian.tar.xz': 'cb3474ff94053018957ebcf1d8a2b45f75dda449', '/tmp/testdeb_0.0.0.orig.tar.gz': 'f250ac0a426b31df24fc2c98050f4fab90e456cd'})}\n",
    'author': 'Nathan J. Mehl',
    'author_email': 'pypi@memory.blank.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/memory/python-dpkg',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
