#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2022 Bootloader.  All rights reserved.
#
# This software is the confidential and proprietary information of
# Bootloader or one of its subsidiaries.  You shall not disclose this
# confidential information and shall use it only in accordance with the
# terms of the license agreement or other applicable agreement you
# entered into with Bootloader.
#
# BOOTLOADER MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
# SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT
# NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR
# A PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  BOOTLOADER SHALL NOT BE
# LIABLE FOR ANY LOSSES OR DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
# USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

from pathlib import Path
import argparse
import logging
import sys

from majormode.perseus.constant.logging import LOGGING_LEVELS
from majormode.perseus.constant.logging import LOGGING_LEVEL_LITERAL_STRINGS
from majormode.perseus.constant.logging import LoggingLevelLiteral
from majormode.perseus.utils.logging import cast_string_to_logging_level

from bootloader.unreal_engine.utils import project_version
from bootloader.unreal_engine.utils.project_version import DEFAULT_VERSION

DEFAULT_LOGGING_FORMATTER = logging.Formatter("%(asctime)s [%(levelname)s] %(message)s")
DEFAULT_LOGGING_LEVEL = LoggingLevelLiteral.info


def cast_string_to_path(path: str) -> Path:
    return Path(path)


def get_console_handler(logging_formatter: logging.Formatter = DEFAULT_LOGGING_FORMATTER) -> logging.StreamHandler:
    """
    Return a logging handler that sends logging output to the system's
    standard output.


    @param logging_formatter: An object `Formatter` to set for this handler
        to appropriately format logging records.


    @return: An instance of the `StreamHandler` class which write logging
        record, appropriately formatted, to the standard output stream.
    """
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setFormatter(logging_formatter)
    return console_handler


def parse_arguments() -> argparse.Namespace:
    """
    Convert argument strings to objects and assign them as attributes of
    the namespace.


    @return: An instance `Namespace` corresponding to the populated
        namespace.
    """
    parser = argparse.ArgumentParser(description="Unreal Engine project version utility")

    parser.add_argument(
        '--build-number',
        dest='build_number',
        metavar='NUMBER',
        required=False,
        type=int,
        help="specify the build number of the Unreal Engine project"
    )

    parser.add_argument(
        '--default-version',
        action='store_true',
        dest='use_default_version',
        required=False,
        type=bool,
        help="indicate whether to use a default version build number for the Unreal "
             "Engine project if none defined in the configuration file"
    )

    parser.add_argument(
        '--path',
        dest='path',
        metavar='PATH',
        required=True,
        type=cast_string_to_path,
        help="specify the path to the Unreal Engine project"
    )

    parser.add_argument(
        '--logging-level',
        dest='logging_level',
        metavar='LEVEL',
        required=False,
        default=str(LoggingLevelLiteral.info),
        type=cast_string_to_logging_level,
        help=f"specify the logging level ({', '.join(LOGGING_LEVEL_LITERAL_STRINGS)})"
    )

    return parser.parse_args()


def run():
    arguments = parse_arguments()

    setup_logger(logging_formatter=DEFAULT_LOGGING_FORMATTER, logging_level=arguments.logging_level)

    version = project_version.find_project_version_ini_file(
        arguments.path,
        require_semantic_versioning=True,
        default_version=DEFAULT_VERSION if arguments.use_default_version else None,
    )

    if arguments.build_number:
        version = f'{version}+{arguments.build_number}'

    print(version)


def setup_logger(
        logging_formatter: logging.Formatter = DEFAULT_LOGGING_FORMATTER,
        logging_level: LoggingLevelLiteral = DEFAULT_LOGGING_LEVEL,
        logger_name: str = None) -> logging.Logger:
    """
    Setup a logging handler that sends logging output to the system's
    standard output.


    @param logging_formatter: An object `Formatter` to to appropriately
        format logging records.

    @param logging_level: The logging threshold for this logger.  Logging
        messages which are less severe than this value will be ignored;
        logging messages which have severity level or higher will be
        emitted by whichever handler or handlers service this logger,
        unless a handler’s level has been set to a higher severity level
        than `logging_level`.

    @param logger_name: The name of the logger to add the logging handler
        to.  If `logger_name` is `None`, the function attaches the logging
        handler to the root logger of the hierarchy.


    @return: An object `Logger`.
    """
    logger = logging.getLogger(logger_name)
    logger.setLevel(LOGGING_LEVELS[logging_level or DEFAULT_LOGGING_LEVEL])
    logger.addHandler(get_console_handler(logging_formatter=logging_formatter))
    logger.propagate = False
    return logger
