"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * (experimental) VirtualService represents a service inside an AppMesh.
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 * @experimental
 */
class VirtualService extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given an ARN.
     *
     * @experimental
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given mesh and service names.
     *
     * @experimental
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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