"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComponentBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const yaml = require("yaml");
const Validator = require("yaml-validator");
const component_version_bump_1 = require("./component_version_bump");
const index_1 = require("./index");
const utils = require("./utils");
/**
 * @stability stable
 */
class ComponentBuilder {
    /**
     * @stability stable
     */
    constructor(scope, componentDefs, pipeline_name, platform) {
        /**
         * @stability stable
         */
        this.cacheDir = 'component_cache';
        this.componentDeps = componentDefs;
        this.scope = scope;
        this.pipelineName = pipeline_name;
        this.platform = platform;
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir);
        }
    }
    /**
     * @stability stable
     */
    createComponentDependenciesMap() {
        const components = this.componentDeps.map(c => this.createComponent(c)).filter(c => c);
        return components.reduce((a, el) => ({
            ...a,
            [el.name.substr(this.pipelineName.length + 3)]: el,
        }), {}); // remove the prefix
    }
    /**
     * @stability stable
     */
    createComponent(componentConfig) {
        let componentYaml;
        switch (componentConfig.type) {
            case 'local':
                componentYaml = this.assembleComponent(componentConfig, 'local_components');
                break;
            case 'git':
                componentYaml = this.assembleComponent(componentConfig, this.cacheDir);
                break;
            case 'aws_arn':
                return new index_1.ArnComponentRef(componentConfig.arn, `${this.pipelineName} - ${componentConfig.name}`);
            default:
                componentYaml = '';
        }
        if (componentConfig && componentYaml) {
            return new imagebuilder.CfnComponent(this.scope, `${utils.pascalize(componentConfig.name)}Component`, componentYaml);
        }
        return undefined;
    }
    /**
     * @stability stable
     */
    assembleComponent(componentConfig, basedir) {
        let componentPath = componentConfig.type === 'local' ? componentConfig.name : `${this.pipelineName}@${componentConfig.name}`;
        if (componentConfig.path) {
            componentPath = `${componentPath}/${componentConfig.path}`;
        }
        if (this.checkComponentYaml(basedir, componentPath)) {
            // Get metadata YAML
            const metadata = fs.readFileSync(path.join(basedir, componentPath, 'metadata.yaml'), 'utf-8');
            const metayaml = yaml.parse(metadata);
            // Get base component.yaml
            const baseComponent = fs.readFileSync(path.join(basedir, componentPath, 'component.yaml'), 'utf-8');
            const baseYaml = yaml.parse(baseComponent);
            let assembledYaml = baseYaml;
            if (metayaml.distribution_configs) {
                const distributionconfigcomp = metayaml.distribution_configs[this.platform];
                // if there is a platform specific component yaml specified, load it and add steps to the assembled component
                if (distributionconfigcomp) {
                    const componentExtension = fs.readFileSync(path.join(basedir, componentPath, distributionconfigcomp), 'utf-8');
                    const componentExtYaml = yaml.parse(componentExtension);
                    const phase_extensions = componentExtYaml.phase_extensions;
                    // Add phase extensions to base component
                    this.addPhaseExtensions(phase_extensions, assembledYaml);
                }
            }
            this.addDependencyConstants(componentConfig, assembledYaml);
            const name = `${this.pipelineName} - ${componentConfig.name}`;
            const componentVersionBump = new component_version_bump_1.ComponentVersionBump(this.scope, `${utils.pascalize(`${this.pipelineName} ${componentConfig.name}`)}VersionBump`, name);
            return {
                name: name,
                platform: metayaml.platform,
                version: componentVersionBump.nextVersion,
                data: yaml.stringify(assembledYaml),
                tags: {
                    ShortName: utils.pascalize(`${this.pipelineName} ${componentConfig.name}`),
                },
            };
        }
    }
    addDependencyConstants(componentConfig, assembledYaml) {
        if ('constants' in componentConfig) {
            let constantsDict = componentConfig.constants;
            let compConstants = Object.keys(constantsDict).map((k) => {
                let constantObj = {};
                constantObj[k] = {
                    type: 'string',
                    value: constantsDict[k],
                };
                return constantObj;
            });
            if ('constants' in assembledYaml) {
                // Override the keys from the component itself with the ones from the component dependencies.
                let compConstantsKeys = Object.keys(constantsDict);
                assembledYaml.constants = assembledYaml.constants.filter(function (c) {
                    // See if their is an intersection between the keys from component dependency and
                    // the ones defined in the original component.yaml file and remove those from component.yaml file.
                    return Object.keys(c).filter(d => compConstantsKeys.includes(d)).length == 0;
                });
                assembledYaml.constants.splice(assembledYaml.constants.length, 0, ...compConstants);
            }
            else {
                assembledYaml.constants = compConstants;
            }
        }
    }
    addPhaseExtensions(phase_extensions, assembledYaml) {
        for (let phase_extension of phase_extensions) {
            // For prebuild phases, copy the steps before the build step
            if (phase_extension.name === 'prebuild') {
                const buildphase = assembledYaml.phases.find((el) => el.name === 'build');
                if (buildphase) {
                    // Insert the prebuild before the build steps.
                    buildphase.steps.splice(0, 0, ...phase_extension.steps);
                }
                else {
                    // In case the original componen yaml doesn't include a build phase,
                    // copy the phase from the extension directly in the assembled YAML
                    const copied_phase = { ...phase_extension };
                    copied_phase.name = 'build';
                    assembledYaml.phases.push(copied_phase);
                }
            }
            else { // Else just copy the build phase.
                const extphase = assembledYaml.phases.find((el) => el.name === phase_extension.name);
                if (extphase) {
                    extphase.steps.push(...phase_extension.steps);
                }
            }
        }
    }
    checkComponentYaml(baseDir, componentPath) {
        const metadatafile = path.join(baseDir, componentPath, 'metadata.yaml');
        const componentfile = path.join(baseDir, componentPath, 'component.yaml');
        const metadataExists = fs.existsSync(metadatafile);
        const componentYamlExists = fs.existsSync(componentfile);
        const yamlValidator = new Validator({});
        yamlValidator.validate([
            metadatafile,
            componentfile,
        ]);
        return metadataExists && componentYamlExists && yamlValidator.report() == 0;
    }
}
exports.ComponentBuilder = ComponentBuilder;
_a = JSII_RTTI_SYMBOL_1;
ComponentBuilder[_a] = { fqn: "halloumi-ami-pipelines.ComponentBuilder", version: "0.0.26" };
class PhaseExtension {
}
class Step {
}
class Inputs {
}
//# sourceMappingURL=data:application/json;base64,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