# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['snyk']

package_data = \
{'': ['*']}

install_requires = \
['importlib_metadata>=1.6.1,<2.0.0',
 'mashumaro>=1.7,<2.0',
 'poetry-version>=0.1.3,<0.2.0',
 'requests>=2.22,<3.0',
 'retry>=0.9.2,<0.10.0']

setup_kwargs = {
    'name': 'pysnyk',
    'version': '0.5.0',
    'description': 'A Python client for the Snyk API',
    'long_description': '# pysnyk\n\n[![Build Status](https://travis-ci.org/snyk-labs/pysnyk.svg?branch=master)](https://travis-ci.org/snyk-labs/pysnyk)\n\nA Python client for the [Snyk API](https://snyk.docs.apiary.io/#).\n\n## Client\n\nUsing the client requires you to provide your Snyk API token.\n\n```python\nimport snyk\nclient = snyk.SnykClient("<your-api-token>")\n```\n\nBy default the client will connect to the public Snyk service. If you are using a local installation then you can provide the API url as the second argument.\n\n```python\nimport snyk\nclient = snyk.SnykClient("<your-api-token>", "<your-instance-of-snyk>")\n```\n\nBy default the `User-Agent` string for the API requests will be `pysnyk/<version>`. If you would like to send a custom user agent you can do so as follows:\n\n```python\nimport snyk\nclient = snyk.SnykClient("<your-api-token>", user_agent="<your-instance-of-snyk>")\n```\n\nBy default the requests are not retried. If you would like to retry failed requests with `backoff` and `delay` you can do so as follows:\n\n```python\nimport snyk\nclient = snyk.SnykClient("<your-api-token>", tries=4, delay=1, backoff=2)\n```\n\n- `tries` - the maximum number of attempts. **Default:** `1` (no retries)\n- `delay` - initial delay between attempts. **Default:** `1`\n- `backoff` - multiplier applied to delay between attempts. **Default:** `2`\n\n## Organizations\n\nWith the client we can get a list of Snyk organizations you are a member of:\n\n```python\nclient.organizations.all()\n```\n\nThis returns a list of `snyk.models.Organization` objects.\n\nIf you already have the ID of the organization you\'re after you can grab it directly:\n\n```python\nclient.organizations.get("<organization-id>")\n```\n\nThis will return a single `snyk.models.Organization` object.\n\nMost of the API is scoped to organizations, so most other methods are found on the `snyk.models.Organization` objects returned by these two methods.\n\nThe `snyk.models.Organization` object has the following properties related to the API:\n\n* `entitlements` - returns the set of Snyk features available to this account\n* `dependencies`- returns a Manager for packages in use in this organization\n* `licenses` - returns a Manager for licenses currently in use by projects in this organisation\n* `members` - returns a Manager for members\n* `projects` - returns a Manager for associated projects\n* `integrations` - returns a Manager for active integrations\n\n### A note on Managers\n\nManagers provide a consistent API for accessing objects from the Snyk API. Each manager implements the following methods:\n\n* `all()` - return a list of all of the relevant objects\n* `get("<id>")` - return a single instance of the object if it exists\n* `first()` - grab the first instance of the object if one exists\n* `filter(<key>="<value>")` - return a list filtered by one or more key/value pairs\n\n### Projects\n\nOnce you have an organization you\'re likely to want to grab the related projects:\n\n```python\nclient.organizations.first().projects.all()\n```\n\nThis will return a list of `snyk.models.Project` objects.\n\nIn the case where you want to get all of the projects across all of your organizations then you can use the handy method on the client.\n\n```python\nclient.projects.all()\n```\n\nThe `snyk.models.Project` object has the following useful properties and methods:\n\n* `delete()` - deletes the project in question. Be careful as this will delete all associated data too\n* `dependencies` - returns a Manager for packages in use in this project\n* `dependency_graph` - returns a `snyk.models.DependencyGraph` object which represents the full dependency graph of package dependencies\n* `ignores` - returns a Manager for ignore rules set on the project\n* `vulnerabilities` - returns a list of `snyk.models.Vulnerability` objects with information about vulnerabilities in this project\n* `jira_issues` - returns a Manager with access to any associated Jira issues\n* `licenses` - returns a Manager for licenses currently in use by this project\n* `settings` - returns a Manager for interacting with the current project settings\n* `tags` - returns a Manager for interacting with the current project tags\n\nYou can add and delete tags using the manager:\n\n* `tags.add(key, value)` - adds a tag with the provided key/value pair to the project\n* `tags.delete(key, value)` - deletes a tag with the provided key/value pair from the project\n\nIn the case of Projects, as well as filtering by properties (as mentioned above) you can also filter by tag:\n\n```python\nclient.organizations.first().projects.filter(tags = [{"key": "some-key", "value": "some-value"}])\n```\n\nNote that the `settings` Manager can also be used to update settings like so, assuming you have a `snyk.models.Project` object in the variable `project`.\n\n```python\nproject.settings.update(pull_request_test_enabled=True)\n```\n\n### Importing new projects\n\nThe client supports a high-level `import_project` method on organizations for adding new projects to be monitored by Snyk.\n\n```python\norg = client.organizations.first()\norg.import_project("github.com/user/project@branch")\norg.import_project("docker.io/repository:tag")\n```\n\nIf you are targetting a specific manifest file or files you can pass those as an optional argument, for instance:\n\n```python\norg.import_project("github.com/user/project@branch", files=["Gemfile.lock"])\n```\n\nThis method currently only supports importing projects from GitHub and Docker Hub. For other integrations you will need to grab the lower-level `snyk.models.Integration` object from the `snyk.models.Organization.integrations` manager noted above. Other services will be added to this API soon.\n\n\n### Testing for vulnerabilities\n\nThe API also exposes methods to discover vulnerability information about individual packages. These methods are found on the Organization object.\n\n* `test_maven(<package_group_id>, <package_artifact_id>, <version>)` - returns an IssueSet containing vulnerability information for a Maven artifact\n* `test_rubygem(<name>, <version>)` - returns an IssueSet containing vulnerability information for a Ruby Gem\n* `test_python(<name>, <version>)` - returns an IssueSet containing vulnerability information for Python package from PyPi\n* `test_npm(<name>, <version>)` - returns an IssueSet containing vulnerability information for an NPM package\n\n\nHere\'s an example of checking a particular Python package.\n\n```python\n>>> org = client.organizations.first()\n>>> result = org.test_python("flask", "0.12.2")\n>>> assert result.ok\nFalse\n# You can access details of the vulnerabilities too, for example\n>>> result.issues.vulnerabilities[0].title\n\'Improper Input Validation\'\n>>> result.issues.vulnerabilities[0].identifiers\n{\'CVE\': [\'CVE-2018-1000656\'], \'CWE\': [\'CWE-20\']\n```\n\nAs well as testing individual packages you can also test all packages found in various dependency management manifests. The client currently supports the following methods:\n\n* `test_pipfile(<file-handle-or-string>)` - returns an IssueSet for all Python dependencies in a `Pipfile`\n* `test_gemfilelock(<file-handle-or-string>)` - returns an IssueSet for all Ruby dependencies in a `Gemfile`\n* `test_packagejson(<file-handle-or-string>, (<lock-file-handle-or-string>))` - returns an IssueSet for all Javascript dependencies in a `package.json` file. Optionally takes a `package.lock` file\n* `test_gradlefile(<file-handle-or-string>)` - returns an IssueSet for all dependencies in a `Gradlefile`\n* `test_sbt(<file-handle-or-string>)` - returns an IssueSet for all dependencies defined in a `.sbt` file\n* `test_pom(<file-handle-or-string>)` - returns an IssueSet for all dependencies in a Maven `pom.xml` file\n* `test_yarn(<file-handle-or-string>, <lock-file-handle-or-string>)` - returns an IssueSet for all dependencies in Yarn `package.json` and `yarn.lock` files\n* `test_composer(<file-handle-or-string>, <lock-file-handle-or-string>)` - returns an IssueSet for all dependencies in Composer `composer.json` and `composer.lock` files\n\nFor example, here we are testing a Python `Pipfile`.\n\n```python\n>>> org = client.organizations.first()\n>>> file = open("Pipfile")\n>>> org.test_pipfile(file)\n```\n\n### Inviting new users\n\nYou can invite new users to the organization via the API.\n\n```python\n>>> org = client.organizations.first()\n>>> org.invite("example@example.com")\n```\n\nYou can also invite new users as administrators:\n\n```python\n>>> org = client.organizations.first()\n>>> org.invite("example@example.com", admin=True)\n```\n\n\n### Low-level client\n\nAs well as the high-level API of the Snyk client you can use the HTTP methods directly. For these you simply need to pass the path, and optionally a data payload. The full domain, and the authentication details, are already provided by the client.\n\n```python\nclient.get("<path>")\nclient.delete("<path>")\nclient.put("<path>", <data>)\nclient.post("<path>", <data>)\n```\n\nMost of the time you shouldn\'t need to use these. They are mainly useful if new methods are added to the API which are not yet supported in the client. This can also be useful if you want to pass very specific parameters, or to parse the raw JSON output from the API.\n',
    'author': 'Gareth Rushgrove',
    'author_email': 'garethr@snyk.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/snyk-labs/pysnyk',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
