# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['py_ts_interfaces', 'py_ts_interfaces.tests']

package_data = \
{'': ['*']}

install_requires = \
['astroid==2.12.2']

entry_points = \
{'console_scripts': ['py-ts-interfaces = py_ts_interfaces.cli:main']}

setup_kwargs = {
    'name': 'py-ts-interfaces',
    'version': '0.4.0',
    'description': 'A library that converts Python dataclasses with type annotations to a TypeScript interface and serializes them to a file.',
    'long_description': '# py-ts-interfaces\n### Python to TypeScript Interfaces\n\n![MIT License](https://img.shields.io/github/license/cs-cordero/py-ts-interfaces)\n![GitHub Workflow Status](https://img.shields.io/github/workflow/status/cs-cordero/py-ts-interfaces/Continuous%20Integration)\n![PyPI](https://img.shields.io/pypi/v/py-ts-interfaces)\n\n## What is this?\n\nThis library provides utilities that convert Python dataclasses with type\nannotations to a TypeScript `interface` and serializes them to a file.\n\n## Installation\n\n```\npython --version  # requires 3.7+\npip install py-ts-interfaces\n```\n\n## Motivation\n\nIn web applications where Python is used in the backend and TypeScript is used\nin the frontend, it is often the case that the client will make calls to the\nbackend to request some data with some specific pre-defined "shape".  On the\nclient-side, an `interface` for this data is usually defined and if the Python\nbackend authors use typechecking, like with [mypy](http://mypy-lang.org/), the\nproject authors may be typing the JSON response values as well.\n\nThis results in a duplication of code.  If the shape changes in the backend,\nthe related interface must also be reflect its changes in the frontend.  At\nbest, this is annoying to maintain.  At worst, over time the interfaces may\ndiverge and cause bugs.\n\nThis library aims to have a single source of truth that describes the shape of\nthe payload between the backend and the frontend.\n\n## Usage\n\nIn Python, `py-ts-interfaces` exposes a new class object called `Interface`.\nBy subclassing this object, you identify to the also-packaged script that you\nwant it to be serialized to an interface file.\n\n1. First, hook up your dataclasses:\n\n```python\n# views.py\nfrom dataclasses import dataclass\nfrom py_ts_interfaces import Interface\n\n@dataclass\nclass MyComponentProps(Interface):\n    name: str\n    show: bool\n    value: float\n\n@dataclass\nclass WillNotGetPickedUp:  # this doesn\'t subclass Interface, so it won\'t be included\n    name: str\n    value: float\n```\n\n2. In your shell, run the included command and pass in the name of the file or\n   directory you want to use.  By default it will output to a file in your\n   directory called interface.ts\n```\n$ py-ts-interfaces views.py\nCreated interface.ts!\n```\n\nYou may also use the following arguments:\n* `-o, --output [filepath]`:  where the file will be saved. default is `interface.ts`.\n* `-a, --append`:  by default each run will overwrite the output file. this flag\nallows only appends.  Be warned, duplicate interfaces are not tested.\n\n\n3. The resulting file will look like this:\n```typescript\n// interface.ts\ninterface MyComponentProps {\n    name: string;\n    show: boolean;\n    value: number;\n}\n```\n\n## Why @dataclass?\n\n`Dataclass`es were introduced in Python 3.7 and they are great.  Some\nalternatives that I have seen other codebases using are `NamedTuple` and\n`TypedDict`.  All of these objects attempt to do the same thing: group together\npieces of data that belong close together like a struct.\n\nHowever, `dataclass` won out over the other two for the following reasons:\n1. dataclasses are built-in to Python.  As of writing, `NamedTuple` is also\n   built-in to the `typing` module, but `TypedDict` is still considered\n   experimental.\n2. dataclasses cannot be declared and defined inline like you can do with\n   `NamedTuple` and `TypedDict`, e.g., `NamedTuple` can be defined using class\n   inheritance like `class MyNamedTuple(NamedTuple): ...`, but also like\n   `MyNamedTuple = NamedTuple(\'MyNamedTuple\', [(\'name\', str), (\'id\', int)])`.\n   This is a good thing.  Dataclasses require you to use a class style\n   declaration, which not only looks closer to a TypeScript interface\n   declaration, but it avoids the complex metaclass machinery that NamedTuples\n   and TypedDicts use to gain all its features.  Since this library uses the\n   AST and static analysis of the code to determine what data to serialize,\n   this makes the choice a no-brainer.\n3. dataclasses can be made to be immutable (mostly) by setting `frozen=True`.\n   This library does not require it but in later versions we may provide a\n   `partial`ed dataclass decorator that guarantees immutability.\n4. Because we avoid the metaclass machinery of NamedTuples and TypedDicts, it\n   opens up the possibility of writing custom classes that allows `mypy` to\n   typecheck it one way, but gives the AST parser some clues in order to\n   generate TypeScript types that cannot easily be expressed in Python.\n\n## Why define the types in Python instead of TypeScript?\n\nTypeScript is significantly more mature for typing syntax than Python.\nGenerally speaking, you can express any type that Python can do in TypeScript,\nbut _not_ vice versa.\n\nSo defining the types in Python guarantee that you can also express the whole\ninterface in both languages.\n\n## Supported Type Mappings\nPlease note that usage of `T` `U` and `V` in the table below represent\nstand-ins for actual types.  They do not represent actually using generic typed\nvariables.\n\n| Python                          | Typescript                    |\n|:-------------------------------:|:-----------------------------:|\n| None                            | null                          |\n| str                             | string                        |\n| int                             | number                        |\n| float                           | number                        |\n| complex                         | number                        |\n| bool                            | boolean                       |\n| List                            | Array\\<any\\>                  |\n| Tuple                           | [any]                         |\n| List[T]                         | Array[T]                      |\n| Tuple[T, U]                     | [T, U]                        |\n| Optional[T]                     | T \\| null                     |\n| Union[T, U, V]                  | T \\| U \\| V                   |\n\n## Planned Supported Mappings\n\n* String literals\n* Undefined type\n* isNaN type\n* ReadOnly types\n* Excess Properties\n\n## Unsupported/Rejected Mappings\n\nThe primary purpose of this library is to help type, first and foremost, _data_\nmoving back and forth from client to server.  Many of these features, whether they be specific to TypeScript or Python, would be overkill to support.\n\n* void\n* callables/functions\n* enums\n* Dates, datetime, dates, times (send these over as strings and convert them to richer objects on the client)\n* extends\n* generics, TypeVars\n* intersection types\n* mapped types\n* conditional types\n* classes\n\n## Contributing\n\nInterested in contributing?  You\'re awesome!  It\'s not much, but here\'s some notes to get you started [CONTRIBUTING.md](CONTRIBUTING.md).\n\n## Author\n\n[Christopher Sabater Cordero](https://chrisdoescoding.com)\n',
    'author': 'Christopher Cordero',
    'author_email': 'ccordero@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cs-cordero/py-ts-interfaces',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
