"""Phonetic mapping processors."""

from __future__ import annotations

from typing import Dict

from amharic_text_processor.base import BaseProcessor, ProcessorInput, ProcessorOutput


class OldPhoneMapper:
    """Map old phone representations to the newer form using a provided mapping."""

    MAPPING: Dict[str, str] = {
        "ህኡኣ": "ኋ",
        "ልኡኣ": "ሏ",
        "ምኡኣ": "ሟ",
        "ርኡኣ": "ሯ",
        "ስኡኣ": "ሷ",
        "ሽኡኣ": "ሿ",
        "ቅኡኣ": "ቋ",
        "ብኡኣ": "ቧ",
        "ቭኡኣ": "ቯ",
        "ትኡኣ": "ቷ",
        "ችኡኣ": "ቿ",
        "ንኡኣ": "ኗ",
        "ኝኡኣ": "ኟ",
        "ክኡኣ": "ኳ",
        "ዝኡኣ": "ዟ",
        "ዥኡኣ": "ዧ",
        "ድኡኣ": "ዷ",
        "ጅኡኣ": "ጇ",
        "ግኡኣ": "ጓ",
        "ጥኡኣ": "ጧ",
        "ጭኡኣ": "ጯ",
        "ጵኡኣ": "ጷ",
        "ጽኡኣ": "ጿ",
        "ፍኡኣ": "ፏ",
        "ፕኡኣ": "ፗ",
        "ግኡኤ": "ጔ",
        "ክኡኤ": "ኴ",
        "ህኡኤ": "ኌ",
        "ቅኡኤ": "ቌ",
        "ግኡኢ": "ጒ",
        "ክኡኢ": "ኲ",
        "ህኡኢ": "ኊ",
        "ቅኡኢ": "ቊ",
        "ህኧ": "ኸ",
        "ህኣ": "ሀ",
        "ህኡ": "ሁ",
        "ህኢ": "ሂ",
        "ህኤ": "ሄ",
        "ህኦ": "ሆ",
        "ልኧ": "ለ",
        "ልኡ": "ሉ",
        "ልኢ": "ሊ",
        "ልኣ": "ላ",
        "ልኤ": "ሌ",
        "ልኦ": "ሎ",
        "ምኧ": "መ",
        "ምኡ": "ሙ",
        "ምኢ": "ሚ",
        "ምኣ": "ማ",
        "ምኤ": "ሜ",
        "ምኦ": "ሞ",
        "ርኧ": "ረ",
        "ርኡ": "ሩ",
        "ርኢ": "ሪ",
        "ርኣ": "ራ",
        "ርኤ": "ሬ",
        "ርኦ": "ሮ",
        "ስኧ": "ሰ",
        "ስኡ": "ሱ",
        "ስኢ": "ሲ",
        "ስኣ": "ሳ",
        "ስኤ": "ሴ",
        "ስኦ": "ሶ",
        "ሽኧ": "ሸ",
        "ሽኡ": "ሹ",
        "ሽኢ": "ሺ",
        "ሽኣ": "ሻ",
        "ሽኤ": "ሼ",
        "ሽኦ": "ሾ",
        "ቅኧ": "ቀ",
        "ቅኡ": "ቁ",
        "ቅኢ": "ቂ",
        "ቅኣ": "ቃ",
        "ቅኤ": "ቄ",
        "ቅኦ": "ቆ",
        "ብኧ": "በ",
        "ብኡ": "ቡ",
        "ብኢ": "ቢ",
        "ብኣ": "ባ",
        "ብኤ": "ቤ",
        "ብኦ": "ቦ",
        "ቭኧ": "ቨ",
        "ቭኡ": "ቩ",
        "ቭኢ": "ቪ",
        "ቭኣ": "ቫ",
        "ቭኤ": "ቬ",
        "ቭኦ": "ቮ",
        "ትኧ": "ተ",
        "ትኡ": "ቱ",
        "ትኢ": "ቲ",
        "ትኣ": "ታ",
        "ትኤ": "ቴ",
        "ትኦ": "ቶ",
        "ችኧ": "ቸ",
        "ችኡ": "ቹ",
        "ችኢ": "ቺ",
        "ችኣ": "ቻ",
        "ችኤ": "ቼ",
        "ችኦ": "ቾ",
        "ንኧ": "ነ",
        "ንኡ": "ኑ",
        "ንኢ": "ኒ",
        "ንኣ": "ና",
        "ንኤ": "ኔ",
        "ንኦ": "ኖ",
        "ኝኧ": "ኘ",
        "ኝኡ": "ኙ",
        "ኝኢ": "ኚ",
        "ኝኣ": "ኛ",
        "ኝኤ": "ኜ",
        "ኝኦ": "ኞ",
        "ክኧ": "ከ",
        "ክኡ": "ኩ",
        "ክኢ": "ኪ",
        "ክኣ": "ካ",
        "ክኤ": "ኬ",
        "ክኦ": "ኮ",
        "ውኧ": "ወ",
        "ውኡ": "ዉ",
        "ውኢ": "ዊ",
        "ውኣ": "ዋ",
        "ውኤ": "ዌ",
        "ውኦ": "ዎ",
        "ዝኧ": "ዘ",
        "ዝኡ": "ዙ",
        "ዝኢ": "ዚ",
        "ዝኣ": "ዛ",
        "ዝኤ": "ዜ",
        "ዝኦ": "ዞ",
        "ዥኧ": "ዠ",
        "ዥኡ": "ዡ",
        "ዥኢ": "ዢ",
        "ዥኣ": "ዣ",
        "ዥኤ": "ዤ",
        "ዥኦ": "ዦ",
        "ይኧ": "የ",
        "ይኡ": "ዩ",
        "ይኢ": "ዪ",
        "ይኣ": "ያ",
        "ይኤ": "ዬ",
        "ይኦ": "ዮ",
        "ድኧ": "ደ",
        "ድኡ": "ዱ",
        "ድኢ": "ዲ",
        "ድኣ": "ዳ",
        "ድኤ": "ዴ",
        "ድኦ": "ዶ",
        "ጅኧ": "ጀ",
        "ጅኡ": "ጁ",
        "ጅኢ": "ጂ",
        "ጅኣ": "ጃ",
        "ጅኤ": "ጄ",
        "ጅኦ": "ጆ",
        "ግኧ": "ገ",
        "ግኡ": "ጉ",
        "ግኢ": "ጊ",
        "ግኣ": "ጋ",
        "ግኤ": "ጌ",
        "ግኦ": "ጐ",
        "ጥኧ": "ጠ",
        "ጥኡ": "ጡ",
        "ጥኢ": "ጢ",
        "ጥኣ": "ጣ",
        "ጥኤ": "ጤ",
        "ጥኦ": "ጦ",
        "ጭኧ": "ጨ",
        "ጭኡ": "ጩ",
        "ጭኢ": "ጪ",
        "ጭኣ": "ጫ",
        "ጭኤ": "ጬ",
        "ጭኦ": "ጮ",
        "ጵኧ": "ጰ",
        "ጵኡ": "ጱ",
        "ጵኢ": "ጲ",
        "ጵኣ": "ጳ",
        "ጵኤ": "ጴ",
        "ጵኦ": "ጶ",
        "ጽኧ": "ጸ",
        "ጽኡ": "ጹ",
        "ጽኢ": "ጺ",
        "ጽኣ": "ጻ",
        "ጽኤ": "ጼ",
        "ጽኦ": "ጾ",
        "ፍኧ": "ፈ",
        "ፍኡ": "ፉ",
        "ፍኢ": "ፊ",
        "ፍኣ": "ፋ",
        "ፍኤ": "ፌ",
        "ፍኦ": "ፎ",
        "ፕኧ": "ፐ",
        "ፕኡ": "ፑ",
        "ፕኢ": "ፒ",
        "ፕኣ": "ፓ",
        "ፕኤ": "ፔ",
        "ፕኦ": "ፖ",
    }

    def apply(self, data: ProcessorInput) -> ProcessorOutput:
        text = BaseProcessor._extract_text(data)
        replaced = text
        replaced = self._replace(replaced)
        return {"text": replaced, "old_phone_mapped": replaced != text}

    def _replace(self, text: str) -> str:
        for old, new in sorted(self.MAPPING.items(), key=lambda x: len(x[0]), reverse=True):
            text = text.replace(old, new)
        return text
