# -*- coding: utf-8 -*-
"""
Tests for Arithmat.

Copyright (C) 2019 Roberto Pagaria
Copyright (C) 2019 Giovanni Paolini

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see http://www.gnu.org/licenses/.
"""


# This file was *autogenerated* from the file test.sage
from sage.all_cmdline import *   # import sage library

_sage_const_0 = Integer(0); _sage_const_1 = Integer(1); _sage_const_10 = Integer(10); _sage_const_9 = Integer(9); _sage_const_2 = Integer(2); _sage_const_3 = Integer(3); _sage_const_6 = Integer(6); _sage_const_5 = Integer(5); _sage_const_4 = Integer(4); _sage_const_29 = Integer(29); _sage_const_7 = Integer(7); _sage_const_8 = Integer(8); _sage_const_23 = Integer(23); _sage_const_26 = Integer(26); _sage_const_21 = Integer(21); _sage_const_12 = Integer(12); _sage_const_50 = Integer(50); _sage_const_11 = Integer(11); _sage_const_18 = Integer(18); _sage_const_15 = Integer(15); _sage_const_13 = Integer(13); _sage_const_48 = Integer(48); _sage_const_73 = Integer(73)
import unittest
import itertools
from arithmetic_matroid import *
from shnf import signed_hermite_normal_form as shnf


def representation_to_matroid(A):
    """
    Given an integer matrix A, return the associated ArithmeticMatroid.
    """
    E = range(A.ncols())

    def rk(X):
        return A[:,list(X)].rank()

    def m(X):
        return reduce(operator.mul, [d for d in A[:,list(X)].elementary_divisors() if d != _sage_const_0 ], _sage_const_1 )

    return ArithmeticMatroid(E, rk, m)



class TestArithmeticMatroid(unittest.TestCase):

    def test_trivial(self):
        E = range(_sage_const_1 , _sage_const_10 )

        def rk(X):
            return len(X)

        def m(X):
            return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertEqual(M.full_rank(), _sage_const_9 )
        self.assertTrue(M.is_representable())


    def test_valid(self):
        # matroid with representation
        # [ 1, a, 1 ]
        # [ 0, b, 5 ]
        # with a|b and (b,5)=1

        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            a = _sage_const_3 
            b = _sage_const_6 

            X = tuple(sorted(X))
            return {
                (): _sage_const_1 ,
                (_sage_const_1 ,): _sage_const_1 ,
                (_sage_const_2 ,): a,
                (_sage_const_3 ,): _sage_const_1 ,
                (_sage_const_1 ,_sage_const_2 ): b,
                (_sage_const_1 ,_sage_const_3 ): _sage_const_5 ,
                (_sage_const_2 ,_sage_const_3 ): _sage_const_5 *a-b,
                (_sage_const_1 ,_sage_const_2 ,_sage_const_3 ): _sage_const_1 ,
            }[X]

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())
        self.assertEqual(M.full_multiplicity(), _sage_const_1 )


    def test_not_valid(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            a = _sage_const_3 
            b = _sage_const_6 

            X = tuple(sorted(X))
            return {
                (): _sage_const_1 ,
                (_sage_const_1 ,): _sage_const_1 ,
                (_sage_const_2 ,): a,
                (_sage_const_3 ,): _sage_const_1 ,
                (_sage_const_1 ,_sage_const_2 ): b,
                (_sage_const_1 ,_sage_const_3 ): _sage_const_5 ,
                (_sage_const_2 ,_sage_const_3 ): _sage_const_5 *a-b+_sage_const_1 ,
                (_sage_const_1 ,_sage_const_2 ,_sage_const_3 ): _sage_const_1 ,
            }[X]

        M = ArithmeticMatroid(E, rk, m)
        self.assertFalse(M.is_valid())


    def test_valid2(self):
        # valid, not representable, not orientable
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertFalse(M.is_representable())
        self.assertFalse(M.is_orientable())
        self.assertEqual(M.representation(), None)
        self.assertEqual(M.full_multiplicity(), _sage_const_1 )


    def test_relizable(self):
        # representable with a 2x2 matrix
        for a in range(_sage_const_6 ):
            for b in range(_sage_const_1 , _sage_const_6 ):
                A = matrix(ZZ, [[_sage_const_1 , a], [_sage_const_0 , b]])
                M = representation_to_matroid(A)

                self.assertTrue(M.is_valid())
                self.assertTrue(M.is_representable())
                self.assertTrue(M.is_orientable())

                R = ZZ['x, y']
                x, y = R._first_ngens(_sage_const_2 )
                self.assertEqual(M.arithmetic_tutte_polynomial(), x**_sage_const_2  + x*(gcd(a,b)-_sage_const_1 ) + b-gcd(a,b))


    def test_representable2(self):
        A = matrix(ZZ, [[-_sage_const_1 , -_sage_const_29 , -_sage_const_1 , _sage_const_1 ], [_sage_const_1 , -_sage_const_1 , _sage_const_0 , _sage_const_1 ]])
        M = representation_to_matroid(A)

        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())
        self.assertTrue(M.is_orientable())


    def test_representable3(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , _sage_const_0 , -_sage_const_1 ], [_sage_const_6 , _sage_const_1 , _sage_const_1 , -_sage_const_1 , -_sage_const_1 ]])
        M = representation_to_matroid(A)

        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())
        self.assertTrue(M.is_orientable())


    def test_representable4(self):
        A = matrix(ZZ, [[_sage_const_2 , _sage_const_2 , _sage_const_1 , _sage_const_0 , _sage_const_0 ], [_sage_const_1 , _sage_const_5 , -_sage_const_1 , _sage_const_1 , -_sage_const_2 ], [-_sage_const_2 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , -_sage_const_1 ]])
        M = representation_to_matroid(A)

        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())
        self.assertTrue(M.is_orientable())


    def test_representable_random(self):
        A = random_matrix(ZZ,_sage_const_4 ,_sage_const_6 )
        M = representation_to_matroid(A)

        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())
        self.assertTrue(M.is_orientable())


    def test_pseudo(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2 :
                return _sage_const_3 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertFalse(M.is_representable())
        self.assertTrue(M.is_orientable())


    def test_non_representable(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 ]])
        M = representation_to_matroid(A)
        M2 = ArithmeticMatroid(M.groundset(), M.rank, lambda X: M._multiplicity(X)**_sage_const_2 )

        self.assertTrue(M2.is_valid())
        self.assertTrue(M.is_representable())
        self.assertFalse(M2.is_representable())


    def test_non_representable2(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        M = representation_to_matroid(A)
        M2 = ArithmeticMatroid(M.groundset(), M.rank, lambda X: M._multiplicity(X)**_sage_const_2 )

        self.assertTrue(M2.is_valid())
        self.assertTrue(M.is_representable())
        self.assertFalse(M2.is_representable())


    def test_non_orientable(self):
        E = range(_sage_const_1 , _sage_const_7 )

        def rk(A):
            return min(len(A), _sage_const_2 )

        def m(X):
            A = tuple(sorted(X))
            if A == (_sage_const_1 ,_sage_const_2 ,_sage_const_3 ):
                return _sage_const_1 
            if len(A) == _sage_const_0 :
                return _sage_const_1 
            elif A == (_sage_const_1 ,) or A == (_sage_const_2 ,):
                return _sage_const_2 
            elif len(A) == _sage_const_1 :
                return _sage_const_1 
            elif len(A) == _sage_const_2  and len([x for x in A if x <= _sage_const_2 ]) == _sage_const_1 :
                return _sage_const_2 
            elif _sage_const_1  in A and _sage_const_2  in A and len(A) == _sage_const_3 :
                return _sage_const_2 
            elif A == (_sage_const_1 ,_sage_const_2 ):
                return _sage_const_4 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertFalse(M.is_representable())
        self.assertFalse(M.is_orientable())


    def test_non_essential(self):
        A = matrix(ZZ, [[-_sage_const_1 ,  _sage_const_1 ,  _sage_const_0 ], [ _sage_const_6 ,  _sage_const_1 , -_sage_const_1 ], [_sage_const_2 , -_sage_const_3 , _sage_const_0 ], [_sage_const_1 , _sage_const_2 , _sage_const_3 ], [-_sage_const_1 , _sage_const_0 , _sage_const_0 ]])
        M = representation_to_matroid(A)

        self.assertTrue(M.is_valid())
        self.assertTrue(M.is_representable())


    def test_hash(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        s = set([M])
        self.assertEqual(hash(M), hash((frozenset(E), _sage_const_2 , _sage_const_1 , _sage_const_1 )))
        self.assertEqual(M.groundset(), frozenset(E))


    def test_isomorphism(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())
        self.assertFalse(M.is_representable())
        self.assertFalse(M.is_orientable())

        E1 = [_sage_const_4 ,_sage_const_5 ,_sage_const_6 ,_sage_const_7 ,_sage_const_8 ]

        def m1(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_6 ,_sage_const_7 ,_sage_const_8 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M1 = ArithmeticMatroid(E1, rk, m1)
        self.assertTrue(M1.is_valid())
        self.assertFalse(M1.is_representable())
        self.assertFalse(M1.is_orientable())

        self.assertTrue(M.is_isomorphism(M1, {i: i+_sage_const_3  for i in E}))
        self.assertFalse(M.equals(M1))

        M2 = ArithmeticMatroid(copy.copy(E), rk, m)
        self.assertTrue(M.equals(M2))
        self.assertTrue(M2.equals(M))


    def test_isomorphic(self):
        A = matrix(ZZ, [[_sage_const_1 ]])
        B = matrix(ZZ, [[_sage_const_3 ,_sage_const_1 ],[_sage_const_0 ,_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        DN = ToricArithmeticMatroid(B).minor(deletions = [_sage_const_0 ])
        CN = ToricArithmeticMatroid(B).minor(contractions = [_sage_const_0 ])
        self.assertTrue(DN.is_isomorphic(M))
        self.assertFalse(CN.is_isomorphic(M))

    def test_num_representations(self):
        r = _sage_const_3 

        for m in range(_sage_const_23 , _sage_const_26 ): # m(E)
            E = range(r)
            A = matrix(ZZ, r, r, lambda i, j: _sage_const_1  if i == j and i < r-_sage_const_1  else _sage_const_0  if j < r-_sage_const_1  else m if i == r-_sage_const_1  else _sage_const_1 )

            M = representation_to_matroid(A)
            self.assertEqual(M.num_representations(), (euler_phi(m)//_sage_const_2 )**(r-_sage_const_1 ))


    def test_gcd(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_6 , _sage_const_3 , -_sage_const_2 , _sage_const_2 ], [_sage_const_3 , _sage_const_21 , _sage_const_0 , -_sage_const_9 ], [-_sage_const_1 , -_sage_const_4 , _sage_const_3 , -_sage_const_2 ]]))

        self.assertTrue(M.is_torsion_free())
        self.assertFalse(M.is_surjective())
        self.assertTrue(M.is_representable())
        self.assertTrue(M.is_gcd())
        self.assertFalse(M.is_strong_gcd())

        M1 = M.reduction()

        self.assertTrue(M1.is_torsion_free())
        self.assertTrue(M1.is_surjective())
        self.assertTrue(M1.is_representable())
        self.assertTrue(M1.is_gcd())

        self.assertTrue(M1.is_strong_gcd())



class TestDualAndMinor(unittest.TestCase):
    # TODO: test delete and contract, test multiplicity

    def test_dual_arithmetic_matroid(self):
        # test of DualArithmeticMatroid

        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())

        M1 = M.dual()
        self.assertIsInstance(M1, DualArithmeticMatroid)
        self.assertTrue(M1.is_valid())

        M2 = M1.dual()
        self.assertIsInstance(M2, ArithmeticMatroid)
        self.assertNotIsInstance(M2, DualArithmeticMatroid)
        self.assertEqual(M, M2)

        M3 = M1.minor()
        self.assertIsInstance(M3, DualArithmeticMatroid)
        # self.assertEqual(M1, M3) # this is not an equality for (non-arithmetic) matroids
        self.assertTrue(M3.is_valid())


    def test_minor_arithmetic_matroid(self):
        # test of MinorArithmeticMatroid

        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())

        M1 = M.minor(contractions=[_sage_const_1 ], deletions=[_sage_const_2 ])
        self.assertIsInstance(M1, MinorArithmeticMatroid)
        self.assertTrue(M1.is_valid())

        M2 = M1.dual()
        self.assertTrue(M2.is_valid())
        self.assertIsInstance(M2, DualArithmeticMatroid)
        self.assertNotEqual(M, M2)


        N1 = M.dual()
        self.assertIsInstance(N1, DualArithmeticMatroid)
        self.assertTrue(N1.is_valid())

        N2 = N1.minor(contractions=[_sage_const_2 ], deletions=[_sage_const_1 ])
        self.assertTrue(N2.is_valid())
        self.assertIsInstance(N2, DualArithmeticMatroid)
        self.assertEqual(M2, N2)

        N3 = N1.minor(contractions=[_sage_const_1 ], deletions=[_sage_const_2 ])
        self.assertTrue(N3.is_valid())
        self.assertIsInstance(N3, DualArithmeticMatroid)
        self.assertNotEqual(M2, N3)


    def test_contract_delete(self):
        A = matrix(ZZ, [[_sage_const_2 ,_sage_const_0 ,_sage_const_4 ],[_sage_const_0 ,_sage_const_3 ,_sage_const_0 ]])
        M = ToricArithmeticMatroid(A)
        D = M.delete([_sage_const_1 ,_sage_const_2 ])
        C = M.contract([_sage_const_1 ,_sage_const_2 ])
        self.assertEqual(C.full_multiplicity(), _sage_const_6 )
        self.assertEqual(D.full_multiplicity(), _sage_const_2 )
        self.assertEqual(C.multiplicity([]), _sage_const_12 )
        self.assertEqual(D.multiplicity([]), _sage_const_1 )
        self.assertEqual(C,M.minor(contractions=[_sage_const_1 ,_sage_const_2 ], deletions=[]))
        self.assertEqual(D,M.minor(contractions=[], deletions=[_sage_const_1 ,_sage_const_2 ]))


    def test_dual_linear_matroid(self):
        A = matrix(QQ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])

        def m(X):
            return _sage_const_1 

        M = LinearArithmeticMatroid(A, multiplicity_function=m)
        self.assertTrue(M.is_valid())

        M1 = M.dual()
        self.assertTrue(M1.is_valid())

        M2 = M1.dual()
        self.assertTrue(M2.is_valid())
        self.assertNotEqual(M, M2) # multiplicity functions are not equal in the sense of ==
        self.assertTrue(M.equals(M2))
        self.assertNotEqual(M, M1)
        self.assertNotEqual(M1, M2)


    def test_minor_linear_matroid(self):
        A = matrix(QQ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])

        def m(X):
            return _sage_const_1 

        M = LinearArithmeticMatroid(A, multiplicity_function=m)
        self.assertTrue(M.is_valid())

        M1 = M.minor(contractions=frozenset([_sage_const_1 ]), deletions=frozenset([_sage_const_2 ]))
        self.assertIsInstance(M1, LinearArithmeticMatroid)
        self.assertTrue(M1.is_valid())

        M1a = M.contract([_sage_const_1 ]).delete([_sage_const_2 ])
        self.assertNotEqual(M1, M1a) # multiplicity functions are not equal in the sense of ==
        self.assertTrue(M1.equals(M1a))

        M2 = M1.dual()
        self.assertTrue(M2.is_valid())
        self.assertIsInstance(M2, LinearArithmeticMatroid)
        self.assertNotEqual(M, M2)
        self.assertFalse(M.equals(M2))


        N1 = M.dual()
        self.assertIsInstance(N1, LinearArithmeticMatroid)
        self.assertTrue(N1.is_valid())

        N2 = N1.minor(contractions=frozenset([_sage_const_2 ]), deletions=frozenset([_sage_const_1 ]))
        self.assertTrue(N2.is_valid())
        self.assertIsInstance(N2, LinearArithmeticMatroid)
        self.assertNotEqual(M2, N2) # multiplicity functions are not equal in the sense of ==
        self.assertTrue(M2.equals(N2))

        N3 = N1.minor(contractions=frozenset([_sage_const_1 ]), deletions=frozenset([_sage_const_2 ]))
        self.assertTrue(N3.is_valid())
        self.assertIsInstance(N3, LinearArithmeticMatroid)
        self.assertNotEqual(M2, N3)
        self.assertFalse(M2.equals(N3))


    def test_delete(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        N = M.delete([_sage_const_2 ,_sage_const_3 ])
        self.assertEqual(N.groundset(), frozenset([_sage_const_1 ,_sage_const_4 ,_sage_const_5 ]))

        N = N.delete([_sage_const_5 ])
        self.assertEqual(N.groundset(), frozenset([_sage_const_1 ,_sage_const_4 ]))
        self.assertFalse(repr(N) is None)


    def test_contract(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_2  and all(x in [_sage_const_3 ,_sage_const_4 ,_sage_const_5 ] for x in X):
                return _sage_const_2 
            else:
                return _sage_const_1 

        M = ArithmeticMatroid(E, rk, m)
        N = M.contract([_sage_const_2 ,_sage_const_3 ])
        self.assertEqual(N.groundset(), frozenset([_sage_const_1 ,_sage_const_4 ,_sage_const_5 ]))

        N = N.delete([_sage_const_5 ])
        self.assertEqual(N.groundset(), frozenset([_sage_const_1 ,_sage_const_4 ]))




class TestToric(unittest.TestCase):

    def test_without_Q(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        M = ToricArithmeticMatroid(A)

        self.assertEqual(M.full_rank(), M.rank(M.groundset()))

        self.assertEqual(M._Q, matrix(ZZ, _sage_const_2 , _sage_const_0 ))
        self.assertEqual(M.rank([_sage_const_0 ]), _sage_const_1 )
        self.assertEqual(M.rank([_sage_const_0 ,_sage_const_1 ]), _sage_const_2 )
        self.assertEqual(M.rank([_sage_const_0 ,_sage_const_1 ,_sage_const_2 ]), _sage_const_2 )

        self.assertEqual(M._multiplicity([_sage_const_1 ,_sage_const_2 ]), _sage_const_1 )

        # minor
        M2 = M.minor(contractions=[], deletions=[_sage_const_1 ])
        self.assertEqual(M2.groundset(), frozenset([_sage_const_0 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]))
        self.assertEqual(M2._A, matrix(ZZ, [[-_sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]]))

        M2 = M.minor(contractions=[_sage_const_1 ], deletions=[])
        self.assertEqual(M2.groundset(), frozenset([_sage_const_0 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]))
        self.assertEqual(M2._Q, matrix(ZZ, [[]]))
        self.assertEqual(M2._multiplicity([_sage_const_0 ,_sage_const_2 ]), _sage_const_1 )

        # dual
        M1 = M.dual()
        self.assertEqual(M1.rank(M1.groundset()), _sage_const_4 )
        M2 = M1.dual()
        self.assertTrue(M.equals(M2))

        # check representation
        self.assertTrue(M.check_representation(A))
        self.assertTrue(M.is_representable())
        self.assertEqual(M.representation(), A)
        self.assertEqual(M.representation(ordered_groundset=[_sage_const_5 ,_sage_const_4 ,_sage_const_3 ,_sage_const_2 ,_sage_const_1 ,_sage_const_0 ]), matrix(ZZ, [[_sage_const_7 , _sage_const_2 , -_sage_const_1 , _sage_const_0 , _sage_const_1 , -_sage_const_1 ], [_sage_const_5 , _sage_const_2 , -_sage_const_2 , -_sage_const_1 , _sage_const_1 , _sage_const_6 ]]))

        # orientability
        self.assertTrue(M.is_orientable())


    def test_with_Q(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        Q = matrix(ZZ, [[_sage_const_5 , _sage_const_9 , _sage_const_1 ], [-_sage_const_3 , -_sage_const_2 , -_sage_const_1 ]])
        M = ToricArithmeticMatroid(A, torus_matrix=Q)

        self.assertEqual(M.full_rank(), M.rank(M.groundset()))

        self.assertEqual(M._Q.ncols(), _sage_const_0 )
        self.assertEqual(M._multiplicity([_sage_const_0 ]), _sage_const_1 )


    def test_with_Q_2(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_0 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        Q = matrix(ZZ, [[_sage_const_3 , _sage_const_9 , _sage_const_6 ], [_sage_const_0 , _sage_const_0 , _sage_const_0 ]])
        M = ToricArithmeticMatroid(A, torus_matrix=Q)

        self.assertEqual(M.full_rank(), M.rank(M.groundset()))

        self.assertEqual(M._Q, matrix(ZZ, [[_sage_const_3 ], [_sage_const_0 ]]))
        self.assertEqual(M.rank([_sage_const_0 ]), _sage_const_0 )
        self.assertEqual(M.rank([_sage_const_1 ,_sage_const_2 ]), _sage_const_1 )

        self.assertEqual(M._multiplicity([_sage_const_0 ]), _sage_const_1 )
        self.assertEqual(M._multiplicity([_sage_const_1 ]), _sage_const_3 )
        self.assertEqual(M._multiplicity([_sage_const_1 ,_sage_const_2 ]), _sage_const_1 )


    def test_with_Q_3(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 ]])
        Q = matrix(ZZ, [[_sage_const_5 ], [_sage_const_1 ]])
        M = ToricArithmeticMatroid(A, Q)

        self.assertEqual(M.full_rank(), M.rank(M.groundset()))
        self.assertEqual(M._Q.ncols(), _sage_const_0 )


    def test_minor(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        M = ToricArithmeticMatroid(A)

        self.assertEqual(M.full_rank(), M.rank(M.groundset()))

        self.assertEqual(M._Q, matrix(ZZ, _sage_const_2 , _sage_const_0 ))
        self.assertEqual(M.rank([_sage_const_0 ]), _sage_const_1 )
        self.assertEqual(M.rank([_sage_const_0 ,_sage_const_1 ]), _sage_const_2 )
        self.assertEqual(M.rank([_sage_const_0 ,_sage_const_1 ,_sage_const_2 ]), _sage_const_2 )

        self.assertEqual(M.multiplicity([_sage_const_1 ,_sage_const_2 ]), _sage_const_1 )

        # minor
        M2 = M.minor(contractions=[], deletions=[_sage_const_1 ])
        self.assertEqual(M2.groundset(), frozenset([_sage_const_0 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]))
        self.assertEqual(M2._A, matrix(ZZ, [[-_sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]]))

        M2 = M.minor(contractions=[_sage_const_1 ], deletions=[])
        self.assertEqual(M2.groundset(), frozenset([_sage_const_0 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ]))
        self.assertEqual(M2._Q, matrix(ZZ, [[]]))
        self.assertEqual(M2.multiplicity([_sage_const_0 ,_sage_const_2 ]), _sage_const_1 )


    def test_minor2(self):
        A = matrix(ZZ, [[_sage_const_2 ,_sage_const_1 ],[_sage_const_0 ,_sage_const_3 ]])
        M = ToricArithmeticMatroid(A).minor(contractions=[_sage_const_0 ], deletions=[])
        N = ToricArithmeticMatroid(A).minor(contractions=[], deletions=[_sage_const_0 ])
        self.assertFalse(M.is_torsion_free())
        self.assertTrue(N.is_torsion_free())
        self.assertFalse(N.is_isomorphic(M))


    def test_ordered_groundset(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        M = ToricArithmeticMatroid(A, ordered_groundset=['a', 'b', 'f', 'e', 'd', 'c'])

        self.assertEqual(M.groundset(), frozenset(['a', 'b', 'f', 'e', 'd', 'c']))
        self.assertEqual(M.full_rank(), M.rank(M.groundset()))

        self.assertEqual(M._Q, matrix(ZZ, _sage_const_2 , _sage_const_0 ))
        self.assertEqual(M.rank(['a']), _sage_const_1 )
        self.assertEqual(M.rank(['a', 'b']), _sage_const_2 )
        self.assertEqual(M.rank(['f', 'a', 'b']), _sage_const_2 )

        self.assertEqual(M._multiplicity(['b', 'f']), _sage_const_1 )

        # test failure if there are repetition in the ordered groundset
        with self.assertRaises(AssertionError):
            M = ToricArithmeticMatroid(A, ordered_groundset=['a', 'b', 'f', 'e', 'd', 'b'])


    def test_equivalence(self):
        # without Q
        A = matrix(ZZ, [[_sage_const_1 , _sage_const_1 , _sage_const_2 ], [_sage_const_0 , _sage_const_7 , _sage_const_7 ]])
        B = matrix(ZZ, [[-_sage_const_1 , -_sage_const_1 , _sage_const_2 ], [-_sage_const_1 , _sage_const_6 , -_sage_const_5 ]])

        M = ToricArithmeticMatroid(A)
        N = ToricArithmeticMatroid(B)

        self.assertTrue(M.is_equivalent(N))
        self.assertTrue(N.is_equivalent(M))

        self.assertTrue(M.is_isomorphic(N))

        C = matrix(ZZ, [[_sage_const_1 , _sage_const_2 , _sage_const_3 ], [_sage_const_0 , _sage_const_7 , _sage_const_7 ]])
        O = ToricArithmeticMatroid(C)
        self.assertFalse(M.is_equivalent(O))
        self.assertFalse(N.is_equivalent(O))
        self.assertFalse(O.is_equivalent(M))
        self.assertFalse(O.is_equivalent(N))

        self.assertTrue(M.is_isomorphic(O))


    def test_non_equivalent_representations(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_6 , _sage_const_3 , -_sage_const_2 , _sage_const_2 ], [_sage_const_3 , _sage_const_21 , _sage_const_0 , -_sage_const_9 ], [-_sage_const_1 , -_sage_const_4 , _sage_const_3 , -_sage_const_2 ]]))

        self.assertEqual(M.num_representations(), _sage_const_4 )
        matroids = [ToricArithmeticMatroid(A) for A in M.all_representations()]
        for N, O in itertools.combinations(matroids, _sage_const_2 ):
            self.assertFalse(N.is_equivalent(O))
            self.assertTrue(N.is_isomorphic(O))


    def test_decomposition(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 , _sage_const_5 , -_sage_const_1 , _sage_const_5 ], [_sage_const_1 , _sage_const_7 , _sage_const_2 , _sage_const_6 ], [_sage_const_0 , -_sage_const_2 , -_sage_const_2 , -_sage_const_1 ]]))
        self.assertEqual(M.decomposition(), SetPartition([[_sage_const_2 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_3 ]]))
        self.assertTrue(M.is_decomposable())


    def test_decomposition2(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 , _sage_const_5 , -_sage_const_10 , _sage_const_50 ], [_sage_const_0 , _sage_const_0 , _sage_const_2 , _sage_const_6 ], [_sage_const_0 , _sage_const_0 , -_sage_const_2 , -_sage_const_1 ]]))
        self.assertEqual(M.decomposition(), SetPartition([[_sage_const_0 ,_sage_const_1 ], [_sage_const_2 ,_sage_const_3 ]]))
        self.assertTrue(M.is_decomposable())


    def test_decomposition3(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 , _sage_const_1 ], [_sage_const_0 , _sage_const_2 ]]))
        self.assertEqual(M.decomposition(), SetPartition([[_sage_const_0 ,_sage_const_1 ]]))
        self.assertFalse(M.is_decomposable())


    def test_poset_of_layers(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_3 ]]))
        P = M.poset_of_layers()
        self.assertEqual(len(P), _sage_const_7 )
        self.assertTrue(P.has_bottom())
        self.assertFalse(P.has_top())
        self.assertEqual(len(P.cover_relations()), _sage_const_10 )
        self.assertTrue(P.is_isomorphic(Poset(data=(range(_sage_const_7 ), [(_sage_const_0 ,_sage_const_1 ), (_sage_const_0 ,_sage_const_2 ), (_sage_const_0 ,_sage_const_3 ), (_sage_const_1 ,_sage_const_4 ), (_sage_const_2 ,_sage_const_4 ), (_sage_const_2 ,_sage_const_5 ), (_sage_const_2 ,_sage_const_6 ), (_sage_const_3 ,_sage_const_4 ), (_sage_const_3 ,_sage_const_5 ), (_sage_const_3 ,_sage_const_6 )]))))


    def test_poset_of_layers_relabeled(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_3 ]]), ordered_groundset=['a', 'c', 'b'])
        P = M.poset_of_layers()
        self.assertEqual(len(P), _sage_const_7 )
        self.assertTrue(P.has_bottom())
        self.assertFalse(P.has_top())
        self.assertEqual(len(P.cover_relations()), _sage_const_10 )
        self.assertTrue(P.is_isomorphic(Poset(data=(range(_sage_const_7 ), [(_sage_const_0 ,_sage_const_1 ), (_sage_const_0 ,_sage_const_2 ), (_sage_const_0 ,_sage_const_3 ), (_sage_const_1 ,_sage_const_4 ), (_sage_const_2 ,_sage_const_4 ), (_sage_const_2 ,_sage_const_5 ), (_sage_const_2 ,_sage_const_6 ), (_sage_const_3 ,_sage_const_4 ), (_sage_const_3 ,_sage_const_5 ), (_sage_const_3 ,_sage_const_6 )]))))
        self.assertEqual(set(S for (S, x) in P), set([(), ('a',), ('b',), ('c',), ('a', 'c', 'b'), ('a', 'b')]))


    def test_poset_of_layers_root_system_C2(self):
        # see [DGP17]
        A = matrix(ZZ, [[_sage_const_2 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ,], [_sage_const_0 ,_sage_const_2 ,_sage_const_1 ,-_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        P = M.poset_of_layers()
        self.assertEqual(len(P), _sage_const_11 )
        self.assertEqual(len(P.cover_relations()), _sage_const_18 )
        homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()
        self.assertEqual([homology[i].ngens() for i in range(_sage_const_2 )], [_sage_const_0 ,_sage_const_3 ])
        self.assertEqual(homology[_sage_const_1 ].ngens(), M.arithmetic_tutte_polynomial()(x=_sage_const_0 , y=_sage_const_0 ))


    def test_poset_of_layers_root_system_C3(self):
        # see [DGP17]
        A = matrix(ZZ, [[_sage_const_2 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_0 ,_sage_const_0 ], [_sage_const_0 ,_sage_const_2 ,_sage_const_0 ,_sage_const_1 ,-_sage_const_1 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_2 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,-_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        P = M.poset_of_layers()
        homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()
        self.assertEqual([homology[i].ngens() for i in range(_sage_const_3 )], [_sage_const_0 ,_sage_const_0 ,_sage_const_15 ])
        self.assertEqual(homology[_sage_const_2 ].ngens(), M.arithmetic_tutte_polynomial()(x=_sage_const_0 , y=_sage_const_0 ))


    def test_poset_of_layers_root_system_B2(self):
        # see [DGP17]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ,], [_sage_const_0 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        P = M.poset_of_layers()
        homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()
        self.assertEqual([homology[i].ngens() for i in range(_sage_const_2 )], [_sage_const_0 ,_sage_const_1 ])
        self.assertEqual(homology[_sage_const_1 ].ngens(), M.arithmetic_tutte_polynomial()(x=_sage_const_0 , y=_sage_const_0 ))


    def test_poset_of_layers_root_system_B3(self):
        # see [DGP17]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_0 ,_sage_const_0 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_0 ,_sage_const_1 ,-_sage_const_1 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,-_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        P = M.poset_of_layers()
        homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()
        self.assertEqual([homology[i].ngens() for i in range(_sage_const_3 )], [_sage_const_0 ,_sage_const_0 ,_sage_const_6 ])
        self.assertEqual(homology[_sage_const_2 ].ngens(), M.arithmetic_tutte_polynomial()(x=_sage_const_0 , y=_sage_const_0 ))


    def test_poset_of_layers_root_system_D3(self):
        # see [DGP17]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_0 ,_sage_const_0 ], [_sage_const_1 ,-_sage_const_1 ,_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,-_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ]])
        M = ToricArithmeticMatroid(A)
        P = M.poset_of_layers()
        homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()
        self.assertEqual([homology[i].ngens() for i in range(_sage_const_3 )], [_sage_const_0 ,_sage_const_0 ,_sage_const_3 ])
        self.assertEqual(homology[_sage_const_2 ].ngens(), M.arithmetic_tutte_polynomial()(x=_sage_const_0 , y=_sage_const_0 ))


    def test_different_posets_of_layers(self):
        # see [PP19, Section 8]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,_sage_const_3 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,_sage_const_5 ]])
        B = matrix(ZZ, [[_sage_const_1 ,_sage_const_4 ,_sage_const_1 ,_sage_const_6 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,_sage_const_5 ]])
        C = matrix(ZZ, [[_sage_const_1 ,_sage_const_2 ,_sage_const_2 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])

        # check signed Hermite normal form
        self.assertEqual(shnf(A), matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_3 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,-_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]]))
        self.assertEqual(shnf(B), matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]]))
        self.assertEqual(shnf(C), matrix(ZZ, [[_sage_const_1 ,_sage_const_2 ,_sage_const_2 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]]))

        MA = ToricArithmeticMatroid(A)
        MB = ToricArithmeticMatroid(B)
        MC = ToricArithmeticMatroid(C)

        self.assertTrue(MA.is_isomorphic(MB))
        self.assertTrue(MA.is_isomorphic(MC))
        self.assertTrue(MB.is_isomorphic(MC))

        self.assertFalse(MA.poset_of_layers().is_isomorphic(MB.poset_of_layers()))
        self.assertFalse(MA.poset_of_layers().is_isomorphic(MC.poset_of_layers()))
        self.assertFalse(MB.poset_of_layers().is_isomorphic(MC.poset_of_layers()))


    def test_independence_poset(self):
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_3 ]])
        M = ToricArithmeticMatroid(A)
        P = M.arithmetic_independence_poset()

        self.assertEqual(len(P), _sage_const_9 )
        self.assertEqual(len(P.cover_relations()), _sage_const_13 )
        self.assertEqual(P.rank(), _sage_const_2 )
        self.assertTrue(P.has_bottom())


    def test_independence_poset_relabeled(self):
        M = ToricArithmeticMatroid(matrix(ZZ, [[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_1 ,_sage_const_3 ]]), ordered_groundset=['a', 'c', 'b'])
        P = M.arithmetic_independence_poset()

        self.assertEqual(len(P), _sage_const_9 )
        self.assertEqual(len(P.cover_relations()), _sage_const_13 )
        self.assertEqual(P.rank(), _sage_const_2 )
        self.assertTrue(P.has_bottom())
        self.assertEqual(set(S for (S, x) in P), set([(), ('a',), ('b',), ('c',), ('a', 'c'), ('a', 'b'), ('c', 'b')]))


    def test_nonshellable_poset_of_layers(self):
        # see [PP19, Section 8]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_3 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,-_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])
        B = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])
        C = matrix(ZZ, [[_sage_const_1 ,_sage_const_2 ,_sage_const_2 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])

        for X in [A, B, C]:
            M = ToricArithmeticMatroid(X)
            P = M.poset_of_layers()
            homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()

            from sage.homology.homology_group import HomologyGroup
            self.assertEqual(homology, {
                _sage_const_0 : HomologyGroup(_sage_const_0 , ZZ),    # trivial group
                _sage_const_1 : HomologyGroup(_sage_const_1 , ZZ, [_sage_const_5 ]),   # Z/5Z
                _sage_const_2 : HomologyGroup(_sage_const_48 , ZZ),   # Z^48
                })


    def test_nonshellable_arithmetic_independence_poset(self):
        # see [PP19, Section 8]
        A = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_3 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,-_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])
        B = matrix(ZZ, [[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ,-_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])
        C = matrix(ZZ, [[_sage_const_1 ,_sage_const_2 ,_sage_const_2 ,_sage_const_1 ], [_sage_const_0 ,_sage_const_5 ,_sage_const_0 ,_sage_const_5 ], [_sage_const_0 ,_sage_const_0 ,_sage_const_5 ,-_sage_const_5 ]])

        for X in [A, B, C]:
            M = ToricArithmeticMatroid(X)
            P = M.arithmetic_independence_poset()
            homology = P.subposet([a for a in P if a != P.bottom()]).order_complex(on_ints=True).homology()

            from sage.homology.homology_group import HomologyGroup
            self.assertEqual(homology, {
                _sage_const_0 : HomologyGroup(_sage_const_0 , ZZ),    # trivial group
                _sage_const_1 : HomologyGroup(_sage_const_1 , ZZ, [_sage_const_5 ]),   # Z/5Z
                _sage_const_2 : HomologyGroup(_sage_const_73 , ZZ),   # Z^73
                })



class TestReduction(unittest.TestCase):

    def test_example(self):
        E = [_sage_const_1 ,_sage_const_2 ,_sage_const_3 ,_sage_const_4 ,_sage_const_5 ,_sage_const_6 ]

        def rk(X):
            return min(_sage_const_2 , len(X))

        def m(X):
            if len(X) == _sage_const_0 :
                return _sage_const_1 
            elif len(X) == _sage_const_1 :
                if X == frozenset([_sage_const_1 ]) or X == frozenset([_sage_const_2 ]):
                    return _sage_const_2 
                else:
                    return _sage_const_1 
            elif sum(_sage_const_1  for i in X if i not in [_sage_const_1 ,_sage_const_2 ]) >= _sage_const_2 :
                return _sage_const_1 
            elif len(X) == _sage_const_2 :
                if X == frozenset([_sage_const_1 ,_sage_const_2 ]):
                    return _sage_const_4 
                else:
                    return _sage_const_2 
            elif _sage_const_3  in X:
                # X = {1,2,3}
                assert X == frozenset([_sage_const_1 ,_sage_const_2 ,_sage_const_3 ])
                return _sage_const_1 
            else:
                # X = {1,2,j} with j > 3
                return _sage_const_2 

        M = ArithmeticMatroid(E, rk, m)
        self.assertTrue(M.is_valid())

        M1 = M.reduction()
        self.assertEqual(M1.multiplicity([_sage_const_1 ,_sage_const_2 ,_sage_const_3 ]), _sage_const_2 )

        for X in powerset(E):
            if tuple(sorted(X)) != (_sage_const_1 ,_sage_const_2 ,_sage_const_3 ):
                self.assertEqual(M1.multiplicity(X), M.multiplicity(X))

        self.assertFalse(M1.is_valid())

        self.assertTrue(M1.is_torsion_free())
        self.assertTrue(M1.is_surjective())


    def test_dual(self):
        A = matrix(ZZ, [[-_sage_const_1 , _sage_const_1 , _sage_const_0 , -_sage_const_1 , _sage_const_2 , _sage_const_7 ], [_sage_const_6 , _sage_const_1 , -_sage_const_1 , -_sage_const_2 , _sage_const_2 , _sage_const_5 ]])
        M = representation_to_matroid(A)
        M2 = ArithmeticMatroid(M.groundset(), M.rank, lambda X: M._multiplicity(X)**_sage_const_2 )

        self.assertTrue(M.is_valid())
        self.assertTrue(M2.is_valid())
        self.assertFalse(M2.is_representable())

        self.assertFalse(M.dual().reduction().is_isomorphic(M.reduction()))
        self.assertTrue(M.dual().reduction().is_isomorphic(M.reduction().dual()))
        self.assertTrue(M2.dual().reduction().is_isomorphic(M2.reduction().dual()))



class TestSHNF(unittest.TestCase):

    def test_1x1(self):
        for x in range(-_sage_const_5 , _sage_const_5 ):
            A = matrix(ZZ, [[x]])
            self.assertEqual(shnf(A), matrix(ZZ, [[abs(x)]]))


    def test_2x2(self):
        A = matrix(ZZ, [[_sage_const_5 , _sage_const_8 ], [_sage_const_0 , _sage_const_3 ]])
        self.assertEqual(shnf(A), matrix(ZZ, [[_sage_const_5 , _sage_const_1 ], [_sage_const_0 , _sage_const_3 ]]))


    def random_test(self, r, n, all=False):
        A = random_matrix(ZZ, r, n)
        U = random_matrix(ZZ, r, r, algorithm='unimodular')

        if all:
            for diag in itertools.product([_sage_const_1 , -_sage_const_1 ], repeat=n):
                S = diagonal_matrix(diag)
                self.assertEqual(shnf(A), shnf(U*A*S))

        else:
            S = diagonal_matrix([sage.misc.prandom.choice([_sage_const_1 , -_sage_const_1 ]) for i in range(n)])
            self.assertEqual(shnf(A), shnf(U*A*S))


    def test_random_1x2(self):
        self.random_test(_sage_const_1 , _sage_const_2 , all=True)

    def test_random_1x7(self):
        self.random_test(_sage_const_1 , _sage_const_7 )

    def test_random_2x2(self):
        self.random_test(_sage_const_2 , _sage_const_2 , all=True)

    def test_random_2x3(self):
        self.random_test(_sage_const_2 , _sage_const_3 , all=True)

    def test_random_2x4(self):
        self.random_test(_sage_const_2 , _sage_const_4 , all=True)

    def test_random_3x4(self):
        self.random_test(_sage_const_3 , _sage_const_4 , all=True)

    def test_random_4x6(self):
        self.random_test(_sage_const_4 , _sage_const_6 )

    def test_random_6x10(self):
        self.random_test(_sage_const_6 , _sage_const_10 )


    def test_cardinality_four_orbit(self):
        # in these examples, an orbit of cardinality 4 occurs
        # see [PP19, Example 6.8]
        for A in [
            matrix(ZZ, [[_sage_const_1 , _sage_const_1 , _sage_const_1 ], [_sage_const_0 , _sage_const_2 , _sage_const_3 ], [_sage_const_0 , _sage_const_0 , _sage_const_6 ]]),
            matrix(ZZ, [[_sage_const_1 , _sage_const_1 , _sage_const_1 ], [_sage_const_0 , _sage_const_2 , _sage_const_4 ], [_sage_const_0 , _sage_const_0 , _sage_const_8 ]]),
        ]:
            self.assertEqual(shnf(A), A)

            for i in range(_sage_const_3 ):
                U = random_matrix(ZZ, _sage_const_3 , _sage_const_3 , algorithm='unimodular')
                for diag in itertools.product([_sage_const_1 , -_sage_const_1 ], repeat=_sage_const_3 ):
                    S = diagonal_matrix(diag)
                    self.assertEqual(shnf(U*A*S), A)

        for u in [_sage_const_1 ,_sage_const_2 ,_sage_const_4 ,_sage_const_5 ]:
            A = matrix(ZZ, [[_sage_const_1 , _sage_const_1 , u], [_sage_const_0 , _sage_const_2 , _sage_const_3 ], [_sage_const_0 , _sage_const_0 , _sage_const_6 ]])
            self.assertEqual(shnf(A), matrix(ZZ, [[_sage_const_1 , _sage_const_1 , _sage_const_1 ], [_sage_const_0 , _sage_const_2 , _sage_const_3 ], [_sage_const_0 , _sage_const_0 , _sage_const_6 ]]))


    def test_is_in_hermite_form(self):
        # check that the output of shnf is in Hermite normal form
        for i in range(_sage_const_10 ):
            A = random_matrix(ZZ, _sage_const_3 , _sage_const_4 )
            B = shnf(A)
            self.assertEqual(B, B.echelon_form())


    def test_zero_pivot(self):
        # in this example, the second column has a zero pivot
        A = matrix(ZZ, [[_sage_const_1 , -_sage_const_1 , -_sage_const_1 , _sage_const_11 ], [_sage_const_1 , -_sage_const_1 , _sage_const_0 , -_sage_const_1 ], [_sage_const_3 , -_sage_const_3 , _sage_const_7 , -_sage_const_2 ]])
        U = random_matrix(ZZ, _sage_const_3 , _sage_const_3 , algorithm='unimodular')
        for diag in itertools.product([_sage_const_1 , -_sage_const_1 ], repeat=_sage_const_4 ):
            S = diagonal_matrix(diag)
            self.assertEqual(shnf(U*A*S), shnf(A))
            self.assertEqual(shnf(U*A), shnf(A))


    def test_not_reducible(self):
        # in this example, the matrix seems a block matrix but it isn't
        A = matrix(ZZ, [[_sage_const_2 ,_sage_const_2 ,_sage_const_0 ,_sage_const_0 ,_sage_const_0 ],[_sage_const_0 ,_sage_const_4 ,_sage_const_0 ,_sage_const_0 ,_sage_const_4 ],[_sage_const_0 ,_sage_const_0 ,_sage_const_4 ,_sage_const_4 ,_sage_const_0 ],[_sage_const_0 ,_sage_const_0 ,_sage_const_0 ,_sage_const_8 ,_sage_const_8 ]])
        for diag in itertools.product([_sage_const_1 , -_sage_const_1 ], repeat=_sage_const_5 ):
            S = diagonal_matrix(diag)
            self.assertEqual(shnf(A*S), shnf(A))



if __name__ == '__main__':
    unittest.main()

