import datetime
import os
from unittest import TestCase

import pandas as pd
from sklearn.model_selection import train_test_split

from rulelearn.algorithms.ripper import RipperExplainer
from rulelearn.algorithms.ripper.ripper_ruleset_generator import RipperRuleSetGenerator
from rulelearn.trxf.classifier.ruleset_classifier import RuleSetClassifier, RuleSelectionMethod, \
    WeightMetric, ConfidenceMetric
from rulelearn.trxf.pmml_export import NyokaSerializer
from rulelearn.trxf.pmml_export.pmml_exporter import PmmlExporter
from rulelearn.trxf.pmml_export.reader.trxf_reader import TrxfReader
from tests.trxf.utilities import create_test_ruleset, DATA_FRAME

TIMESTAMP = datetime.datetime(1970, 1, 1, tzinfo=datetime.timezone.utc)


class TestPmmlExporter(TestCase):
    def test_export(self):
        reader = TrxfReader()
        reader.load_data_dictionary(DATA_FRAME)
        serializer = NyokaSerializer(TIMESTAMP)
        exporter = PmmlExporter(reader, serializer)

        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources/toto.pmml')) as file:
            expected = file.read()
        test_classifier = RuleSetClassifier([(create_test_ruleset())], RuleSelectionMethod.FIRST_HIT, default_label=0)
        actual = exporter.export(test_classifier)
        self.assertEqual(expected, actual)

    def test_export_with_missing_data_dict_should_raise(self):
        reader = TrxfReader()
        serializer = NyokaSerializer()
        exporter = PmmlExporter(reader, serializer)

        test_classifier = RuleSetClassifier([(create_test_ruleset())], RuleSelectionMethod.FIRST_HIT, default_label=0)
        self.assertRaises(AssertionError, exporter.export, test_classifier)

    def test_ripper_iris(self):
        data = pd.read_csv('https://archive.ics.uci.edu/ml/machine-learning-databases/iris/iris.data',
                           header=None,
                           names=['sepal_length', 'sepal_width', 'petal_length', 'petal_width', 'class_name'],
                           dtype={'sepal_length': float,
                                  'sepal_width': float,
                                  'petal_length': float,
                                  'petal_width': float,
                                  'class_name': str})

        print('HI', data.info())

        col_names = ['sepal_length', 'sepal_width', 'petal_length', 'petal_width', 'class_name']

        x_train, x_test, y_train, y_test = train_test_split(
            data.loc[:, col_names[:-1]],
            data.loc[:, col_names[-1]],
            random_state=0
        )
        estimator = RipperExplainer(d=2, k=2, pruning_threshold=50)
        estimator.fit(x_train, y_train)
        dnf_rule_set_list = estimator.explain_multiclass()
        classifier = RuleSetClassifier(dnf_rule_set_list[:2],
                                       rule_selection_method=RuleSelectionMethod.WEIGHTED_MAX,
                                       confidence_metric=ConfidenceMetric.LAPLACE,
                                       weight_metric=WeightMetric.CONFIDENCE,
                                       default_label='Iris-virginica')
        classifier.update_rules_with_metrics(x_test, y_test)
        reader = TrxfReader()
        reader.load_data_dictionary(x_train)
        serializer = NyokaSerializer(TIMESTAMP)
        exporter = PmmlExporter(reader, serializer)
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources/iris.pmml')) as file:
            expected = file.read()
        actual = exporter.export(classifier)
        self.assertEqual(expected, actual)

    def test_ripper_adult(self):
        data_type = {'age': float,
                     'workclass': str,
                     'fnlwgt': float,
                     'education': str,
                     'education-num': float,
                     'marital-status': str,
                     'occupation': str,
                     'relationship': str,
                     'race': str,
                     'sex': str,
                     'capital-gain': float,
                     'capital-loss': float,
                     'native-country': str,
                     'hours-per-week': float,
                     'label': str}

        col_names = ['age', 'workclass', 'fnlwgt', 'education',
                     'education-num', 'marital-status', 'occupation',
                     'relationship', 'race', 'sex',
                     'capital-gain', 'capital-loss', 'hours-per-week',
                     'native-country', 'label']

        df = pd.read_csv('https://archive.ics.uci.edu/ml/machine-learning-databases/adult/adult.data',
                         header=None,
                         delimiter=', ',
                         engine='python',
                         names=col_names,
                         dtype=data_type)
        df.columns = df.columns.str.replace('-', '_')

        # first tests all cells then find all rows
        df = df[(df.astype(str) != '?').all(axis=1)]
        data_train, data_test = train_test_split(df, test_size=0.5, random_state=1)

        train_labels = data_train.columns[:-1]
        test_label = data_test.columns[-1]

        x_train = data_train.loc[:, train_labels]
        y_train = data_train.loc[:, test_label]

        x_test = data_test.loc[:, train_labels]
        y_test = data_test.loc[:, test_label]

        generator = RipperRuleSetGenerator()
        ruleset = generator.generate(x_train, y_train, '>50K', d=64, k=2, pruning_threshold=100)
        classifier = RuleSetClassifier([ruleset],
                                       rule_selection_method=RuleSelectionMethod.WEIGHTED_MAX,
                                       confidence_metric=ConfidenceMetric.LAPLACE,
                                       weight_metric=WeightMetric.CONFIDENCE,
                                       default_label='<=50K')
        classifier.update_rules_with_metrics(x_test, y_test)
        reader = TrxfReader()
        reader.load_data_dictionary(x_test)
        serializer = NyokaSerializer(TIMESTAMP)
        exporter = PmmlExporter(reader, serializer)
        filename = 'adult.pmml'
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources', filename)) as file:
            expected = file.read()
        actual = exporter.export(classifier)
        self.assertEqual(expected, actual)

    def test_ripper_wifi(self):
        data = pd.read_csv('https://archive.ics.uci.edu/ml/machine-learning-databases/00422/wifi_localization.txt',
                           header=None,
                           delimiter='\t',
                           names=['X1', 'X2', 'X3', 'X4', 'X5', 'X6', 'X7', 'Y'],
                           dtype={'X1': float,
                                  'X2': float,
                                  'X3': float,
                                  'X4': float,
                                  'X5': float,
                                  'X6': float,
                                  'X7': float,
                                  'Y': str})

        x = data.loc[:, ['X1', 'X2', 'X3', 'X4', 'X5', 'X6', 'X7']]
        y = data.loc[:, 'Y']

        x_train, x_test, y_train, y_test = train_test_split(x, y, random_state=0)

        estimator = RipperExplainer(d=64, k=2, pruning_threshold=20)

        estimator.fit(x_train, y_train)
        dnf_rule_set_list = estimator.explain_multiclass()
        classifier = RuleSetClassifier(dnf_rule_set_list[:-1],
                                       rule_selection_method=RuleSelectionMethod.WEIGHTED_MAX,
                                       confidence_metric=ConfidenceMetric.LAPLACE,
                                       weight_metric=WeightMetric.CONFIDENCE,
                                       default_label='4')
        classifier.update_rules_with_metrics(x_test, y_test)
        reader = TrxfReader()
        reader.load_data_dictionary(x_train)
        serializer = NyokaSerializer(TIMESTAMP)
        exporter = PmmlExporter(reader, serializer)
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources/wifi.pmml')) as file:
            expected = file.read()
        actual = exporter.export(classifier)
        # print(actual)
        self.assertEqual(expected, actual)
