# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dit_cli']

package_data = \
{'': ['*']}

install_requires = \
['click>=7.0,<8.0', 'toml>=0.10.0,<0.11.0']

entry_points = \
{'console_scripts': ['dit = dit_cli.cli:main']}

setup_kwargs = {
    'name': 'dit-cli',
    'version': '0.3.2',
    'description': 'The CLI for dit',
    'long_description': '# dit - The universal container file\n![Short Demo](https://github.com/isaiahshiner/dit-cli/blob/master/docs/gifs/short.gif)\n\nDit is a new way of storing data, designed to be as generic as possible. It uses embedded scripts for validation, custom print functions, and more. Dits can be used to transport data across different formats and platforms, so sharing data is much easier. See more at [the official website.](https://www.ditabase.io/)\n\n## Install\n\n    pip install dit-cli\n\nDit can be installed with [pip for python](https://pip.pypa.io/en/stable/installing/). Note that you will need Python 3.8 and Node.js (for JS scripts).\n\n## Usage\n\n    dit validate [filename]\nValidate a dit file. Returns either an error message, or \'dit is valid, no errors found\'.\n\n    dit query [filename] [query_string]\nValidate a dit file, and then return a value based on a query string. A query should resemble a `@@` variable sequence found in dit code.\n\n    \'@@top_level_object\' -> serialize the entire object\n    \'@@top_level_object.some_value\' -> serialize that value, whatever it is\n    \'@@print(top_level_object)\' -> Use the object\'s print function, if it has one.\n\n## dit quickstart\n\nTo start, you\'ll need a dit. You can see examples in [tests/dits](https://github.com/isaiahshiner/dit-cli/tree/master/tests/dits).\n\nLet\'s make a simple dit, `name.dit`:\n\n```\nName {\n    String value;\n}\n```\n\nThis is a dit class, with a single contained (instance) variable. String is the only primitive type. Everything is either a string, object, or list.\n\nNow that we have a class, we can make an object and assign it.\n\n```\nName name;\nname.value = \'John\';\n```\n\nGreat! Now we can get the stored name by querying the file:\n\n    dit query name.dit \'@@name.value\'\n    "John"\n\nThe double at symbol `@@` is the dit escape sequence, used to reference variables in code and query strings. Variables are always referenced as `.` operations.\n\nWe can ask for the entire `name` object, rather than a specific part:\n\n    dit query name.dit \'@@name\'\n    {"class":"Name","value":"John"}\n\nThis will serialize the object. The default serialization language is Javascript, which means JSON, but this is highly configurable. We would rather a `Name` object be represented by it\'s value. Let\'s add a print function for that:\n\n```\nName {\n    String value;\n    print value;\n}\n```\n\n    dit query name.dit \'@@print(name)\'\n    "John"\n\n`print()` will try to use a print function before serializing. It will always return a useful value, never `null`/`None`.\n\nBut we have a problem: There\'s nothing to stop something like this:\n\n    dit query name.dit \'@@print(name)\'\n    "4jZw3ef\\n"\n\nWe need to make sure it\'s really a name. Let\'s add a validator!\n\n```\nName {\n    String value;\n    print value;\n    validator Javascript {{\n        // This is real Javascript (nodejs)\n        let name = @@value;\n        if (!/^[A-Z][A-z]+$/.test(name)) {\n            return `Not a valid name: "${name}"`\n        }\n        return true;\n    }}\n}\n```\n\nDit will interpret everything in the double brace `{{}}` section as Javascript code, and use it to build a .js file. `@@value` will be converted into a JS compatible string before running the code. You can see all the language configurations in ~/.dit-languages. This is how new languages will be added and customized.\n\nIf a validator returns any value other than \'true\' (case insensitive), that value is assumed to be an error message, and will cause a `ValidationError`.\n\n    dit query name.dit \'@@print(name)\'\n    ValidationError<name>: Not a valid name: "4jZw3ef\\n"\n\nNow let\'s make this a little more interesting. How about a `FullName` class?\n\n```\nFullName {\n    Name givenName;\n    list Name middleNames;\n    Name familyName;\n    print Python {{\n        # Python and Javascript are supported out of the box\n        full = @@givenName\n        for mid in @@middleNames: # Returns a list of Python dictionaries\n            full += \' \' + mid\n        full += \' \' + @@familyName\n        return full\n    }}\n}\n```\n\nA full name contains several Name objects. `middleNames` is a list, since people can have multiple middle names. Note that a list can be any shape of list: 2D, 3D, jagged, etc, they\'re all just `list ClassName varName`;\n\nThe print script nicely puts everything together. There\'s not really any need for a validator, since all the `Names` will be validated before ever reaching this class.\n\nFor assigning a `FullName` object, we still have the `name` variable from before, but we don\'t want to manually create a different variable for each object.\n\n```\nName firstName;\nName middleName1;\nName middleName2;\n```\n\nSo, we can use an assigner, which will create an anonymous object for us. Assigners are somewhere in between a constructor and a function. They can only have assignment statements currently, but they will probably look more like arbitrary functions in the future, with scripts to validate and transform parameters before assignment. You could have a large object defined by a single JSON, just by peeling it apart and assigning values. Then print it back as a JSON, JSON in a different schema, XML, CSV... ah, but I digress.\n\n```\n// It\'s okay for assigners to have single letter names.\nName n(nameParam) {\n    value = nameParam;\n    // We could have more parameters and assignments if needed\n}\n\nFullName myName;\n\n// Assign an existing object\nmyName.givenName = name;\n\n// Use the \'n\' assigner inside a list\nmyName.middleNames = [n(\'Leopold\'), n(\'Koser\')];\n\n// Assign the string directly\nmyName.familyName.value = \'Shiner\';\n```\n\nCool! Let\'s give it a try:\n\n    dit query name.dit \'@@print(myName)\'\n    "Hi! My name is Isaiah Leopold Koser Shiner"\n    dit query name.dit \'@@myName\' | python -m json.tool > output.json\n    cat output.json\n    {\n        "class": "FullName",\n        "print": "Hi! My name is Isaiah Leopold Koser Shiner",\n        "givenName": {\n            "class": "Name",\n            "print": "Isaiah",\n            "value": "Isaiah"\n        },\n        "middleNames": [\n            {\n                "class": "Name",\n                "print": "Leopold",\n                "value": "Leopold"\n            },\n            {\n                "class": "Name",\n                "print": "Koser",\n                "value": "Koser"\n            }\n        ],\n        "familyName": {\n            "class": "Name",\n            "print": "Shiner",\n            "value": "Shiner"\n        }\n    }\n\n\nFinally, let\'s take a quick look at inheritance.\n\n```\nOtherName {\n    extends Name;\n    validator Python {{\n        if @@value == \'Tom\':\n            return \'I don\\\'t like the name Tom."\n        return True\n    }}\n}\n```\nChild classes have all the fields of their parents, so `value` exists in `OtherName` implicitly. All the `Names` are validated by the parent, then the child. Print functions must be explicitly inherited by type, like `print Name;`.\n\n\n`extends` must be the first thing in a class, if it is to appear at all. To inherit from multiple classes, just separate with commas, like `extends Name, AndSomeOtherObject;`. If there are name conflicts, the first extended class take\'s precedence, then the second class must explicitly be called out:\n\n```\nC {\n    extends A, B;\n    print name; // gets name from A\n    print B.name; // gets name from B\n    // Remember that this is only if both A and B have a variable called \'name\'\n}\n```\n\nThe real documentation is entirely WIP right now, so if you have questions, please don\'t hesitate to shoot me a message on [Discord](https://discord.gg/7shhUxy) or email me at isaiah@ditabase.io. You can also make issues on the [issues tracker.](https://github.com/isaiahshiner/dit-cli/issues)\n\n## Why? What\'s the point?\n\nThe long answer is written [here.](docs/whats-the-point.md)\n\nThe short answer is that there\'s nothing out there that does everything.\n\n* You can write your own custom validation code for each project, each situation, but that is a horrible violation of [DRY](https://en.wikipedia.org/wiki/Don%27t_repeat_yourself). Not that *you\'re* repeating yourself, but that everyone else is repeating each other.\n  \n* Schema.org has a wonderful set of schemas, for almost every situation imaginable, but that\'s all they are, just schemas. I know where I would start if I was laying out a [Person Schema](https://schema.org/Person), but validation is still my problem. And furthermore, Schema.org is a relatively closed system. I can\'t just whip up a child of another schema if I want to, the way I can with dit.\n\n* JSON-Schema, JSON-LD, IPLD, all suffer from the general problem that they choose a specific way to implement things, which makes them wonderful, but not universal. Dit is [not trying to replace](https://xkcd.com/927/) every other way to write down data, only be a bridge between them. Dit relies heavily on JSON because it\'s so good, but in the edge cases, you can use CSV, or something else custom. Even if one format can cover 60%, the network effect of including the other 40% is incredibly valuable.\n\nWe need the validation, schemas, formatting, and even the discussion and community to all be in one place. The end game is to have a repository of open source dit classes, so that any data from any context can be used in any other context. I hope you agree, and you\'ll consider trying dit.\n\n## Links\n* [Official Website](https://www.ditabase.io/)\n\n* [Discord Chat](https://discord.gg/7shhUxy)\n\n* isaiah@ditabase.io\n\n* [Issue Tracker](https://github.com/isaiahshiner/dit-cli/issues)',
    'author': 'Isaiah Shiner',
    'author_email': 'shiner.isaiah@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://www.ditabase.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
