#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Template utilities for MetObs Toolkit.

This module provides the Template class and related functions for handling
template files and mappings between data/metadata columns and toolkit conventions.
"""


import sys
import logging
from typing import Union


import pandas as pd
from pytz import all_timezones
from metobs_toolkit.io_collection.filereaders import JsonFileReader
from metobs_toolkit.backend_collection.errorclasses import MetobsTemplateError
import metobs_toolkit.backend_collection.printing_collection as printing

logger = logging.getLogger("<metobs_toolkit>")

# Blacklists are created for column names, which are also used as a specific
# column that is often generated by the toolkit.

column_data_blacklist = (
    []
)  # When this column is found, an underscore is added to the name,
column_meta_blacklist = [
    "geometry",
    "lon",
    "lat",
    "assumed_import_frequency",
    "dataset_resolution",
    "LCZ",
    "altitude",
    "dataset_resolution",
    "dt_start",
    "dt_end",
]


def _get_empty_templ_dict() -> dict:
    """
    Create and return an empty template dictionary with the required structure.

    Returns
    -------
    dict
        An empty template dictionary with default values.
    """
    logger.debug("Entering _get_empty_templ_dict()")
    templ_dict = {
        # data file
        "data_related": {
            "structure": None,  # long, wide or single-station
            "timestamp": {
                "datetime_column": None,
                "datetime_fmt": None,
                "date_column": None,
                "date_fmt": None,
                "time_column": None,
                "time_fmt": None,
                "timezone": "UTC",
            },
            "name_column": None,
            "obstype_mapping": [
                {
                    "tlk_obstype": None,
                    "columnname": None,
                    "unit": None,
                    "description": None,
                },
            ],
        },
        # Meta data file
        "metadata_related": {
            "name_column": None,
            "lat_column": None,
            "lon_column": None,
            "altitude_column": None,
            "columns_to_include": [],
        },
        # extra settings
        "single_station_name": "dummy_station_name",
    }
    return templ_dict


class Template:
    """
    Contains all information and methods to work with a template.

    This class manages the mapping between data/metadata columns and the toolkit's
    conventions, as well as template file reading and validation.
    """

    def __init__(self) -> None:
        """Initialize a Template instance."""
        self.data_namemap = {"name": None}  # name --> name column in data
        self.metadata_namemap = {"name": None}  # name --> name column in metadata

        # obstypes details
        self.obscolumnmap = {}  # toolkitname --> datacolumnname
        self.obsdetails = {}  # obsname(tlk) --> {unit: , description: ..}

        self.metacolmapname = {}  # toolkitname --> metadatacolumnname

        # Special always required
        self.dataformat = (
            "long"  # long or wide (single station is converted to long on import)
        )

        # For single station data
        self.data_is_single_station = False  # datafmt is assumed to be long, but name column is not required in the data
        self.single_station_name = None

        self.timestampinfo = {
            "datetimecolumn": None,
            "time_column": None,
            "date_column": None,
            "fmt": None,
        }

        # Extra options (not required)
        self.tz = None

        # Not actively used attributes
        self.filepath = None

    def get_info(self, printout=True) -> Union[None, str]:
        """
        Print an overview of the Template.

        Returns
        -------
        None
        """
        logger.debug(f"Entering get_info() for {self}")
        infostr = ""
        infostr += printing.print_fmt_title("General info of Template")
        infostr += printing.print_fmt_section("Data obstypes map")

        for key, val in self.obscolumnmap.items():
            infostr += printing.print_fmt_line(f"{key}: {val}", identlvl=1)
            infostr += printing.print_fmt_line(
                f"raw data in {self.obsdetails[key]['unit']}", 2
            )
            infostr += printing.print_fmt_line(
                f"description: {self.obsdetails[key]['description']}", 2
            )

        infostr += printing.print_fmt_section("Data extra mapping info")
        infostr += printing.print_fmt_line(
            f"name column (data) <---> {str(self.data_namemap['name'])}"
        )

        if self.data_is_single_station:
            infostr += printing.print_fmt_line(
                f"single station name <---> {self.single_station_name}"
            )

        infostr += printing.print_fmt_section("Data timestamp map")
        for key, val in self.timestampinfo.items():
            infostr += printing.print_fmt_line(f"{key} <---> {str(val)}")
        infostr += printing.print_fmt_line(f"{'Timezone'} <---> {self.tz}")

        infostr += printing.print_fmt_section("Metadata map")
        for key, val in self.metacolmapname.items():
            infostr += printing.print_fmt_line(f"{key} <---> {str(val)}")

        if printout:
            print(infostr)
        else:
            return infostr

    # =============================================================================
    # Setters
    # =============================================================================
    def _set_dataname(self, datanamecolumn: str) -> None:
        """
        Set the data name-column.

        Parameters
        ----------
        datanamecolumn : str
            The name of the data column to use as the station name.

        Returns
        -------
        None
        """
        logger.debug(f"Entering _set_dataname() for {self}")
        self.data_namemap["name"] = str(datanamecolumn)

    def _set_dataformat(self, datafmt: str) -> None:
        """
        Set the data format.

        Parameters
        ----------
        datafmt : str
            The data format. Should be 'long', 'wide', or 'single_station'.

        Returns
        -------
        None

        Raises
        ------
        SystemExit
            If the data format is not recognized.
        """
        logger.debug(f"Entering _set_dataformat() for {self}")
        if str(datafmt) == "long":
            self.dataformat = "long"
        elif str(datafmt) == "wide":
            self.dataformat = "wide"
        elif str(datafmt) == "single_station":
            # Data format is long, but indicate that data represents a single station
            self.dataformat = "long"
            self.data_is_single_station = True
        elif datafmt is None:
            # no dataformat has been set --> this is valid in a metadata-only case
            self.dataformat = None
        else:
            sys.exit(f"{datafmt} is not a known dataformat.")

    # =============================================================================
    # Getters (used by other classes to extract specific data from a template)
    # =============================================================================
    def _is_data_long(self) -> bool:
        """Return True if the data format is 'long'."""
        return self.dataformat == "long"

    def _is_data_single_station(self) -> bool:
        """Return True if the data represents a single station."""
        return self.data_is_single_station

    def _get_single_station_default_name(self) -> str:
        """Return the default name for a single station."""
        return str(self.single_station_name)

    def _get_wide_obstype(self) -> str:
        """
        Get the name of the wide obstype in toolkit space.

        Returns
        -------
        str
            The name of the obstype for wide data.
        """
        logger.debug(f"Entering _get_wide_obstype() for {self}")
        return list(self.obsdetails.keys())[0]

    def _get_tz(self) -> str:
        """
        Get the timezone string.

        Returns
        -------
        str
            The timezone string.
        """
        logger.debug(f"Entering _get_tz() for {self}")
        return self.tz

    def _get_data_name_map(self) -> dict:
        """
        Get the name-map for the data file.

        Returns
        -------
        dict
            Mapping from data column name to 'name'.
        """
        logger.debug(f"Entering _get_data_name_map() for {self}")
        return {self.data_namemap["name"]: "name"}

    def _get_metadata_name_map(self) -> dict:
        """
        Get the name-map for the metadata file.

        Returns
        -------
        dict
            Mapping from metadata column name to 'name'.
        """
        logger.debug(f"Entering _get_metadata_name_map() for {self}")
        return {self.metadata_namemap["name"]: "name"}

    def _get_metadata_column_map(self) -> dict:
        """
        Get the mapper for all metadata columns (including name, lat, lon, altitude).

        Returns
        -------
        dict
            Mapping from metadata column name to toolkit name.
        """
        logger.debug(f"Entering _get_metadata_column_map() for {self}")
        return {val: key for key, val in self.metacolmapname.items()}

    def _get_obs_column_map(self) -> dict:
        """
        Get the mapper for all observation columns of the data (for long format).

        Returns
        -------
        dict
            Mapping from data column name to toolkit obstype name.

        Raises
        ------
        AssertionError
            If the datetime mapping is invalid.
        """
        logger.debug(f"Entering _get_obs_column_map() for {self}")
        self._check_if_datetime_is_mapped()
        columnmmap = {}
        if self.dataformat == "long":
            # add all obstype columns
            for key, val in self.obscolumnmap.items():
                columnmmap[val] = key
        else:
            pass  # no mapping done on wide
        return columnmmap

    def _get_all_mapped_data_cols_in_tlk_space(self) -> list:
        """
        Get all mapped columns in toolkit space.

        Returns
        -------
        list
            List of all mapped columns in toolkit space.
        """
        logger.debug(f"Entering _get_all_mapped_data_cols_in_tlk_space() for {self}")
        mapped_cols = ["name", "datetime"]
        if self.dataformat == "long":
            mapped_cols.extend(list(self.obscolumnmap.keys()))
        else:
            # wide only represents one obstype
            mapped_cols.append(list(self.obsdetails.keys())[0])
        return mapped_cols

    def _get_original_obstype_columnname(self, obstypename: str) -> str:
        """
        Get the original column name for a given toolkit obstype.

        Parameters
        ----------
        obstypename : str
            The toolkit obstype name.

        Returns
        -------
        str
            The original data column name.
        """
        logger.debug(f"Entering _get_original_obstype_columnname() for {self}")
        return str(self.obscolumnmap[obstypename])

    def _get_input_unit_of_tlk_obstype(self, obstypename: str) -> str:
        """
        Get the input unit for a given toolkit obstype.

        Parameters
        ----------
        obstypename : str
            The toolkit obstype name.

        Returns
        -------
        str
            The input unit.
        """
        logger.debug(f"Entering _get_input_unit_of_tlk_obstype() for {self}")
        return str(self.obsdetails[obstypename]["unit"])

    def _get_description_of_tlk_obstype(self, obstypename: str) -> str:
        """
        Get the description for a given toolkit obstype.

        Parameters
        ----------
        obstypename : str
            The toolkit obstype name.

        Returns
        -------
        str
            The description.
        """
        logger.debug(f"Entering _get_description_of_tlk_obstype() for {self}")
        return str(self.obsdetails[obstypename]["description"])

    # =============================================================================
    # Validity checkers
    # =============================================================================
    def _check_if_datetime_is_mapped(self) -> None:
        """
        Test if the required template details are present to construct a timestamp column.

        Returns
        -------
        None

        Raises
        ------
        MetobsTemplateError
            If the timezone is not known or the timestamp mapping is invalid.
        """
        logger.debug(f"Entering _check_if_datetime_is_mapped() for {self}")
        ts_info = self.timestampinfo

        # Check if timezone is known
        if self._get_tz() not in all_timezones:
            raise MetobsTemplateError(f"{self._get_tz()} is not a known timezone.")

        # situation 1:  datetime column is present
        if ts_info["datetimecolumn"] is not None:
            assert (
                ts_info["fmt"] is not None
            ), "Datetimes are assumed to be present in ONE column, but no datetime format is specified."
            if ts_info["time_column"] is not None:
                self.timestampinfo["time_column"] = None
                logger.warning(
                    f"The mapping of the time column ({ts_info['time_column']}) is ignored because of the presence of a datetime column."
                )
            if ts_info["date_column"] is not None:
                self.timestampinfo["date_column"] = None
                logger.warning(
                    f"The mapping of the date column ({ts_info['date_column']}) is ignored because of the presence of a datetime column."
                )
            return

        # Situation 2: a separate date and time columns is present.
        if (ts_info["time_column"] is not None) & (ts_info["date_column"] is not None):
            assert (
                ts_info["fmt"] is not None
            ), "Datetimes are assumed to be present as a date and time column, but no formats are specified."
            return
        sys.exit(
            "The timestamps are not correctly mapped (either by using a datetime column, or by a time and date column)"
        )

    def _data_template_compatibility_test(self, datacolumns: list) -> None:
        """
        Check the compatibility of the template and the columns of the data.

        Parameters
        ----------
        datacolumns : list
            List of column names in the data.

        Returns
        -------
        None

        Raises
        ------
        MetobsTemplateError
            If required columns are missing or incorrectly mapped.
        """
        logger.debug(f"Entering _data_template_compatibility_test() for {self}")
        # check datetime
        self._check_if_datetime_is_mapped()
        if self.timestampinfo["datetimecolumn"] is not None:
            if not (self.timestampinfo["datetimecolumn"] in datacolumns):
                raise MetobsTemplateError(
                    f'The column {self.timestampinfo["datetimecolumn"]} is incorrectly mapped in the template as the "datetime" column. The template is therefore not valid with the data.'
                )

        if self.timestampinfo["time_column"] is not None:
            if not (self.timestampinfo["time_column"] in datacolumns):
                raise MetobsTemplateError(
                    f'The column {self.timestampinfo["time_column"]} is incorrectly mapped in the template as the "_time" column. The template is therefore not valid with the data.'
                )

        if self.timestampinfo["date_column"] is not None:
            if not (self.timestampinfo["date_column"] in datacolumns):
                raise MetobsTemplateError(
                    f'The column {self.timestampinfo["date_column"]} is incorrectly mapped in the template as the "_date" column. The template is therefore not valid with the data.'
                )

        if self._is_data_long():
            # check name column
            if not self._is_data_single_station():
                if not (self.data_namemap["name"] in datacolumns):
                    raise MetobsTemplateError(
                        f'The column {self.data_namemap["name"]} is not (or incorrectly) mapped in the template as the "name" column. The template is therefore not valid with the data.'
                    )

            # check if template has obstypes not present in the data
            for mapped_obscol in self.obscolumnmap.values():
                if mapped_obscol not in datacolumns:
                    msg = f"{mapped_obscol} is a mapped observation, present in the template but not found in the data! This obstype will be ignored."
                    logger.warning(msg)

            # check if data has obstypes not present in the template
            for data_col in datacolumns:
                if data_col in [
                    self.data_namemap["name"],
                    self.timestampinfo["datetimecolumn"],
                    self.timestampinfo["time_column"],
                    self.timestampinfo["date_column"],
                ]:
                    continue

                if data_col not in self.obscolumnmap.values():
                    msg = f"{data_col} is present in the datafile, but not found in the template! This column will be ignored."
                    logger.warning(msg)

        else:
            # wide tests:
            # all columns are assumed to be station names
            pass

    def _metadata_template_compatibility_test(self, metadatacolumns: list) -> None:
        """
        Check the compatibility of the template and the columns of the metadata.

        Parameters
        ----------
        metadatacolumns : list
            List of column names in the metadata.

        Returns
        -------
        None

        Raises
        ------
        MetobsTemplateError
            If required columns are missing or incorrectly mapped.
        """
        logger.debug(f"Entering _metadata_template_compatibility_test() for {self}")
        # check name column (must be present if multiple stations are in the data)
        if not self._is_data_single_station():
            if not (self.metadata_namemap["name"] in metadatacolumns):
                raise MetobsTemplateError(
                    f'The column {self.metadata_namemap["name"]} is not (or incorrectly) mapped in the template as the "name" column. The template is therefore not valid with the metadata.'
                )

        # check if template contains mapped columns not present in the metadata
        for mapped_col in self.metacolmapname.values():
            if mapped_col not in metadatacolumns:
                msg = f"{mapped_col} is a mapped metadata column, present in the template but not found in the metadata! This column will be ignored."
                logger.warning(msg)

        # check if metadata has columns which are not mapped by the template
        unmapped = (
            set(metadatacolumns) - set(self.metacolmapname.values()) - set(["name"])
        )
        if bool(unmapped):
            msg = f"The following columns are found in the metadata, but not in the template and are therefore ignored: \n{list(unmapped)}"
            logger.warning(msg)

    def _apply_blacklist(self, columns: list, on_data: bool) -> dict:
        """
        Create a mapper for columns in the data or metadata file, which have
        a specific meaning by the toolkit and are thus present in the blacklists.

        If a column is found in the blacklist, but will also be mapped, then it
        will not be included in the mapper.

        The mapper adds a "_original" postfix to the column names.

        Parameters
        ----------
        columns : list
            List of column names to check.
        on_data : bool
            If True, applies the data blacklist; otherwise, applies the metadata blacklist.

        Returns
        -------
        dict
            Mapping from original column names to renamed column names.
        """
        logger.debug(f"Entering _apply_blacklist() for {self}")
        if on_data:
            blacklist = column_data_blacklist
        else:
            blacklist = column_meta_blacklist

        to_rename = [col for col in columns if col in blacklist]

        if on_data:
            # if the column is mapped by the template, remove it from the to_rename
            mapped_set = set(
                [
                    self.data_namemap["name"],  # name col
                    self.timestampinfo["datetimecolumn"],
                    self.timestampinfo["date_column"],
                    self.timestampinfo["time_column"],
                ]
            )
            mapped_set = mapped_set.union(set(self._get_obs_column_map().keys()))

        else:
            # on metadata
            mapped_set = set([self.metadata_namemap["name"]])
            mapped_set = mapped_set.union(set(self._get_metadata_column_map().keys()))

        mapped_set = mapped_set - set([None])

        to_rename = set(to_rename) - mapped_set
        blacklist_mapper = {col: f"{col}_original" for col in to_rename}

        if bool(blacklist_mapper):
            if on_data:
                msg = f"The following data columns are renamed because of special meaning by the toolkit: {blacklist_mapper}"
            else:
                msg = f"The following metadata columns are renamed because of special meaning by the toolkit: {blacklist_mapper}"
            logger.warning(msg)

        return blacklist_mapper

    # =============================================================================
    # Other methods
    # =============================================================================

    def read_template_from_file(
        self, jsonpath: str, templatefile_is_url: bool = False
    ) -> None:
        """
        Read the template file (JSON), and update the attributes of this Template.

        Parameters
        ----------
        jsonpath : str
            Path or URL to the template JSON file.
        templatefile_is_url : bool, optional
            If True, treat jsonpath as a URL. Default is False.

        Returns
        -------
        None

        Raises
        ------
        MetobsTemplateError
            If required keys are missing in the template file.
        """
        logger.debug(f"Entering read_template_from_file() for {self}")
        jsonreader = JsonFileReader(file_path=jsonpath, is_url=templatefile_is_url)
        jsonreader.read()

        tml_dict = jsonreader.data

        # set attributes
        self.data_namemap = {"name": tml_dict["data_related"]["name_column"]}
        self.metadata_namemap = {"name": tml_dict["metadata_related"]["name_column"]}
        self._set_dataformat(tml_dict["data_related"]["structure"])
        self.single_station_name = str(tml_dict["single_station_name"])

        if tml_dict["data_related"]["timestamp"]["datetime_column"] is None:
            dt_fmt = f'{tml_dict["data_related"]["timestamp"]["date_fmt"]} {tml_dict["data_related"]["timestamp"]["time_fmt"]}'
        else:
            dt_fmt = f'{tml_dict["data_related"]["timestamp"]["datetime_fmt"]}'

        self.timestampinfo = {
            "datetimecolumn": tml_dict["data_related"]["timestamp"]["datetime_column"],
            "time_column": tml_dict["data_related"]["timestamp"]["time_column"],
            "date_column": tml_dict["data_related"]["timestamp"]["date_column"],
            "fmt": dt_fmt,
        }

        self.tz = str(tml_dict["data_related"]["timestamp"]["timezone"])

        for obsdict in tml_dict["data_related"]["obstype_mapping"]:
            self.obscolumnmap[obsdict["tlk_obstype"]] = obsdict["columnname"]
            self.obsdetails[obsdict["tlk_obstype"]] = {
                "unit": obsdict["unit"],
                "description": obsdict["description"],
            }

        self.metacolmapname["name"] = tml_dict["metadata_related"]["name_column"]
        if tml_dict["metadata_related"]["lat_column"] is not None:
            self.metacolmapname["lat"] = tml_dict["metadata_related"]["lat_column"]
        if tml_dict["metadata_related"]["lon_column"] is not None:
            self.metacolmapname["lon"] = tml_dict["metadata_related"]["lon_column"]
        try:
            if tml_dict["metadata_related"]["altitude_column"] is not None:
                self.metacolmapname["altitude"] = tml_dict["metadata_related"][
                    "altitude_column"
                ]
        except KeyError:
            raise MetobsTemplateError(
                f'"altitude_column" is not present in the templatefile ({jsonpath}). Rebuild the template to fix this.'
            )

        for extra_col in tml_dict["metadata_related"]["columns_to_include"]:
            self.metacolmapname[extra_col] = extra_col


def update_known_obstype_with_original_data(
    known_obstypes: list, template: Template
) -> list:
    """
    Update the known obstypes with template information.

    Parameters
    ----------
    known_obstypes : list
        List of known obstype objects to update.
    template : Template
        The Template instance containing mapping information.

    Returns
    -------
    list
        The updated list of known obstype objects.
    """
    logger.debug("Entering update_known_obstype_with_original_data()")
    if template._is_data_long():
        orig_data_column_name_map = template._get_obs_column_map()
        for orig_name, obsname in orig_data_column_name_map.items():

            # set original name (column)
            known_obstypes[obsname].original_name = orig_name

            # set original unit
            input_unit = template._get_input_unit_of_tlk_obstype(obsname)
            known_obstypes[obsname].original_unit = input_unit
    else:
        # wide structured data
        obsname = template._get_wide_obstype()
        # set original name (column)
        known_obstypes[obsname].original_name = "Irrelevant for wide-data"
        # Set original units
        input_unit = template._get_input_unit_of_tlk_obstype(obsname)
        known_obstypes[obsname].original_unit = input_unit

    return known_obstypes
