#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Logging utilities for the MetObs Toolkit.

Provides functions to add file and stream handlers to the toolkit logger.

Created on Fri Aug  2 14:23:30 2024

@author: thoverga
"""

import os
import logging
from datetime import datetime

logger = logging.getLogger("<metobs_toolkit>")


def add_FileHandler(
    trglogfile: str,
    setlvl: str = "DEBUG",
    logformat: str = "LOG:: %(levelname)s - %(message)s",
    clearlog: bool = True,
) -> None:
    """
    Add a FileHandler to the Toolkit logger.

    A FileHandler directs the logs generated by the `metobs_toolkit` to a file.

    Parameters
    ----------
    trglogfile : str
        Path of the target log file.
    setlvl : {"DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"}, optional
        The logger level for the FileHandler. See
        https://docs.python.org/3/library/logging.html#levels for more details.
        The default is "DEBUG".
    logformat : str, optional
        The format of the log messages. The default is
        "LOG:: %(levelname)s - %(message)s".
    clearlog : bool, optional
        If True, the `trglogfile` is cleared before adding the FileHandler.
        The default is True.

    Returns
    -------
    None
    """
    logger.debug("Entering add_FileHandler()")
    if clearlog:
        if os.path.isfile(trglogfile):
            os.remove(trglogfile)

    # Create the Handler for logging data to a file - will be inherited for children
    file_handler = logging.FileHandler(filename=trglogfile)
    file_handler.setLevel(setlvl.upper())  # set handler level
    # Create a Formatter for formatting the log messages
    file_logger_formatter = logging.Formatter(logformat)
    file_handler.setFormatter(file_logger_formatter)
    # Add the Handler to the Logger
    rootlog = logging.getLogger("<metobs_toolkit>")
    rootlog.addHandler(file_handler)

    rootlog.debug(f"FileHandler set at {datetime.now()}")


def add_StreamHandler(
    setlvl: str = "DEBUG", logformat: str = "LOG:: %(levelname)s - %(message)s"
) -> None:
    """
    Add a StreamHandler to the Toolkit logger.

    A StreamHandler directs the logs generated by the `metobs_toolkit` to `sys.stderr`.

    Parameters
    ----------
    setlvl : str, optional
        The logger level for the StreamHandler. Must be one of ["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"].
        See https://docs.python.org/3/library/logging.html#levels for more details. Default is "DEBUG".
    logformat : str, optional
        The format string for log messages. Default is "LOG:: %(levelname)s - %(message)s".

    Returns
    -------
    None
    """
    logger.debug("Entering add_StreamHandler()")
    # Get rootlogger
    rootlog = logging.getLogger("<metobs_toolkit>")
    rootlog.setLevel(logging.DEBUG)  # set rootlogger on debug

    # Create StreamHandler
    streamhandler = logging.StreamHandler()
    streamhandler.setLevel(setlvl.upper())
    stream_logger_formatter = logging.Formatter(logformat)
    streamhandler.setFormatter(stream_logger_formatter)

    # Add the Handler to the Logger
    rootlog.addHandler(streamhandler)

    rootlog.info(f"StreamHandler set at {datetime.now()}")
