"""
Created on 2022-09-02
modified version, original from JustPy
@author: wf (modification swelcker)
"""
import asyncio
import fnmatch
import inspect
import json
import os
import pathlib
import sys
import traceback
import typing
import uuid

from fastapi import FastAPI
from itsdangerous import Signer
from starlette.endpoints import HTTPEndpoint
from starlette.requests import Request
from starlette.responses import (HTMLResponse, JSONResponse, PlainTextResponse,
                                 Response)
from starlette.routing import Route
from starlette.templating import Jinja2Templates

from ..jpcore.component import Component
from ..jpcore.justpy_config import (AGGRID, AGGRID_ENTERPRISE, BOKEH,
                                    COOKIE_MAX_AGE, CRASH, DECKGL, FAVICON,
                                    FRONTEND_ENGINE_TYPE, HIGHCHARTS, KATEX,
                                    LATENCY, NO_INTERNET, PLOTLY, QUASAR,
                                    QUASAR_VERSION, SECRET_KEY,
                                    SESSION_COOKIE_NAME, SESSIONS,
                                    STATIC_DIRECTORY, STATIC_NAME, TAILWIND,
                                    VEGA, config)
from ..jpcore.template import Context
from ..jpcore.webpage import WebPage

cookie_signer = Signer(str(SECRET_KEY))


def create_component_file_list():
    file_list = []
    component_dir = os.path.join(STATIC_DIRECTORY, "components")
    if os.path.isdir(component_dir):
        for file in os.listdir(component_dir):
            if fnmatch.fnmatch(file, "*.js"):
                file_list.append(f"/components/{file}")
    return file_list


component_file_list = create_component_file_list()
grand_parent = pathlib.Path(__file__).parent.parent.resolve()
template_dir = f"{grand_parent}/justpy/templates"

lib_dir = os.path.join(template_dir, "js", FRONTEND_ENGINE_TYPE)
# remove .js extension
FRONTEND_ENGINE_LIBS = [
    fn[:-3] for fn in os.listdir(lib_dir) if fnmatch.fnmatch(fn, "*.js")
]

TEMPLATES_DIRECTORY = config("TEMPLATES_DIRECTORY", cast=str, default=template_dir)

templates = Jinja2Templates(directory=TEMPLATES_DIRECTORY)

template_options = {
    "tailwind": TAILWIND,
    "quasar": QUASAR,
    "quasar_version": QUASAR_VERSION,
    "highcharts": HIGHCHARTS,
    "aggrid": AGGRID,
    "aggrid_enterprise": AGGRID_ENTERPRISE,
    "static_name": STATIC_NAME,
    "component_file_list": component_file_list,
    "no_internet": NO_INTERNET,
    "katex": KATEX,
    "plotly": PLOTLY,
    "bokeh": BOKEH,
    "deckgl": DECKGL,
    "vega": VEGA,
}


async def handle_event(data_dict, com_type=0, page_event=False):
    # com_type 0: websocket, con_type 1: ajax
    connection_type = {0: "websocket", 1: "ajax"}
    event_data = data_dict["event_data"]
    try:
        p = WebPage.instances[event_data["page_id"]]
    except:
        return
    event_data["page"] = p
    if com_type == 0:
        event_data["websocket"] = WebPage.sockets[event_data["page_id"]][
            event_data["websocket_id"]
        ]
    # The page_update event is generated by the reload_interval Ajax call
    if event_data["event_type"] == "page_update":
        build_list = p.build_list()
        return {"type": "page_update", "data": build_list}

    if page_event:
        c = p
    else:
        component_id = event_data["id"]
        c = Component.instances.get(component_id, None)
        if c is not None:
            event_data["target"] = c

    try:
        if c is not None:
            before_result = await c.run_event_function("before", event_data, True)
    except:
        pass
    try:
        if c is not None:
            if hasattr(c, "on_" + event_data["event_type"]):
                event_result = await c.run_event_function(
                    event_data["event_type"], event_data, True
                )
            else:
                event_result = None

        else:
            event_result = None

    except Exception as e:
        # raise Exception(e)
        if CRASH:
            print(traceback.format_exc())
            sys.exit(1)
        event_result = None

    # If page is not to be updated, the event_function should return anything but None
    if event_result is None:
        if com_type == 0:  # WebSockets communication
            if LATENCY:
                await asyncio.sleep(LATENCY / 1000)
            await p.update()
        elif com_type == 1:  # Ajax communication
            build_list = p.build_list()
    try:
        if c is not None:
            after_result = await c.run_event_function("after", event_data, True)
    except:
        pass
    if com_type == 1 and event_result is None:
        dict_to_send = {
            "type": "page_update",
            "data": build_list,
            "page_options": {
                "display_url": p.display_url,
                "title": p.title,
                "redirect": p.redirect,
                "open": p.open,
                "favicon": p.favicon,
            },
        }
        return dict_to_send


class JustpyApp:
    """
    a justpy application is a special FastAPI Addition

    """

    app = None

    def __init__(self) -> None:
        super().__init__()
        self.app = self

    def route_as_text(self, route):
        """
        get a string representation of the given route
        """
        text = f"{route.__class__.__name__}(name: {route.name}, path: {route.path}, format: {route.path_format},  regex: {route.path_regex})"
        if isinstance(route, Route):
            text += f"func: {route.endpoint.__name__}"
        return text

    def add_jproute(self, path: str, wpfunc: typing.Callable, name: str = None):
        """
        add a route for the given Webpage returning func

        Args:
            path: the path to use as route
            wpfunc: a Webpage returning func
            name: the name of the route
        """
        endpoint = self.response(wpfunc)
        if name is None:
            name = wpfunc.__name__
        self.router.add_route(path, endpoint, name=name, include_in_schema=False)

    def jproute(
        self, path: str, name: typing.Optional[str] = None
    ) -> typing.Callable:  # pragma: nocover
        """
        justpy route decorator

        function will we "wrapped" as a response and a route added

        Args:
            func(typing.Callable): the function to convert to a reponse
        """

        def routeResponse(func: typing.Callable) -> typing.Callable:
            """
            decorator for the given func

            Args:
                func: the callable func

            Returns:
                endpoint: an endpoint that has been routed

            """
            endpoint = self.response(func)
            self.router.add_route(
                path,
                endpoint,
                name=name if name is not None else func.__name__,
                include_in_schema=False,
            )
            self.route(path)
            return endpoint

        return routeResponse

    def response(self, func: typing.Callable):
        """
        response decorator converts a function to a response

        see also https://github.com/justpy-org/justpy/issues/532
        castAsEndPoint

        Args:
            func: the function (returning a WebPage) to convert to a response
        """

        async def funcResponse(request) -> HTMLResponse:
            """
            decorator function to apply the function to the request and
            return it as a response

            Args:
                request: the request to apply the function to

            Returns:
                response: a HTMLResponse applying the justpy infrastructure

            """
            new_cookie = self.handle_session_cookie(request)
            wp = await self.get_page_for_func(request, func)
            htmlresponse = self.get_response_for_load_page(request, wp)
            htmlresponse = self.set_cookie(request, htmlresponse, wp, new_cookie)
            if LATENCY:
                await asyncio.sleep(LATENCY / 1000)
            return htmlresponse

        # return the decorated function, thus allowing access to the func
        # parameter in the funcResponse later when applied
        return funcResponse

    async def get_page_for_func(self, request, func: typing.Callable) -> WebPage:
        """
        get the Webpage for the given func

        Args:
            request: the request to pass to the given function
            func: the function

        Returns:
            WebPage: the Webpage returned by the given function
        """
        # @TODO - get rid of the global func_to_run concept that isn't
        # in scope here (anymore) anyways
        func_to_run = func
        func_parameters = len(inspect.signature(func_to_run).parameters)
        assert (
            func_parameters < 2
        ), f"Function {func_to_run.__name__} cannot have more than one parameter"
        if inspect.iscoroutinefunction(func_to_run):
            if func_parameters == 1:
                load_page = await func_to_run(request)
            else:
                load_page = await func_to_run()
        else:
            if func_parameters == 1:
                load_page = func_to_run(request)
            else:
                load_page = func_to_run()
        return load_page

    def get_response_for_load_page(
        self, request: Request, load_page: WebPage
    ) -> HTMLResponse:
        """
        get the response for the given webpage

        Args:
            request: the request to handle
            load_page(WebPage): the webpage to wrap with justpy and

        Returns:
            response: the response for the given load_page
        """
        page_type = type(load_page)
        assert issubclass(
            page_type, WebPage
        ), f"Function did not return a web page but a {page_type.__name__}"
        if len(load_page) == 0 and not load_page.html:
            error_html = """<span style="color:red">Web page is empty - you might want to add components</span>"""
            return HTMLResponse(error_html, 500)
        page_options = {
            "reload_interval": load_page.reload_interval,
            "body_style": load_page.body_style,
            "body_classes": load_page.body_classes,
            "css": load_page.css,
            "head_html": load_page.head_html,
            "body_html": load_page.body_html,
            "display_url": load_page.display_url,
            "dark": load_page.dark,
            "title": load_page.title,
            "redirect": load_page.redirect,
            "highcharts_theme": load_page.highcharts_theme,
            "debug": load_page.debug,
            "events": load_page.events,
            "favicon": load_page.favicon if load_page.favicon else FAVICON,
        }
        if load_page.use_cache:
            page_dict = load_page.cache
        else:
            page_dict = load_page.build_list()
        template_options["tailwind"] = load_page.tailwind
        context = {
            "request": request,
            "page_id": load_page.page_id,
            "justpy_dict": json.dumps(page_dict, default=str),
            "use_websockets": json.dumps(WebPage.use_websockets),
            "options": template_options,
            "page_options": page_options,
            "html": load_page.html,
            "frontend_engine_type": FRONTEND_ENGINE_TYPE,
            "frontend_engine_libs": FRONTEND_ENGINE_LIBS,
        }
        # wrap the context in a context object to make it available
        context_obj = Context(context)
        context["context_obj"] = context_obj
        response = templates.TemplateResponse(load_page.template_file, context)
        return response

    def handle_session_cookie(self, request: Request) -> typing.Union[bool, Response]:
        """
        handle the session cookie for this request

        Returns:
            True if a new cookie and session has been created
        """
        # Handle web requests
        session_cookie = request.cookies.get(SESSION_COOKIE_NAME)
        new_cookie = None
        if SESSIONS:
            new_cookie = False
            if session_cookie:
                try:
                    session_id = cookie_signer.unsign(session_cookie).decode("utf-8")
                except:
                    return PlainTextResponse("Bad Session")
                request.state.session_id = session_id
                request.session_id = session_id
            else:
                # Create new session_id
                request.state.session_id = str(uuid.uuid4().hex)
                request.session_id = request.state.session_id
                new_cookie = True

        return new_cookie

    def set_cookie(
        self,
        request: Request,
        response: Response,
        load_page: WebPage,
        new_cookie: typing.Union[bool, Response],
    ) -> Response:
        """
        set the cookie_value

        Args:
            request: the request
            response: the response to be sent
            load_page: the WebPage to handle
            new_cookie: True if there is a new cookie. Or Response if cookie was invalid

        Returns:
            response: the response object
        """
        if isinstance(new_cookie, Response):
            return new_cookie
        if SESSIONS and new_cookie:
            cookie_value = cookie_signer.sign(request.state.session_id)
            cookie_value = cookie_value.decode("utf-8")
            response.set_cookie(
                SESSION_COOKIE_NAME, cookie_value, max_age=COOKIE_MAX_AGE, httponly=True
            )
            for k, v in load_page.cookies.items():
                response.set_cookie(k, v, max_age=COOKIE_MAX_AGE, httponly=True)
        return response


class JustpyAjaxEndpoint(HTTPEndpoint):
    """
    Justpy specific HTTPEndpoint/app (ASGI application)
    """

    def __init__(self, scope, receive, send):
        """
        constructor
        """
        HTTPEndpoint.__init__(self, scope, receive, send)

    async def post(self, request: Request):
        """
        Handles post method. Used in Ajax mode for events when websockets disabled

        Args:
            request: the request to handle
        """
        data_dict = await request.json()
        # {'type': 'event', 'event_data': {'event_type': 'beforeunload', 'page_id': 0}}
        if data_dict["event_data"]["event_type"] == "beforeunload":
            return await self.on_disconnect(data_dict["event_data"]["page_id"])

        session_cookie = request.cookies.get(SESSION_COOKIE_NAME)
        if SESSIONS and session_cookie:
            session_id = cookie_signer.unsign(session_cookie).decode("utf-8")
            data_dict["event_data"]["session_id"] = session_id

        # data_dict['event_data']['session'] = request.session
        msg_type = data_dict["type"]
        data_dict["event_data"]["msg_type"] = msg_type
        page_event = True if msg_type == "page_event" else False
        result = await handle_event(data_dict, com_type=1, page_event=page_event)
        if result:
            if LATENCY:
                await asyncio.sleep(LATENCY / 1000)
            return JSONResponse(result)
        else:
            return JSONResponse(False)

    async def on_disconnect(self, page_id):
        await WebPage.instances[
            page_id
        ].on_disconnect()  # Run the specific page disconnect function
        return JSONResponse(False)
